﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/msgpack.h>

#include <nn/news/news_ApiAdmin.h>
#include <nn/news/news_Data.h>
#include <nn/news/news_Database.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_NewsCommand.h"

#if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace
{
    const size_t NewsDataSizeMax = 10 * 1024 * 1024;

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    void DumpNewsInfo(const void* buffer, size_t size) NN_NOEXCEPT
    {
        nne::nlib::MemoryInputStream stream;

        stream.Init(buffer, size);

        NN_UTIL_SCOPE_EXIT
        {
            stream.Close();
        };

        nne::nlib::msgpack::JsonStreamParserSettings parserSettings;

        parserSettings.format = nne::nlib::msgpack::JsonStreamParserSettings::kFormatMsgpack;
        parserSettings.max_depth = 12;
        parserSettings.token_buffer_size = 2048;

        nne::nlib::msgpack::JsonStreamParser parser;

        parser.Init(parserSettings);
        parser.Open(&stream);

        NN_UTIL_SCOPE_EXIT
        {
            parser.Close();
        };

        struct Node
        {
            nne::nlib::msgpack::JsonStreamParser::Event event;
            int elementCount;
        };

        char map[256] = {};
        char key[256] = {};

        while (parser.HasNext())
        {
            nne::nlib::msgpack::JsonStreamParser::Event event = parser.Next();

            switch (event)
            {
            case nne::nlib::msgpack::JsonStreamParser::kEventKeyName:
                {
                    nn::util::Strlcpy(key, parser.GetToken().buf, sizeof (key));
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventStartMap:
                {
                    nn::util::Strlcpy(map, key, sizeof (map));
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventEndMap:
                {
                    map[0] = '\0';
                }
                break;
            case nne::nlib::msgpack::JsonStreamParser::kEventString:
                {
                    if (nn::util::Strncmp(map, "subject", sizeof (map)) == 0 &&
                        nn::util::Strncmp(key, "text", sizeof (key)) == 0)
                    {
                        NN_LOG("[Subject]\n%s\n", parser.GetToken().buf);
                    }
                }
                break;
            default:
                break;
            }
        }
    }

    Result GetSystemVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        int32_t version;
        nn::settings::fwdbg::GetSettingsItemValue(&version, sizeof (version), "news", "system_version");

        NN_LOG("%d\n", version);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetSystemVersion(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        const char* versionString = option.GetTarget();

        if (std::string(versionString) == "")
        {
            NN_LOG("You must specify a version.\n");
            NN_RESULT_SUCCESS;
        }

        char* end = nullptr;
        int32_t version = std::strtol(versionString, &end, 10);

        if (*end != '\0' || version <= 0)
        {
            NN_LOG("%s is not valid version.\n", versionString);
            NN_RESULT_SUCCESS;
        }

        nn::settings::fwdbg::SetSettingsItemValue("news", "system_version", &version, sizeof (version));

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result SetPassphrase(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        const char* passphrase = option.GetTarget();

        if (std::string(passphrase) == "")
        {
            NN_LOG("You must specify a passphrase.\n");
            NN_RESULT_SUCCESS;
        }

        const nn::ApplicationId NewsApplicationId = {0x0100000000001000};

        NN_RESULT_DO(nn::news::SetPassphrase(NewsApplicationId, passphrase));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result Post(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        const char* path = option.GetTarget();

        if (!devmenuUtil::IsAbsolutePath(path))
        {
            DEVMENUCOMMAND_LOG("'%s' is not an absolute path.\n", path);
            NN_RESULT_SUCCESS;
        }

        char* buffer = new char[NewsDataSizeMax];

        NN_ABORT_UNLESS(buffer);

        NN_UTIL_SCOPE_EXIT
        {
            delete[] buffer;
        };

        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        if (fileSize > NewsDataSizeMax)
        {
            DEVMENUCOMMAND_LOG("'%s' is too big.\n", path);
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer, static_cast<size_t>(fileSize)));

        NN_RESULT_DO(nn::news::PostLocalNews(buffer, static_cast<size_t>(fileSize)));

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetNewsList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        char* buffer = new char[NewsDataSizeMax];

        NN_ABORT_UNLESS(buffer);

        NN_UTIL_SCOPE_EXIT
        {
            delete[] buffer;
        };

        nn::news::Database db;

        NN_RESULT_DO(db.Open());

        NN_UTIL_SCOPE_EXIT
        {
            db.Close();
        };

        static nn::news::NewsRecord s_Records[nn::news::DataCountMax] = {};
        int count = 0;

        // 新着順
        NN_RESULT_DO(db.GetList(&count, s_Records, "", "received_at DESC, news_id DESC", 0, NN_ARRAY_SIZE(s_Records)));

        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("Count = %d\n", count);
        NN_LOG("----------------------------------------------------------------------\n");

        if (count > 0)
        {
            for (int i = 0; i < count; i++)
            {
                nn::news::Data data;

                NN_RESULT_DO(data.Open(s_Records[i]));

                NN_UTIL_SCOPE_EXIT
                {
                    data.Close();
                };

                size_t readSize;
                NN_RESULT_DO(data.Read(&readSize, 0, buffer, NewsDataSizeMax));

                NN_LOG("NewsId: %s\n", s_Records[i].newsId.value);

                DumpNewsInfo(buffer, readSize);

                NN_LOG("----------------------------------------------------------------------\n");
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetTopicList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        static nn::news::TopicId s_TopicIds[nn::news::TopicCountMax] = {};
        int count = 0;

        NN_RESULT_DO(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
            nn::news::SubscriptionStatusFilter_Subscribed));

        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("Subscribed Topic Count = %d\n", count);
        NN_LOG("----------------------------------------------------------------------\n");

        if (count > 0)
        {
            for (int i = 0; i < count; i++)
            {
                NN_LOG("%s\n", s_TopicIds[i].value);
            }

            NN_LOG("----------------------------------------------------------------------\n");
        }

        NN_RESULT_DO(nn::news::GetTopicList(&count, s_TopicIds, NN_ARRAY_SIZE(s_TopicIds),
            nn::news::SubscriptionStatusFilter_Unsubscribed));

        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("Unsubscribed Topic Count = %d\n", count);
        NN_LOG("----------------------------------------------------------------------\n");

        if (count > 0)
        {
            for (int i = 0; i < count; i++)
            {
                NN_LOG("%s\n", s_TopicIds[i].value);
            }

            NN_LOG("----------------------------------------------------------------------\n");
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearStorage(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        NN_RESULT_DO(nn::news::ClearStorage());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result ClearSubscriptionStatusAll(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        NN_RESULT_DO(nn::news::ClearSubscriptionStatusAll());

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage:\n"
        "  * configuration\n"
        "       " DEVMENUCOMMAND_NAME " news get-system-version\n"
        "       " DEVMENUCOMMAND_NAME " news set-system-version <version>\n"
        "       " DEVMENUCOMMAND_NAME " news set-passphrase <passphrase>\n"
        "  * news\n"
        "       " DEVMENUCOMMAND_NAME " news post <msgpack_absolute_path>\n"
        "       " DEVMENUCOMMAND_NAME " news get-news-list\n"
        "  * topic\n"
        "       " DEVMENUCOMMAND_NAME " news get-topic-list\n"
        "  * debug\n"
        "       " DEVMENUCOMMAND_NAME " news clear-storage\n"
        "       " DEVMENUCOMMAND_NAME " news clear-subscription-status-all\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"get-system-version", GetSystemVersion},
        {"set-system-version", SetSystemVersion},
        {"set-passphrase", SetPassphrase},
        {"post", Post},
        {"get-news-list", GetNewsList},
        {"get-topic-list", GetTopicList},
        {"clear-storage", ClearStorage},
        {"clear-subscription-status-all", ClearSubscriptionStatusAll},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result NewsCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu news command. See '" DEVMENUCOMMAND_NAME " news --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}

#endif // #if defined (NN_TOOL_DEVMENUCOMMANDSYSTEM)
