﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <curl/curl.h>
#include <nn/netdiag/netdiag_GlobalIpAddressApi.h>
#include <nn/netdiag/netdiag_NatApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiForNetworkMiddleware.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nifm/nifm_ApiInternetConnectionStatus.h>
#include <nn/socket.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_NetworkCommand.h"
#include "network/DevMenuCommand_NetworkUtil.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    Result ConfirmInternetConnection(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        *outValue = false;

        if (option.HasKey("--disconnect"))
        {
            nn::nifm::NetworkConnection networkConnetion;
            NN_RESULT_DO(nn::nifm::SetRequestRequirementPreset(networkConnetion.GetRequestHandle(), nn::nifm::RequirementPreset_None));
            networkConnetion.SubmitRequestAndWait();
            NN_RESULT_DO(networkConnetion.GetResult());
        }

        nn::nifm::InternetConnectionStatus internetConnectionStatus;
        nn::nifm::ClientId clientId = nn::nifm::GetClientId();

        NN_UTIL_SCOPE_EXIT
        {
            clientId = { nn::nifm::InvalidClientIdValue };
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(clientId));
        };

        if (option.HasKey("--exclusive"))
        {
            NN_RESULT_DO(nn::nifm::SetExclusiveClient(clientId));

            // プロセス独占後，全インターネット接続が切断されるまで最長で30秒待つ
            nn::Result connectionResult = nn::ResultSuccess();
            for (int i = 0; i < 30; ++i)
            {
                connectionResult = nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus);
                if (connectionResult.IsFailure())
                {
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
            }
            NN_RESULT_THROW_UNLESS(connectionResult.IsFailure(), nn::nifm::ResultInternetConnection());
        }

        uint32_t tryCountMax = 1;
        if (option.HasKey("--try"))
        {
            tryCountMax = std::strtoul(option.GetValue("--try"), nullptr, 10);
            if (tryCountMax < 1)
            {
                DEVMENUCOMMAND_LOG("--try option requires count (> 0)\n");
                NN_RESULT_SUCCESS;
            }
        }

        uint32_t tryCount = 0;
        uint32_t tryIntervalInSecond = 0;
        do
        {
            NN_ALIGNAS(nn::os::ThreadStackAlignment) static char stackOfThread[4 * 1024];
            nn::os::ThreadType CancelThread;

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(tryIntervalInSecond));
            tryIntervalInSecond += 3; // 再試行ごとに 3s 増加

            // 疎通確認を行う利用要求の提出
            nn::nifm::NetworkConnection networkConnection;
            NN_RESULT_DO(nn::nifm::SetRequestConnectionConfirmationOption(networkConnection.GetRequestHandle(), nn::nifm::ConnectionConfirmationOption_Forced));

            // 45s 以内に成功しなければキャンセル
            networkConnection.SubmitRequest();
            NN_RESULT_DO(nn::os::CreateThread(&CancelThread,
                [](void* p) {
                auto pNetworkConnection = reinterpret_cast<nn::nifm::NetworkConnection*>(p);
                int milliseconds = 45 * 1000;
                bool isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;

                while (milliseconds > 0 && !isCompleted)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
                    milliseconds -= 16;
                    isCompleted = pNetworkConnection->GetRequestState() != nn::nifm::RequestState_OnHold;
                }

                if (!isCompleted)
                {
                    pNetworkConnection->CancelRequest();
                }
            },
                &networkConnection, &stackOfThread, sizeof(stackOfThread), 24));
            nn::os::StartThread(&CancelThread);
            nn::os::DestroyThread(&CancelThread);

            auto result = networkConnection.GetResult();
            if(result.IsFailure())
            {
                DEVMENUCOMMAND_LOG("Request Rejected(%d/%d): 0x%08x\n", tryCount + 1, tryCountMax, result.GetInnerValueForDebug());
                continue;
            }

            NN_RESULT_DO(nn::nifm::GetInternetConnectionStatus(&internetConnectionStatus));

            DEVMENUCOMMAND_LOG("Internet Connection: ");
            switch (internetConnectionStatus.internetAvailability)
            {
            case nifm::InternetAvailability_Confirmed:
                DEVMENUCOMMAND_LOG("Confirmed\n");
                *outValue = true;
                break;
            case nifm::InternetAvailability_NotConfirmed:
                DEVMENUCOMMAND_LOG("NotConfirmed\n");
                break;
            case nifm::InternetAvailability_InProgress:
                DEVMENUCOMMAND_LOG("InProgress\n");
                break;
            case nifm::InternetAvailability_Invalid:
                DEVMENUCOMMAND_LOG("Invalid\n");
                break;
            default:
                NN_ABORT("Invalid internetConnectionStatus.internetAvailability");
                break;
            }

        } while (!*outValue && ++tryCount < tryCountMax);

        NN_RESULT_SUCCESS;
    }

    Result GetGlobalIpAddr(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        DEVMENUCOMMAND_LOG("Waiting for network availability...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if(!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::netdiag::GlobalIpAddress addr;
        auto result = nn::netdiag::GetGlobalIpAddress(&addr);
        if (result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("[RESULT] failed. module=%d, description=%d, result=0x%08x\n",
                result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        DEVMENUCOMMAND_LOG("[RESULT] %s\n", addr.value);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetNatType(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        DEVMENUCOMMAND_LOG("Waiting for network availability...\n");
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::netdiag::NatType natType;
        auto result = nn::netdiag::DetectNatType(&natType);

        if (result.IsFailure())
        {
            DEVMENUCOMMAND_LOG("[RESULT] failed. module=%d, description=%d, result=0x%08x\n",
                result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        switch (natType)
        {
        case nn::netdiag::NatType_A:
            {
                DEVMENUCOMMAND_LOG("[RESULT] NAT type A\n");
            }
            break;
        case nn::netdiag::NatType_B:
            {
                DEVMENUCOMMAND_LOG("[RESULT] NAT type B\n");
            }
            break;
        case nn::netdiag::NatType_C:
            {
                DEVMENUCOMMAND_LOG("[RESULT] NAT type C\n");
            }
            break;
        case nn::netdiag::NatType_D:
            {
                DEVMENUCOMMAND_LOG("[RESULT] NAT type D\n");
            }
            break;
        case nn::netdiag::NatType_Z:
            {
                // サポート上 Z は F と表示
                // (http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=165372838)
                DEVMENUCOMMAND_LOG("[RESULT] NAT type F\n");
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result GetWlanMacAddr(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        nn::nifm::NetworkInterfaceInfo info;
        int outCount = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::EnumerateNetworkInterfaces(&info, &outCount, 1, nn::nifm::EnumerateNetworkInterfacesFilter_Ieee80211));

        if ( outCount == 0 )
        {
            NN_LOG("No available network interfaces.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        const auto& m = info.macAddress.data;
        char buffer[sizeof("00:00:00:00:00:00")];
        nn::util::SNPrintf(buffer, sizeof(buffer), "%02x:%02x:%02x:%02x:%02x:%02x", m[0], m[1], m[2], m[3], m[4], m[5]);

        DEVMENUCOMMAND_LOG("[RESULT] %s\n", buffer);

        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    void ShowDevelopmentOnlyLog() NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("Error: Enable the network testing mode before using this command.\n(See \"Network\" tab in DevMenu or run `debug enable-nifm-testing-mode` command of DevMenuCommand)\n");
    }

    Result EnableWirelessCommunicationForTest(bool* outValue, const Option&) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nifm::SetWirelessCommunicationEnabledForTest(true))
            NN_RESULT_CATCH(nifm::ResultDevelopmentOnly)
            {
                ShowDevelopmentOnlyLog();
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result DisableWirelessCommunicationForTest(bool* outValue, const Option&) NN_NOEXCEPT
    {
        NN_RESULT_TRY(nifm::SetWirelessCommunicationEnabledForTest(false))
            NN_RESULT_CATCH(nifm::ResultDevelopmentOnly)
            {
                ShowDevelopmentOnlyLog();
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    Result TraceRoute(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        if (option.GetTarget()[0] == '\0')
        {
            DEVMENUCOMMAND_LOG("usage: " DEVMENUCOMMAND_NAME " network trace-route <hostname>\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        // 最大ホップ数
        const int HopCountMax = 32;
        // 同一ホップ数の Ping を何回繰り返すか
        const int TryCount = 3;

        // Ping のタイムアウト
        const nn::TimeSpan PingTimeout = nn::TimeSpan::FromSeconds(3);
        // タイムアウトが 3 秒 (3000 ms) なので x は 4 個でよい。
        const int TimeStringLength = sizeof ("<xxxx ms ") - 1;

        // ローカル環境でも traceroute できるようにする。
        nn::nifm::SetLocalNetworkMode(true);
        nn::nifm::SubmitNetworkRequestAndWait();

        if (!nn::nifm::IsNetworkAvailable())
        {
            DEVMENUCOMMAND_LOG("Network is not available.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        nn::nifm::NetworkProfileData profile = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::GetCurrentNetworkProfile(&profile));

        nn::socket::InAddr ipAddr = {};
        nn::socket::InAddr subnetMask = {};
        nn::socket::InAddr defaultGateway = {};
        nn::socket::InAddr dns1 = {};
        nn::socket::InAddr dns2 = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::GetCurrentIpConfigInfo(&ipAddr, &subnetMask, &defaultGateway, &dns1, &dns2));

        NN_LOG("======================================================================\n");
        NN_LOG("CurrentNetworkInfo:\n");
        NN_LOG("    Name           = %s\n", profile.name);

        switch (profile.networkInterfaceType)
        {
        case nn::nifm::NetworkInterfaceType_Ieee80211:
            NN_LOG("    Interface      = Ieee80211\n");
            break;
        case nn::nifm::NetworkInterfaceType_Ethernet:
            NN_LOG("    Interface      = Ethernet\n");
            break;
        default:
            NN_LOG("    Interface      = Unknown\n");
            break;
        }

        NN_LOG("    IpAddress      = %s\n", nn::socket::InetNtoa(ipAddr));
        NN_LOG("    SubnetMask     = %s\n", nn::socket::InetNtoa(subnetMask));
        NN_LOG("    DefaultGateway = %s\n", nn::socket::InetNtoa(defaultGateway));
        NN_LOG("    Dns1           = %s\n", nn::socket::InetNtoa(dns1));
        NN_LOG("    Dns2           = %s\n", nn::socket::InetNtoa(dns2));
        NN_LOG("======================================================================\n");

        nn::socket::InAddr targetAddr = {};

        if (!devmenuUtil::network::GetIpAddress(&targetAddr, option.GetTarget()))
        {
            DEVMENUCOMMAND_LOG("Your hostname (%s) could not be resolved.\n", option.GetTarget());
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_LOG("* %s [%s] ...\n", option.GetTarget(), nn::socket::InetNtoa(targetAddr));

        for (int i = 1; i <= HopCountMax; i++)
        {
            nn::socket::InAddr replySourceAddr = {};
            bool isReplied = false;

            char timeString[TimeStringLength * TryCount + 1] = {};
            char infoString[1024] = {};

            bool isReached = false;

            for (int c = 0; c < TryCount; c++)
            {
                int offset = TimeStringLength * c;

                nn::os::Tick start = nn::os::GetSystemTick();

                devmenuUtil::network::PingResult result = devmenuUtil::network::Ping(&isReached, &replySourceAddr,
                    targetAddr, i, PingTimeout);

                if (result == devmenuUtil::network::PingResult_Success)
                {
                    int length = nn::util::SNPrintf(&timeString[offset], sizeof (timeString) - offset,
                        "<%lld ms ", (nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds());

                    for (int n = length; n < TimeStringLength; n++)
                    {
                        timeString[offset + n] = ' ';
                    }

                    isReplied = true;
                }
                else
                {
                    nn::util::SNPrintf(&timeString[offset], sizeof (timeString) - offset, " *       ");

                    if (result == devmenuUtil::network::PingResult_TimedOut)
                    {
                        nn::util::SNPrintf(infoString, sizeof (infoString), "Timed out.");
                    }
                    else
                    {
                        nn::util::SNPrintf(infoString, sizeof (infoString), "Error occurred. (%s)",
                            devmenuUtil::network::GetPingResultString(result));
                    }
                }
            }

            if (isReplied)
            {
                devmenuUtil::network::HostName hostName = {};

                if (devmenuUtil::network::GetHostName(&hostName, replySourceAddr))
                {
                    nn::util::SNPrintf(infoString, sizeof (infoString), "%s [%s]",
                        hostName.value, nn::socket::InetNtoa(replySourceAddr));
                }
                else
                {
                    nn::util::SNPrintf(infoString, sizeof (infoString), "%s",
                        nn::socket::InetNtoa(replySourceAddr));
                }
            }

            NN_LOG("%2d %s %s\n", i, timeString, infoString);

            if (isReached)
            {
                break;
            }
        }

        *outValue = true;
        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " network confirm-internet-connection [--disconnect] [--exclusive] [--try <count>]\n"
        "       " DEVMENUCOMMAND_NAME " network get-global-ip-addr\n"
        "       " DEVMENUCOMMAND_NAME " network get-nat-type\n"
        "       " DEVMENUCOMMAND_NAME " network get-wlan-mac-addr\n"
        "       " DEVMENUCOMMAND_NAME " network enable-wireless-communication-for-test\n"
        "       " DEVMENUCOMMAND_NAME " network disable-wireless-communication-for-test\n"
        // "       " DEVMENUCOMMAND_NAME " network trace-route <hostname>\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"confirm-internet-connection", ConfirmInternetConnection},
        {"get-global-ip-addr", GetGlobalIpAddr},
        {"get-nat-type", GetNatType},
        {"get-wlan-mac-addr", GetWlanMacAddr},
        {"enable-wireless-communication-for-test", EnableWirelessCommunicationForTest},
        {"disable-wireless-communication-for-test", DisableWirelessCommunicationForTest},
        {"trace-route", TraceRoute},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result NetworkCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

    curl_global_init(CURL_GLOBAL_DEFAULT);
    NN_UTIL_SCOPE_EXIT
    {
        curl_global_cleanup();
    };

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu network command. See '" DEVMENUCOMMAND_NAME " network --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
