﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>

#include <nn/util/util_Optional.h>

#include <nn/ns/srv/ns_ApplicationStoragePriority.h>
#include <nn/ns/ns_ApplicationContentMetaSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>

#include "DevMenuCommand_HashUtil.h"
#include "DevMenuCommand_Result.h"

using namespace nn;

namespace devmenuUtil
{
namespace
{
nn::Result ReadContentMeta(nn::ncm::AutoBuffer* outBuffer, const nn::ncm::ContentMetaKey & key, ncm::StorageId storageId) NN_NOEXCEPT
{
    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase( &db, storageId ));

    nn::ncm::ContentId contentId;
    NN_RESULT_DO( db.GetContentIdByType( &contentId, key, nn::ncm::ContentType::Meta ) );

    nn::ncm::ContentStorage contentStorage;
    NN_RESULT_DO( nn::ncm::OpenContentStorage( &contentStorage, storageId ) );

    nn::ncm::Path filePath;
    contentStorage.GetPath( &filePath, contentId );

    std::string filePathStr = filePath.string;

    NN_RESULT_DO( nn::ncm::ReadContentMetaPath( outBuffer, filePathStr.c_str() ) );

    NN_RESULT_SUCCESS;
}

ncm::StorageContentMetaKey CompareAndReturn(util::optional<ncm::StorageContentMetaKey>& current, ncm::StorageContentMetaKey target) NN_NOEXCEPT
{
    if (current)
    {
        if (current->key.version < target.key.version)
        {
            return target;
        }
        else if (current->key.version > target.key.version)
        {
            return *current;
        }
        else
        {
            // バージョンが同じ場合、ストレージの優先度で比較
            nn::ns::srv::ApplicationStoragePriority p1(current->storageId);
            nn::ns::srv::ApplicationStoragePriority p2(target.storageId);

            if (p1 < p2)
            {
                return target;
            }
            else
            {
                return *current;
            }
        }
    }
    else
    {
        return target;
    }
}
Result GetContentMetaStorage(ncm::StorageId* outValue, const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
{
    if (!ncm::IsUniqueStorage(storageId))
    {
        NN_RESULT_DO(nn::ns::GetContentMetaStorage(outValue, key));
    }
    else
    {
        *outValue = storageId;
    }
    NN_RESULT_SUCCESS;
}

// DB に存在するかの確認
bool VerifyDatabase(const ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
{
    ncm::StorageId targetStorageId;
    if(GetContentMetaStorage(&targetStorageId, key, storageId).IsFailure())
    {
        return false;
    }

    nn::ncm::ContentMetaDatabase db;
    auto result = nn::ncm::OpenContentMetaDatabase(&db, targetStorageId);
    if (result.IsFailure())
    {
        return false;
    }
    bool has;
    result = db.Has(&has, key);
    if (result.IsFailure())
    {
        return false;
    }
    return has;
}
}

nn::Result GetApplicationRomIdHash(nn::ncm::Hash* outHash, const nn::ncm::ContentMetaKey& key, ncm::StorageId storageId) NN_NOEXCEPT
{
    ncm::StorageId targetStorageId;
    NN_RESULT_DO(GetContentMetaStorage(&targetStorageId, key, storageId));

    nn::ncm::AutoBuffer metaBuffer;
    NN_RESULT_DO(ReadContentMeta(&metaBuffer, key, targetStorageId));

    nn::ncm::PackagedContentMetaReader reader( metaBuffer.Get(), metaBuffer.GetSize() );
    *outHash = *reader.GetDigest();

    NN_RESULT_SUCCESS;
}

nn::Result VerifyApplicationAndPatchCombination(const nn::ncm::ApplicationId& id, nn::ncm::StorageId appStorage, nn::ncm::StorageId patchStorage) NN_NOEXCEPT
{
    // アプリとパッチの組み合わせを NS と同じようなロジックで取得する
    // Application Record を取得、アプリとパッチを決める
    int offset = 0;
    const int KeyCount = 16;
    ncm::StorageContentMetaKey keys[KeyCount];

    util::optional<ncm::StorageContentMetaKey> app;
    util::optional<ncm::StorageContentMetaKey> patch;

    for(;;)
    {
        int count;
        NN_RESULT_DO(ns::ListApplicationRecordContentMeta(&count, keys, KeyCount, id, offset));
        for (int i = 0; i < count; ++i)
        {
            if (keys[i].key.type == ncm::ContentMetaType::Application && VerifyDatabase(keys[i].key, appStorage))
            {
                app = CompareAndReturn(app, keys[i]);
            }
            else if (keys[i].key.type == ncm::ContentMetaType::Patch && VerifyDatabase(keys[i].key, patchStorage))
            {
                patch = CompareAndReturn(patch, keys[i]);
            }
        }
        offset += count;
        if (count != KeyCount)
        {
            break;
        }
    }
    NN_RESULT_THROW_UNLESS(appStorage == nn::ncm::StorageId::Any || app, ResultApplicationNotFound());
    NN_RESULT_THROW_UNLESS(patchStorage == nn::ncm::StorageId::Any || patch, ResultPatchNotFound());

    if (app && patch)
    {
        ncm::Hash appHash;
        NN_RESULT_DO(GetApplicationRomIdHash(&appHash, app->key, appStorage));
        {
            ncm::StorageId targetStorageId;
            NN_RESULT_DO(GetContentMetaStorage(&targetStorageId, patch->key, patchStorage));

            nn::ncm::AutoBuffer metaBuffer;
            NN_RESULT_DO(ReadContentMeta(&metaBuffer, patch->key, targetStorageId));

            nn::ncm::PackagedContentMetaReader reader( metaBuffer.Get(), metaBuffer.GetSize() );
            nn::ncm::PatchMetaExtendedDataReader patchReader(reader.GetExtendedData(), reader.GetExtendedDataSize());

            auto header = patchReader.GetHeader();
            for (int i = 0; i < static_cast<int>(header->historyCount); ++i)
            {
                auto historyHeader = patchReader.GetPatchHistoryHeader(i);
                if (historyHeader->key.type == nn::ncm::ContentMetaType::Application)
                {
                    if (std::memcmp(&appHash, &historyHeader->digest, sizeof(ncm::Hash)) == 0)
                    {
                        NN_RESULT_SUCCESS;
                    }
                }
            }
        }
        NN_RESULT_THROW(ResultBadApplicationForThePatch());
    }

    NN_RESULT_SUCCESS;
}

}
