﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/account.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/friends.h>
#include <nn/friends/friends_ApiAdmin.h>
#include <nn/friends/friends_Setting.h>
#include <nn/nifm.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_FriendsCommand.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

#if defined (NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#define strtoull _strtoui64
#endif

#define GET_USER(index) \
    do                                                      \
    {                                                       \
        user = GetUser(index);                              \
                                                            \
        if (!user)                                          \
        {                                                   \
            NN_LOG("You must specify an existent user.\n"); \
            *outValue = false;                              \
                                                            \
            NN_RESULT_SUCCESS;                              \
        }                                                   \
    }                                                       \
    while (NN_STATIC_CONDITION(0))

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    bool ConvertHexId(uint64_t* outId, const char* idString) NN_NOEXCEPT
    {
        if (nn::util::Strnlen(idString, 17) != 16)
        {
            return false;
        }

        char* end = nullptr;
        *outId = static_cast<uint64_t>(strtoull(idString, &end, 16));

        if (*end)
        {
            return false;
        }

        return true;
    }

    bool ConvertFlag(bool* outFlag, const char* flagString) NN_NOEXCEPT
    {
        if (nn::util::Strncmp(flagString, "true", sizeof ("true")) == 0)
        {
            *outFlag = true;
        }
        else if (nn::util::Strncmp(flagString, "false", sizeof ("false")) == 0)
        {
            *outFlag = false;
        }
        else
        {
            return false;
        }

        return true;
    }

    bool ConvertPresencePermission(nn::friends::PresencePermission* outPermission, const char* permissionString) NN_NOEXCEPT
    {
        if (nn::util::Strncmp(permissionString, "self", sizeof ("self")) == 0)
        {
            *outPermission = nn::friends::PresencePermission_Self;
        }
        else if (nn::util::Strncmp(permissionString, "favorite_friends", sizeof ("favorite_friends")) == 0)
        {
            *outPermission = nn::friends::PresencePermission_FavoriteFriends;
        }
        else if (nn::util::Strncmp(permissionString, "friends", sizeof ("friends")) == 0)
        {
            *outPermission = nn::friends::PresencePermission_Friends;
        }
        else
        {
            return false;
        }

        return true;
    }

    const char* ConvertPresencePermission(nn::friends::PresencePermission permission) NN_NOEXCEPT
    {
        if (permission == nn::friends::PresencePermission_Self)
        {
            return "self";
        }
        if (permission == nn::friends::PresencePermission_FavoriteFriends)
        {
            return "favorite_friends";
        }
        if (permission == nn::friends::PresencePermission_Friends)
        {
            return "friends";
        }

        return "";
    }

    nn::account::Uid GetUser(const char* indexString) NN_NOEXCEPT
    {
        if (!indexString)
        {
            return nn::account::InvalidUid;
        }

        char* end = nullptr;
        int index = static_cast<int>(std::strtol(indexString, &end, 10));

        if (*end)
        {
            return nn::account::InvalidUid;
        }

        nn::account::Uid users[nn::account::UserCountMax];
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

        if (index >= count)
        {
            return nn::account::InvalidUid;
        }

        return users[index];
    }

    bool ConnectNetwork() NN_NOEXCEPT
    {
        NN_LOG("Connect to network...\n");

        nn::nifm::SubmitNetworkRequestAndWait();

        NN_LOG("Connect to network... done!\n");

        return nn::nifm::IsNetworkAvailable();
    }

    Result WaitAsync(nn::friends::AsyncContext& async, const char* name) NN_NOEXCEPT
    {
        nn::os::SystemEvent completionEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(async.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        nn::Result result = async.GetResult();

        if (result.IsSuccess())
        {
            NN_LOG("%s ... done!\n", name);
        }
        else
        {
            NN_LOG("%s ... failed. e = %08x, code = %03d-%04d\n", name, result.GetInnerValueForDebug(), result.GetModule(), result.GetDescription());
        }

        if (nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
        {
            NN_LOG("Please link the available network service account.\n");
        }

        return result;
    }

    Result GetFriendList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (ConnectNetwork())
        {
            NN_LOG("Synchronize ...\n");

            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SyncFriendList(&async, user));

            if (WaitAsync(async, "Synchronize").IsFailure())
            {
                NN_LOG("[WARNING] The friend list cache is displayed.\n");
            }
        }
        else
        {
            NN_LOG("[WARNING] The network is not available. The friend list cache is displayed.\n");
        }

        static nn::friends::Friend friends[nn::friends::FriendCountMax] = {};
        int count;
        nn::friends::FriendFilter filter = {};

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, user, 0, nn::friends::FriendCountMax, filter));

        NN_LOG("+------------------+----------+-----------------------+\n");
        NN_LOG("  nsa_id           | favorite | nickname\n");
        NN_LOG("+------------------+----------+-----------------------+\n");

        for (int i = 0; i < count; i++)
        {
            NN_LOG("  %016llx | %s    | %s\n",
                friends[i].GetAccountId().id, friends[i].IsFavorite() ? "true " : "false", friends[i].GetNickname());
        }

        NN_LOG("+------------------+----------+-----------------------+\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result DeleteFriend(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--nsa_id"))
        {
            NN_LOG("You must specify a network service account ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::account::NetworkServiceAccountId accountId = {};

        if (!ConvertHexId(&accountId.id, option.GetValue("--nsa_id")))
        {
            NN_LOG("The network service account ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Delete friend ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::DeleteFriend(&async, user, accountId));

        NN_RESULT_TRY(WaitAsync(async, "Delete friend"))
            NN_RESULT_CATCH(nn::friends::ResultNotFriend)
            {
                NN_LOG("The specified user is not friend.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultNetworkServiceAccountNotExists)
            {
                NN_LOG("The specified user doesn't exist.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result ChangeFavoriteFlag(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--nsa_id"))
        {
            NN_LOG("You must specify a network service account ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::account::NetworkServiceAccountId accountId = {};

        if (!ConvertHexId(&accountId.id, option.GetValue("--nsa_id")))
        {
            NN_LOG("The network service account ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!option.HasKey("--flag"))
        {
            NN_LOG("You must specify a flag. Set one of [true|false]\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        bool flag = false;

        if (!ConvertFlag(&flag, option.GetValue("--flag")))
        {
            NN_LOG("The flag format is invalid. Set one of [true|false]\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Synchronize ...\n");
        {
            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SyncFriendList(&async, user));

            if (WaitAsync(async, "Synchronize").IsFailure())
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        }

        NN_LOG("Change favorite flag ...\n");
        {
            nn::friends::FriendSetting setting;

            NN_RESULT_TRY(setting.Initialize(user, accountId))
                NN_RESULT_CATCH(nn::friends::ResultNotFriend)
                {
                    NN_LOG("The specified user is not friend.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY;

            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(setting.ChangeFavoriteFlag(&async, flag));

            if (WaitAsync(async, "Change favorite flag").IsFailure())
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        }

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetSentFriendRequestList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Get sent friend request ...\n");

        static nn::friends::FriendRequest requests[nn::friends::SendFriendRequestCountMax] = {};
        int count;

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&async, &count, requests, user,
            0, nn::friends::SendFriendRequestCountMax, nn::friends::RequestListType_Sent));

        if (WaitAsync(async, "Get sent friend request").IsFailure())
        {
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("+------------------+------------------+-----------------------+\n");
        NN_LOG("  request_id       | nsa_id           | nickname\n");
        NN_LOG("+------------------+------------------+-----------------------+\n");

        for (int i = 0; i < count; i++)
        {
            NN_LOG("  %016llx | %016llx | %s\n",
                requests[i].GetRequestId().value, requests[i].GetAccountId().id, requests[i].GetNickname().name);
        }

        NN_LOG("+------------------+------------------+-----------------------+\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result SendFriendRequest(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--friend_code"))
        {
            NN_LOG("You must specify a friend code.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::FriendCode friendCode = {};
        nn::util::Strlcpy(friendCode.value, option.GetValue("--friend_code"), sizeof (friendCode.value));

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::ProfileExtra profile = {};

        NN_LOG("Find user ...\n");
        {
            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetProfileExtra(&async, &profile, user, friendCode));

            NN_RESULT_TRY(WaitAsync(async, "Find user"))
                NN_RESULT_CATCH(nn::friends::ResultInvalidFriendCodeFormat)
                {
                    if (std::isdigit(friendCode.value[0]) &&
                        std::isdigit(friendCode.value[1]) &&
                        std::isdigit(friendCode.value[2]) &&
                        std::isdigit(friendCode.value[3]) &&
                        friendCode.value[4] == '-' &&
                        std::isdigit(friendCode.value[5]) &&
                        std::isdigit(friendCode.value[6]) &&
                        std::isdigit(friendCode.value[7]) &&
                        std::isdigit(friendCode.value[8]) &&
                        friendCode.value[9] == '-' &&
                        std::isdigit(friendCode.value[10]) &&
                        std::isdigit(friendCode.value[11]) &&
                        std::isdigit(friendCode.value[12]) &&
                        std::isdigit(friendCode.value[13]))
                    {
                        // TORIAEZU: チェックサムの仕様を表に出すかどうか未定なので、一見正しいフォーマットは NotExists とする。
                        NN_LOG("The specified user doesn't exist.\n");
                    }
                    else
                    {
                        NN_LOG("The friend code format is wrong.\n");
                    }
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(nn::friends::ResultNetworkServiceAccountNotExists)
                {
                    NN_LOG("The specified user doesn't exist.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY;
        }

        NN_LOG("Send friend request ...\n");
        {
            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SendFriendRequest(&async, user, profile.GetAccountId(), nn::friends::RequestType_FriendCode));

            NN_RESULT_TRY(WaitAsync(async, "Send friend request"))
                NN_RESULT_CATCH(nn::friends::ResultOwnNetworkServiceAccountSpecified)
                {
                    NN_LOG("The specified user is oneself.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(nn::friends::ResultAlreadyFriend)
                {
                    NN_LOG("The specified user is already friend.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(nn::friends::ResultAlreadyRequested)
                {
                    NN_LOG("The friend request has already been sent to the specified user.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(nn::friends::ResultFriendRequestReceptionNotAllowed)
                {
                    NN_LOG("The specified user is not allowed friend request.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH(nn::friends::ResultBlockedByMe)
                {
                    NN_LOG("The specified user is blocked by me.\n");
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
                NN_RESULT_CATCH_ALL
                {
                    *outValue = false;

                    NN_RESULT_SUCCESS;
                }
            NN_RESULT_END_TRY;
        }

        *outValue = true;

        NN_RESULT_SUCCESS;
    } // NOLINT(impl/function_size)

    Result CancelFriendRequest(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--request_id"))
        {
            NN_LOG("You must specify a request ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::RequestId requestId = {};

        if (!ConvertHexId(&requestId.value, option.GetValue("--request_id")))
        {
            NN_LOG("The request ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Cancel friend request ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::CancelFriendRequest(&async, user, requestId));

        NN_RESULT_TRY(WaitAsync(async, "Cancel friend request"))
            NN_RESULT_CATCH(nn::friends::ResultRequestStateAlreadyChanged)
            {
                NN_LOG("The specified request was not pending status.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultRequestNotFound)
            {
                NN_LOG("The specified request was not found.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetReceivedFriendRequestList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Get received friend request ...\n");

        static nn::friends::FriendRequest requests[100] = {};
        int count;

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&async, &count, requests, user,
            0, 100, nn::friends::RequestListType_Received));

        if (WaitAsync(async, "Get received friend request").IsFailure())
        {
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("+------------------+------------------+-----------------------+\n");
        NN_LOG("  request_id       | nsa_id           | nickname\n");
        NN_LOG("+------------------+------------------+-----------------------+\n");

        for (int i = 0; i < count; i++)
        {
            NN_LOG("  %016llx | %016llx | %s\n",
                requests[i].GetRequestId().value, requests[i].GetAccountId().id, requests[i].GetNickname().name);
        }

        NN_LOG("+------------------+------------------+-----------------------+\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result AcceptFriendRequest(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--request_id"))
        {
            NN_LOG("You must specify a request ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::RequestId requestId = {};

        if (!ConvertHexId(&requestId.value, option.GetValue("--request_id")))
        {
            NN_LOG("The request ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Accept friend request ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&async, user, requestId));

        NN_RESULT_TRY(WaitAsync(async, "Accept friend request"))
            NN_RESULT_CATCH(nn::friends::ResultRequestStateAlreadyChanged)
            {
                NN_LOG("The specified request was not pending status.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultRequestNotFound)
            {
                NN_LOG("The specified request was not found.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result RejectFriendRequest(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--request_id"))
        {
            NN_LOG("You must specify a request ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::RequestId requestId = {};

        if (!ConvertHexId(&requestId.value, option.GetValue("--request_id")))
        {
            NN_LOG("The request ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Reject friend request ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::RejectFriendRequest(&async, user, requestId));

        NN_RESULT_TRY(WaitAsync(async, "Reject friend request"))
            NN_RESULT_CATCH(nn::friends::ResultRequestStateAlreadyChanged)
            {
                NN_LOG("The specified request was not pending status.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultRequestNotFound)
            {
                NN_LOG("The specified request was not found.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetBlockedUserList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (ConnectNetwork())
        {
            NN_LOG("Synchronize ...\n");

            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SyncBlockedUserList(&async, user));

            if (WaitAsync(async, "Synchronize").IsFailure())
            {
                NN_LOG("[WARNING] The blocked user list cache is displayed.\n");
            }
        }
        else
        {
            NN_LOG("[WARNING] The network is not available. The blocked user list cache is displayed.\n");
        }

        static nn::friends::BlockedUser users[nn::friends::BlockedUserCountMax] = {};
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, users, user, 0, nn::friends::BlockedUserCountMax));

        NN_LOG("+------------------+-----------------------+\n");
        NN_LOG("  nsa_id           | nickname\n");
        NN_LOG("+------------------+-----------------------+\n");

        for (int i = 0; i < count; i++)
        {
            NN_LOG("  %016llx | %s\n", users[i].GetAccountId().id, users[i].GetNickname());
        }

        NN_LOG("+------------------+-----------------------+\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result BlockUser(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--nsa_id"))
        {
            NN_LOG("You must specify a network service account ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::account::NetworkServiceAccountId accountId = {};

        if (!ConvertHexId(&accountId.id, option.GetValue("--nsa_id")))
        {
            NN_LOG("The network service account ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Block user ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::BlockUser(&async, user, accountId, nn::friends::BlockReason_BadFriendRequest));

        NN_RESULT_TRY(WaitAsync(async, "Block user"))
            NN_RESULT_CATCH(nn::friends::ResultAlreadyBlocked)
            {
                NN_LOG("The specified user has already been blocked.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultNetworkServiceAccountNotExists)
            {
                NN_LOG("The specified user doesn't exist.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result UnblockUser(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--nsa_id"))
        {
            NN_LOG("You must specify a network service account ID.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::account::NetworkServiceAccountId accountId = {};

        if (!ConvertHexId(&accountId.id, option.GetValue("--nsa_id")))
        {
            NN_LOG("The network service account ID format is invalid.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Unblock user ...\n");

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::UnblockUser(&async, user, accountId));

        NN_RESULT_TRY(WaitAsync(async, "Unblock user"))
            NN_RESULT_CATCH(nn::friends::ResultNotBlocked)
            {
                NN_LOG("The specified user is not blocked.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH(nn::friends::ResultNetworkServiceAccountNotExists)
            {
                NN_LOG("The specified user doesn't exist.\n");
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetPlayHistoryList(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        static nn::friends::PlayHistory histories[nn::friends::PlayHistoryCountMax] = {};
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, user, 0, nn::friends::PlayHistoryCountMax));

        NN_LOG("+------------------+--------------+------------------+-------------------------------------------------\n");
        NN_LOG("  nsa_id           | local_played | application_id   | name(own, opponent)\n");
        NN_LOG("+------------------+--------------+------------------+-------------------------------------------------\n");

        for (int i = 0; i < count; i++)
        {
            if (histories[i].HasNetworkServiceAccountId())
            {
                NN_LOG("  %016llx | %s        | %016llx | (%s, %s)\n",
                    histories[i].GetAccountId().id, histories[i].IsLocalPlayed() ? "true " : "false",
                    histories[i].GetPlayRecord().appInfo.appId.value, histories[i].GetPlayRecord().myName.name, histories[i].GetPlayRecord().name.name);
            }
            else
            {
                NN_LOG("  -                | %s        | %016llx | (%s, %s)\n",
                    histories[i].IsLocalPlayed() ? "true " : "false",
                    histories[i].GetPlayRecord().appInfo.appId.value, histories[i].GetPlayRecord().myName.name, histories[i].GetPlayRecord().name.name);
            }
        }

        NN_LOG("+------------------+--------------+------------------+-------------------------------------------------\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result DeletePlayHistory(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::DeletePlayHistory(user));

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetUserSetting(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (ConnectNetwork())
        {
            NN_LOG("Synchronize ...\n");

            nn::friends::AsyncContext async;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SyncUserSetting(&async, user));

            if (WaitAsync(async, "Synchronize").IsFailure())
            {
                NN_LOG("[WARNING] The user setting cache is displayed.\n");
            }
        }
        else
        {
            NN_LOG("[WARNING] The network is not available. The user setting cache is displayed.\n");
        }

        nn::friends::UserSetting setting;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(user));

        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("FriendCode:             %s\n", setting.GetFriendCode().value);
        NN_LOG("PresencePermission:     %s\n", ConvertPresencePermission(setting.GetPresencePermission()));
        NN_LOG("FriendRequestReception: %s\n", setting.GetFriendRequestReception() ? "true" : "false");
        NN_LOG("----------------------------------------------------------------------\n");

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result IssueFriendCode(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Issue friend code ...\n");

        nn::friends::UserSetting setting;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(user));

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.IssueFriendCode(&async));

        NN_RESULT_TRY(WaitAsync(async, "Issue friend code"))
            NN_RESULT_CATCH(nn::friends::ResultDisallowFriendCodeIssue)
            {
                NN_LOG("The issue of the friend code is disallowed. (next issuable time = %lld, posix time)\n",
                    setting.GetFriendCodeNextIssuableTime().value);
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH_ALL
            {
                *outValue = false;

                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_LOG("New friend code = %s\n", setting.GetFriendCode().value);
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result ChangePresencePermission(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--permission"))
        {
            NN_LOG("You must specify a permission.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        nn::friends::PresencePermission permission = nn::friends::PresencePermission_Self;

        if (!ConvertPresencePermission(&permission, option.GetValue("--permission")))
        {
            NN_LOG("The permission format is invalid. Set one of [self|favorite_friends|friends]\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Change presence permission ...\n");

        nn::friends::UserSetting setting;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(user));

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.ChangePresencePermission(&async, permission));

        if (WaitAsync(async, "Change presence permission").IsFailure())
        {
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result ChangeFriendRequestReceptionFlag(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        nn::account::Uid user = nn::account::InvalidUid;
        GET_USER(option.GetTarget());

        if (!option.HasKey("--flag"))
        {
            NN_LOG("You must specify a flag.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        bool flag = false;

        if (!ConvertFlag(&flag, option.GetValue("--flag")))
        {
            NN_LOG("The flag format is invalid. Set one of [true|false]\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        if (!ConnectNetwork())
        {
            NN_LOG("The network is not available.\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        NN_LOG("Change friend request reception flag ...\n");

        nn::friends::UserSetting setting;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.Initialize(user));

        nn::friends::AsyncContext async;
        NN_ABORT_UNLESS_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&async, flag));

        if (WaitAsync(async, "Change friend request reception flag").IsFailure())
        {
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result GetBackgroundProcessingStatus(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        NN_UNUSED(option);

        bool isEnabled;
        nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, 1, "friends", "background_processing");

        NN_LOG("%s\n", isEnabled ? "Enabled" : "Disabled");
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    Result SetBackgroundProcessingStatus(bool* outValue, const Option& option) NN_NOEXCEPT
    {
        const char* modeString = option.GetTarget();

        if (std::string(modeString) == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            *outValue = false;

            NN_RESULT_SUCCESS;
        }
        else if (std::string(modeString) != "enable" && std::string(modeString) != "disable")
        {
            NN_LOG("%s is not valid status. Set one of [enable|disable].\n", modeString);
            *outValue = false;

            NN_RESULT_SUCCESS;
        }

        bool isEnabled = (std::string(modeString) == "enable");

        nn::settings::fwdbg::SetSettingsItemValue("friends", "background_processing", &isEnabled, 1);

        // 再起動を促す。
        NN_LOG("\033[32m*** Please reboot the target to allow changes to take effect.\033[m\n");
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    const char HelpMessage[] =
        "usage:\n"
        "  * friend\n"
        "       " DEVMENUCOMMAND_NAME " friends get-friend-list <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends delete-friend <index> --nsa_id <nsa_id>\n"
        "       " DEVMENUCOMMAND_NAME " friends change-favorite-flag <index> --nsa_id <nsa_id> --flag <true|false>\n"
        "  * friend-request (sent)\n"
        "       " DEVMENUCOMMAND_NAME " friends get-sent-friend-request-list <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends send-friend-request <index> --friend_code <friend_code>\n"
        "       " DEVMENUCOMMAND_NAME " friends cancel-friend-request <index> --request_id <request_id>\n"
        "  * friend-request (received)\n"
        "       " DEVMENUCOMMAND_NAME " friends get-received-friend-request-list <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends accept-friend-request <index> --request_id <request_id>\n"
        "       " DEVMENUCOMMAND_NAME " friends reject-friend-request <index> --request_id <request_id>\n"
        "  * blocked-user\n"
        "       " DEVMENUCOMMAND_NAME " friends get-blocked-user-list <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends block-user <index> --nsa_id <nsa_id>\n"
        "       " DEVMENUCOMMAND_NAME " friends unblock-user <index> --nsa_id <nsa_id>\n"
        "  * play-history\n"
        "       " DEVMENUCOMMAND_NAME " friends get-play-history-list <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends delete-play-history <index>\n"
        "  * user-setting\n"
        "       " DEVMENUCOMMAND_NAME " friends get-user-setting <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends issue-friend-code <index>\n"
        "       " DEVMENUCOMMAND_NAME " friends change-presence-permission <index> --permission <self|favorite_friends|friends>\n"
        "       " DEVMENUCOMMAND_NAME " friends change-friend-request-reception-flag <index> --flag <true|false>\n"
//        "       " DEVMENUCOMMAND_NAME " friends change-playlog-permission <index> --permission <self|favorite_friends|friends|everyone>\n"
//        "       " DEVMENUCOMMAND_NAME " friends clear-playlog <index>\n"
        "  * system-setting\n"
        "       " DEVMENUCOMMAND_NAME " friends get-background-processing-status\n"
        "       " DEVMENUCOMMAND_NAME " friends set-background-processing-status <enable|disable>\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        // friend
        {"get-friend-list", GetFriendList},
        {"delete-friend", DeleteFriend},
        {"change-favorite-flag", ChangeFavoriteFlag},
        // friend-request (sent)
        {"get-sent-friend-request-list", GetSentFriendRequestList},
        {"send-friend-request", SendFriendRequest},
        {"cancel-friend-request", CancelFriendRequest},
        // friend-request (received)
        {"get-received-friend-request-list", GetReceivedFriendRequestList},
        {"accept-friend-request", AcceptFriendRequest},
        {"reject-friend-request", RejectFriendRequest},
        // blocked-user
        {"get-blocked-user-list", GetBlockedUserList},
        {"block-user", BlockUser},
        {"unblock-user", UnblockUser},
        // play-history
        {"get-play-history-list", GetPlayHistoryList},
        {"delete-play-history", DeletePlayHistory},
        // user-setting
        {"get-user-setting", GetUserSetting},
        {"issue-friend-code", IssueFriendCode},
        {"change-presence-permission", ChangePresencePermission},
        {"change-friend-request-reception-flag", ChangeFriendRequestReceptionFlag},
        // system-setting
        {"get-background-processing-status", GetBackgroundProcessingStatus},
        {"set-background-processing-status", SetBackgroundProcessingStatus},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result FriendsCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;

        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;

        NN_RESULT_SUCCESS;
    }

    nn::account::InitializeForAdministrator();
    nn::nifm::Initialize();

    nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu friends command. See '" DEVMENUCOMMAND_NAME " friends --help'.\n", option.GetSubCommand());
    *outValue = false;

    NN_RESULT_SUCCESS;
}
