﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/account/account_Result.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

#include <sstream>
#include <curl/curl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/nim/srv/nim_DynamicRightsUrl.h>

#include "DevMenuCommand_Log.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_ELicenseUtil.h"

using namespace nn;

namespace devmenuUtil {

    const std::pair<nn::nim::ELicenseType, const char *> g_ELicenseTypeStringMap[] = {
        {nn::nim::ELicenseType::Unknown,                     "unknown"},
        {nn::nim::ELicenseType::DeviceLinkedPermanent,       "device_linked_permanent"},
        {nn::nim::ELicenseType::Permanent,                   "permanent"},
        {nn::nim::ELicenseType::AccountRestrictivePermanent, "account_restrictive_permanent"},
        {nn::nim::ELicenseType::Temporary,                   "temporary"},
    };

    const std::pair<nn::nim::ELicenseStatus, const char *> g_ELicenseStatusStringMap[] = {
        {nn::nim::ELicenseStatus::Unknown,                                            "unknown"},
        {nn::nim::ELicenseStatus::Assignable,                                         "assignable"},
        {nn::nim::ELicenseStatus::AlreadyAssigned,                                    "already_assigned"},
        {nn::nim::ELicenseStatus::NotAssignableSinceTitleDoesNotSupportDynamicRights, "not_supported"},
        {nn::nim::ELicenseStatus::NotAssignableSinceLimitExceeded,                    "limit_exceeded"},
    };

    const std::pair<nn::nim::RevokeReason, const char *> g_RevokeReasonStringMap[] = {
        {nn::nim::RevokeReason::Unknown,                "unknown"},
        {nn::nim::RevokeReason::Expired,                "expired"},
        {nn::nim::RevokeReason::OtherDeviceAssigned,    "other_device_assigned"},
        {nn::nim::RevokeReason::DuplicateUsageDetected, "duplicate_usage_detected"},
    };

    const std::pair<nn::es::ELicenseScope, const char *> g_ELicenseScopeStringMap[] = {
        {nn::es::ELicenseScope::Account, "account"},
        {nn::es::ELicenseScope::Device,  "device"},
    };

#define DEFINE_GET_STRING_FROM(TYPE, MAP) \
    const char* GetStringFrom(const TYPE& v) NN_NOEXCEPT \
    { \
        for (const auto& p : MAP) \
        { \
            if (v == p.first) \
            { \
                return p.second; \
            } \
        } \
        return "EOC"; \
    }
    DEFINE_GET_STRING_FROM(nn::nim::ELicenseType,   g_ELicenseTypeStringMap)
    DEFINE_GET_STRING_FROM(nn::nim::ELicenseStatus, g_ELicenseStatusStringMap)
    DEFINE_GET_STRING_FROM(nn::nim::RevokeReason,   g_RevokeReasonStringMap)
    DEFINE_GET_STRING_FROM(nn::es::ELicenseScope,   g_ELicenseScopeStringMap)
#undef DEFINE_GET_STRING_FROM

    const char* GetExpireDateString(char* out, int maxOutLen, const es::ExpireDate& expireDate) NN_NOEXCEPT
    {
        if (expireDate.IsPermanent())
        {
            util::SNPrintf(out, maxOutLen, "Permanent");
            return out;
        }
        return GetDateTimeString(out, maxOutLen, expireDate.time);

    }

    void PrintAvailableELicense(const nim::AvailableELicense& l) NN_NOEXCEPT
    {
        DEVMENUCOMMAND_LOG("RightsId:                      0x%016llx\n", l.rightsId);
        DEVMENUCOMMAND_LOG("AccountId:                     0x%016llx\n", l.naId);
        DEVMENUCOMMAND_LOG("LicenseType:   %34s\n", GetStringFrom(l.licenseType));
        DEVMENUCOMMAND_LOG("LicenseStatus: %34s\n", GetStringFrom(l.licenseStatus));
        DEVMENUCOMMAND_LOG("\n");
    }

    void PrintAssignedELicense(const nim::AssignedELicense& l) NN_NOEXCEPT
    {
        char idString[64];
        DEVMENUCOMMAND_LOG("ELicenseId:    %34s\n", l.eLicenseId.ToString(idString, sizeof(idString)));
        DEVMENUCOMMAND_LOG("RightsId:                      0x%016llx\n", l.rightsId);
        DEVMENUCOMMAND_LOG("AccountId:                     0x%016llx\n", l.naId);
        DEVMENUCOMMAND_LOG("LicenseType:   %34s\n", GetStringFrom(l.licenseType));
        DEVMENUCOMMAND_LOG("\n");
    }

    void PrintELicenseInfoWrapper(const es::ELicenseInfoWrapper& e) NN_NOEXCEPT
    {
        char idString[64];
        char expireDateString[64];
        DEVMENUCOMMAND_LOG("ELicenseId:             %34s\n", e.info.eLicenseId.ToString(idString, sizeof(idString)));
        DEVMENUCOMMAND_LOG("RightsId:                               0x%016llx\n", e.info.rightsId);
        DEVMENUCOMMAND_LOG("TicketId:                               0x%016llx\n", e.info.ticketId);
        DEVMENUCOMMAND_LOG("OwnerAccountId:                         0x%016llx\n", e.info.ownerNaId);
        DEVMENUCOMMAND_LOG("ExpireDate:         %12d %25s\n", e.info.expireDate.time.value, GetExpireDateString(expireDateString, sizeof(expireDateString), e.info.expireDate));
        DEVMENUCOMMAND_LOG("TicketOwnerVaId:        %34u\n", e.info.ticketOwnerVaId);
        DEVMENUCOMMAND_LOG("Scope:                  %34s\n", GetStringFrom(e.info.scope));
        DEVMENUCOMMAND_LOG("HasTicket:              %34s\n", e.HasTicket() ? "true" : "false");
        DEVMENUCOMMAND_LOG("\n");
    }

    void PrintELicenseInfoForSystemWrapper(const es::ELicenseInfoForSystemWrapper& e) NN_NOEXCEPT
    {
        char idString[64];
        char expireDateString[64];
        DEVMENUCOMMAND_LOG("ELicenseId:             %34s\n", e.info.eLicenseId.ToString(idString, sizeof(idString)));
        DEVMENUCOMMAND_LOG("RightsId:                               0x%016llx\n", e.info.rightsId);
        DEVMENUCOMMAND_LOG("TicketId:                               0x%016llx\n", e.info.ticketId);
        DEVMENUCOMMAND_LOG("OwnerAccountId:                         0x%016llx\n", e.info.ownerNaId);
        DEVMENUCOMMAND_LOG("ExpireDate:         %12d %25s\n", e.info.expireDate.time.value, GetExpireDateString(expireDateString, sizeof(expireDateString), e.info.expireDate));
        DEVMENUCOMMAND_LOG("TicketOwnerVaId:        %34u\n", e.info.ticketOwnerVaId);
        DEVMENUCOMMAND_LOG("Scope:                  %34s\n", GetStringFrom(e.info.scope));
        DEVMENUCOMMAND_LOG("HasTicket:              %34s\n", e.HasTicket() ? "true" : "false");
        DEVMENUCOMMAND_LOG("ServerInteraction:      %34s\n", e.info.IsServerInteractionRecommended() ? "recommended" : "not recommended");
        DEVMENUCOMMAND_LOG("\n");
    }

    Result GetRightsIdsFromTarget(int* outCount, es::RightsId* outValues, int maxCount, const Option& option) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_MINMAX(maxCount, 0, RightsIdsMax);
        NN_ABORT_UNLESS_MINMAX(option.GetTargetCount(), 0, maxCount);

        for (int i = 0; i < option.GetTargetCount(); i++)
        {
            outValues[i] = std::strtoull(option.GetTarget(i), nullptr, 16);
            if (outValues[i] <= 0)
            {
                DEVMENUCOMMAND_LOG("Invalid rights id:%d %s\n", i, option.GetTarget(i));
                *outCount = -1;
                NN_RESULT_SUCCESS;
            }
        }

        *outCount = option.GetTargetCount();
        NN_RESULT_SUCCESS;
    }

    Result GetELicenseIdsFromTarget(int* outCount, es::ELicenseId* outValues, int maxCount, const Option& option) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_MINMAX(maxCount, 0, ELicenseIdsMax);
        NN_ABORT_UNLESS_MINMAX(option.GetTargetCount(), 0, maxCount);

        for (int i = 0; i < option.GetTargetCount(); i++)
        {
            if (!outValues[i].FromString(option.GetTarget(i)))
            {
                DEVMENUCOMMAND_LOG("Invalid elicense id:%d %s\n", i, option.GetTarget(i));
                *outCount = -1;
                NN_RESULT_SUCCESS;
            }
        }

        *outCount = option.GetTargetCount();
        NN_RESULT_SUCCESS;
    }

    Result GetUidFromOption(account::Uid* outValue, const Option& option) NN_NOEXCEPT
    {
        int userIndex = option.HasKey("--account") ? std::strtol(option.GetValue("--account"), nullptr, 10) : 0;

        account::Uid uids[account::UserCountMax];
        int uidCount;
        NN_RESULT_DO(account::ListAllUsers(&uidCount, uids, account::UserCountMax));

        if (uidCount == 0 || userIndex >= uidCount)
        {
            DEVMENUCOMMAND_LOG("Not found user account index %d\n", userIndex);
            *outValue = account::InvalidUid;
            NN_RESULT_SUCCESS;
        }

        *outValue = uids[userIndex];
        NN_RESULT_SUCCESS;
    }

    Result GetELicenseTypeFromOption(util::optional<nim::ELicenseType>* outValue, const Option& option) NN_NOEXCEPT
    {
        if (option.HasKey("--type"))
        {
            auto str = option.GetValue("--type");
            for (const auto& p : g_ELicenseTypeStringMap)
            {
                if (strcmp(str, p.second) == 0)
                {
                    *outValue = p.first;
                    NN_RESULT_SUCCESS;
                }
            }

            auto type = std::strtol(str, nullptr, 10);
            if (type < 0 || static_cast<int>(nim::ELicenseType::EndOfContents) <= type)
            {
                DEVMENUCOMMAND_LOG("Invalid elicense type %d\n", type);
                *outValue = nim::ELicenseType::EndOfContents;
                NN_RESULT_SUCCESS;
            }
            *outValue = static_cast<nim::ELicenseType>(type);
        }
        else
        {
            *outValue = util::nullopt;
        }
        NN_RESULT_SUCCESS;
    }

    Result GetELicenseStatusFromOption(util::optional<nim::ELicenseStatus>* outValue, const Option& option) NN_NOEXCEPT
    {
        if (option.HasKey("--status"))
        {
            auto str = option.GetValue("--status");
            for (const auto& p : g_ELicenseStatusStringMap)
            {
                if (strcmp(str, p.second) == 0)
                {
                    *outValue = p.first;
                    NN_RESULT_SUCCESS;
                }
            }

            auto status = std::strtol(str, nullptr, 10);
            if (status < 0 || static_cast<int>(nim::ELicenseStatus::EndOfContents) <= status)
            {
                DEVMENUCOMMAND_LOG("Invalid elicense status %d\n", status);
                *outValue = nim::ELicenseStatus::EndOfContents;
                NN_RESULT_SUCCESS;
            }
            *outValue = static_cast<nim::ELicenseStatus>(status);
        }
        else
        {
            *outValue = util::nullopt;
        }
        NN_RESULT_SUCCESS;
    }

    Result GetNintendoAccountId(account::NintendoAccountId* outValue, const account::Uid& uid)
    {
        account::NetworkServiceAccountManager accountManager;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&accountManager, uid));

        NN_RESULT_TRY(accountManager.GetNintendoAccountId(outValue));
            NN_RESULT_CATCH(account::ResultNetworkServiceAccountUnavailable)
            {
                DEVMENUCOMMAND_LOG("User account isn't linked with nintendo account.\n");

                *outValue = account::InvalidNintendoAccountId;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    }

    Result GetTimeSpanFromOption(util::optional<TimeSpan>* outValue, const Option& option, const char* key) NN_NOEXCEPT
    {
        if (option.HasKey(key))
        {
            // TIMESPAN_SPEC ([0-9]+d|[0-9]+h|[0-9]+m|[0-9]+s|[0-9]+ms|[0-9]+us|[0-9]+ns)+"
            TimeSpan timespan;
            const char *spec = option.GetValue(key);
            const size_t length = std::strlen(spec);
            size_t scanned = 0;
            while (scanned < length)
            {
                long long number;
                char order[16];
                int current = 0;
                auto matched = sscanf(spec + scanned, " %lld%s%n", &number, order, &current);
                if (matched > 1)
                {
                    if (strcmp(order, "d") == 0)
                    {
                        timespan += TimeSpan::FromDays(number);
                    }
                    else if (strcmp(order, "h") == 0)
                    {
                        timespan += TimeSpan::FromHours(number);
                    }
                    else if (strcmp(order, "m") == 0)
                    {
                        timespan += TimeSpan::FromMinutes(number);
                    }
                    else if (strcmp(order, "s") == 0)
                    {
                        timespan += TimeSpan::FromSeconds(number);
                    }
                    else if (strcmp(order, "ms") == 0)
                    {
                        timespan += TimeSpan::FromMilliSeconds(number);
                    }
                    else if (strcmp(order, "us") == 0)
                    {
                        timespan += TimeSpan::FromMicroSeconds(number);
                    }
                    else if (strcmp(order, "ns") == 0)
                    {
                        timespan += TimeSpan::FromNanoSeconds(number);
                    }
                    else
                    {
                        DEVMENUCOMMAND_LOG("invalid time spec detected: '%s' (%s)\n", spec + scanned, spec);
                        *outValue = util::nullopt;
                        break;
                    }
                }
                else
                {
                    DEVMENUCOMMAND_LOG("invalid time spec detected: '%s' (%s)\n", spec + scanned, spec);
                    *outValue = util::nullopt;
                    break;
                }

                scanned += current;
            }
            *outValue = timespan;
        }
        else
        {
            *outValue = util::nullopt;
        }

        NN_RESULT_SUCCESS;
    }

    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<ssl::Context*>(pSslContext);
        auto result = context.Create(ssl::Context::SslVersion_Auto);
        if(result.IsSuccess())
        {
            ssl::CertStoreId deviceCertId;
            result = context.RegisterInternalPki(&deviceCertId, ssl::Context::InternalPki_DeviceClientCertDefault);
            if(result.IsFailure())
            {
                DEVMENUCOMMAND_LOG("Failed to register the device unique client certificate.\n");
            }
        }
        else
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }
        return CURLE_OK;
    }

    size_t CurlWriteFunction(char *data, size_t size, size_t nmemb, void* userdata) NN_NOEXCEPT
    {
        std::stringstream *stream = reinterpret_cast<std::stringstream*>(userdata);
        size_t bytesToWrite = size * nmemb;

        stream->write(data, bytesToWrite);
        if (!stream->good())
        {
            return 0;
        }

        return bytesToWrite;
    }

    bool GetByCurl(RAPIDJSON_NAMESPACE::Document* responseBody, const account::NintendoAccountId& naId, const char* path) NN_NOEXCEPT
    {
        CURL* curl = curl_easy_init();
        if(!curl)
        {
            DEVMENUCOMMAND_LOG("curl_easy_init failed.\n");
            return false;
        }

        // API URL
        auto url = std::string(nim::srv::DynamicRights::Dragons::ServerBaseUrl) + path;

        // HTTP Headers
        char nintendoAccountId[64];
        util::SNPrintf(nintendoAccountId, sizeof(nintendoAccountId), "Nintendo-Account-Id: %016llx", naId.id);
        curl_slist* headers = nullptr;
        headers = curl_slist_append(headers, "Content-Type: application/json");
        headers = curl_slist_append(headers, "Accept: application/json");
        headers = curl_slist_append(headers, "Charsets: UTF-8");
        headers = curl_slist_append(headers, nintendoAccountId);

        std::stringstream responseStream;
        curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
        curl_easy_setopt(curl, CURLOPT_HTTPGET, 1);
        curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
        curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, &responseStream);
        CURLcode curlCode = curl_easy_perform(curl);

        bool ret = true;
        if(curlCode != CURLE_OK)
        {
            DEVMENUCOMMAND_LOG("Request Failed. curlcode:%d (%s)\n", curlCode, curl_easy_strerror(curlCode));
            ret = false;
        }

        if (responseStream.str().length() > 0)
        {
            const char* contentType;
            curlCode = curl_easy_getinfo(curl, CURLINFO_CONTENT_TYPE, &contentType);
            if (curlCode == CURLE_OK)
            {
                DEVMENUCOMMAND_LOG("Content-Type: %s\n", contentType);
                if (std::string(contentType).find("application/json") != std::string::npos)
                {
                    responseBody->Parse(responseStream.str().c_str());
                }
                else
                {
                    DEVMENUCOMMAND_LOG("Unexpected Response: %s: %s", contentType, responseStream.str().c_str());
                    ret = false;
                }
            }
        }

        curl_easy_cleanup(curl);
        if (headers != nullptr)
        {
            curl_slist_free_all(headers);
        }
        return ret;
    }

    bool PostByCurl(RAPIDJSON_NAMESPACE::Document* responseBody, const account::NintendoAccountId& naId, const char* path, RAPIDJSON_NAMESPACE::Document& requestBody) NN_NOEXCEPT
    {
        CURL* curl = curl_easy_init();
        if(!curl)
        {
            DEVMENUCOMMAND_LOG("curl_easy_init failed.\n");
            return false;
        }

        // API URL
        auto url = std::string(nim::srv::DynamicRights::Dragons::ServerBaseUrl) + path;

        // HTTP Headers
        char nintendoAccountId[64];
        util::SNPrintf(nintendoAccountId, sizeof(nintendoAccountId), "Nintendo-Account-Id: %016llx", naId.id);
        curl_slist* headers = nullptr;
        headers = curl_slist_append(headers, "Content-Type: application/json");
        headers = curl_slist_append(headers, "Accept: application/json");
        headers = curl_slist_append(headers, "Charsets: UTF-8");
        headers = curl_slist_append(headers, nintendoAccountId);

        RAPIDJSON_NAMESPACE::StringBuffer postBuffer;
        RAPIDJSON_NAMESPACE::Writer<RAPIDJSON_NAMESPACE::StringBuffer> writer(postBuffer);
        requestBody.Accept(writer);

        std::stringstream responseStream;
        curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headers);
        curl_easy_setopt(curl, CURLOPT_POST, 1);
        curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 1L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 2L);
        curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
        curl_easy_setopt(curl, CURLOPT_WRITEDATA, &responseStream);
        curl_easy_setopt(curl, CURLOPT_POSTFIELDS, postBuffer.GetString());
        curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE, postBuffer.GetSize());
        CURLcode curlCode = curl_easy_perform(curl);

        bool ret = true;
        if(curlCode != CURLE_OK)
        {
            DEVMENUCOMMAND_LOG("Request Failed. curlcode:%d (%s)\n", curlCode, curl_easy_strerror(curlCode));
            ret = false;
        }

        if (responseStream.str().length() > 0)
        {
            const char* contentType;
            curlCode = curl_easy_getinfo(curl, CURLINFO_CONTENT_TYPE, &contentType);
            if (curlCode == CURLE_OK)
            {
                DEVMENUCOMMAND_LOG("Content-Type: %s\n", contentType);
                if (std::string(contentType).find("application/json") != std::string::npos)
                {
                    responseBody->Parse(responseStream.str().c_str());
                }
                else
                {
                    DEVMENUCOMMAND_LOG("Unexpected Response: %s: %s", contentType, responseStream.str().c_str());
                    ret = false;
                }
            }
        }

        curl_easy_cleanup(curl);
        if (headers != nullptr)
        {
            curl_slist_free_all(headers);
        }
        return ret;
    }
}

