﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <vector>
#include <cctype>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_StrToUll.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_BgtcCommand.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

namespace {

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option& option);
    };

    template <typename T>
    bool GetSettingsFwdbgValue(T* pOut, const char* keyString)
    {
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(pOut, sizeof(T), "bgtc", keyString);
        bool bResult = (bytes == sizeof(T));
        if (!bResult)
        {
            NN_LOG("Failed to read settings: %s\n", keyString);
        }
        return bResult;
    }

    template <typename T>
    void SetSettingsFwdbgValue(T value, const char* keyString)
    {
        nn::settings::fwdbg::SetSettingsItemValue("bgtc", keyString, &value, sizeof(T));
    }

    bool ParseInteger(int32_t* pOut, const char* pString)
    {
        char* pEnd;
        *pOut = static_cast<int32_t>(std::strtoll(pString, &pEnd, 10));
        return pEnd != nullptr && *pEnd == '\0';
    }

    bool ParseBool(bool* pOut, const char* pString)
    {
        std::string str(pString);
        if (str == "")
        {
            NN_LOG("You must specify a status. Set one of [enable|disable].\n");
            return false;
        }
        else if (str != "enable" && str != "disable")
        {
            NN_LOG("%s is not valid status. Set one of [enable|disable].\n", pString);
            return false;
        }
        else
        {
            *pOut = str == "enable";
            return true;
        }
    }

    Result ConfigSettingInteger(bool* outValue, const Option& option, const char* keyString)
    {
        auto modeString = option.GetTarget();
        int32_t value;
        if (std::string(modeString).empty())
        {
            if ((*outValue = GetSettingsFwdbgValue<int32_t>(&value, keyString)) == true)
            {
                NN_LOG("%s: %d\n", keyString, value);
            }
        }
        else
        {
            if (ParseInteger(&value, modeString))
            {
                SetSettingsFwdbgValue<int32_t>(value, keyString);
                NN_LOG("%s: %d\n", keyString, value);
                *outValue = true;
            }
            else
            {
                *outValue = false;
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result ConfigSettingBool(bool* outValue, const Option& option, const char* keyString)
    {
        auto modeString = option.GetTarget();
        bool isEnabled;
        if (std::string(modeString).empty())
        {
            if ((*outValue = GetSettingsFwdbgValue<bool>(&isEnabled, keyString)) == true)
            {
                NN_LOG("%s: %s\n", keyString, isEnabled ? "Enabled" : "Disabled");
            }
        }
        else
        {
            if (ParseBool(&isEnabled, modeString))
            {
                SetSettingsFwdbgValue<bool>(isEnabled, keyString);
                NN_LOG("%s: %s\n", keyString, isEnabled ? "Enabled" : "Disabled");
                *outValue = true;
            }
            else
            {
                *outValue = false;
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result ConfigHalfAwake(bool* outValue, const Option& option)
    {
        return ConfigSettingBool(outValue, option, "enable_halfawake");
    }

    Result ConfigBatterySaver(bool* outValue, const Option& option)
    {
        return ConfigSettingBool(outValue, option, "enable_battery_saver");
    }

    Result ConfigMinimumIntervalNormal(bool* outValue, const Option& option)
    {
        return ConfigSettingInteger(outValue, option, "minimum_interval_normal");
    }

    Result ConfigMinimumIntervalSave(bool* outValue, const Option& option)
    {
        return ConfigSettingInteger(outValue, option, "minimum_interval_save");
    }

    Result ConfigLeavingHalfAwakeMargin(bool* outValue, const Option& option)
    {
        return ConfigSettingInteger(outValue, option, "leaving_halfawake_margin");
    }

    const char HelpMessage[] =
        "usage: " DEVMENUCOMMAND_NAME " bgtc config-halfawake [enable|disable]\n"
        "       " DEVMENUCOMMAND_NAME " bgtc config-battery-saver [enable|disable]\n"
        "       " DEVMENUCOMMAND_NAME " bgtc config-minimum-interval-normal [interval]\n"
        "       " DEVMENUCOMMAND_NAME " bgtc config-minimum-interval-save [interval]\n"
        "       " DEVMENUCOMMAND_NAME " bgtc config-leaving-halfawake-margin [margin]\n"
        "       " DEVMENUCOMMAND_NAME " bgtc config-leaving-halfawake-margin [margin]\n"
        ""; // 終端

    const SubCommand g_SubCommands[] =
    {
        {"config-halfawake",                ConfigHalfAwake},
        {"config-battery-saver",            ConfigBatterySaver},
        {"config-minimum-interval-normal",  ConfigMinimumIntervalNormal},
        {"config-minimum-interval-save",    ConfigMinimumIntervalSave},
        {"config-leaving-halfawake-margin", ConfigLeavingHalfAwakeMargin},
    };

}   // namespace

//------------------------------------------------------------------------------------------------

Result BgtcCommand(bool* outValue, const Option& option)
{
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu bgtc command. See '" DEVMENUCOMMAND_NAME " bgtc --help'.\n", option.GetSubCommand());
    *outValue = false;
    NN_RESULT_SUCCESS;
}
