﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>
#include <nn/fs.h>

#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>

#include "DevMenuCommand_Label.h"
#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_Option.h"
#include "DevMenuCommand_AlbumCommand.h"

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
#include "album/DevMenuCommandAlbum_ExecuteUploadAction.h"
#include "album/DevMenuCommandAlbum_ExecutePrintAction.h"
#endif
#include "album/DevMenuCommandAlbum_ExecuteDownloadAction.h"
#include "album/DevMenuCommandAlbum_ExecuteCleanAction.h"
#include "album/DevMenuCommandAlbum_ExecuteInfoAction.h"
#include "album/DevMenuCommandAlbum_ExecuteListAction.h"
#include "album/DevMenuCommandAlbum_ProgramOption.h"

using namespace nn;

//------------------------------------------------------------------------------------------------

#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
#define NN_DMC_FILE_CONTENTS_OPTION " [--extra|--regular]"
#else
#define NN_DMC_FILE_CONTENTS_OPTION ""
#endif

#define NN_DMC_STORAGE_OPTION " --storage <builtin|sdcard>"
#define NN_DMC_DIR_OPTION     " --dir <path>"

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
namespace {

    const char HelpMessage[] =
        "\n"
        "usage>\n"
        "  $ " DEVMENUCOMMAND_NAME " album download" NN_DMC_STORAGE_OPTION NN_DMC_DIR_OPTION " [--force] [--check-empty]\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "  $ " DEVMENUCOMMAND_NAME " album upload" NN_DMC_STORAGE_OPTION NN_DMC_DIR_OPTION " [--force] [--check-empty] [--raw]\n"
#endif
        "  $ " DEVMENUCOMMAND_NAME " album list" NN_DMC_STORAGE_OPTION NN_DMC_FILE_CONTENTS_OPTION "\n"
        "  $ " DEVMENUCOMMAND_NAME " album info\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "  $ " DEVMENUCOMMAND_NAME " album info <file-path>\n"
        "  $ " DEVMENUCOMMAND_NAME " album print-encoded-applicationid <ApplicationId>\n"
        "  $ " DEVMENUCOMMAND_NAME " album print-decoded-applicationid <EncodedApplicationId>" NN_DMC_FILE_CONTENTS_OPTION "\n"
#endif
        "  $ " DEVMENUCOMMAND_NAME " album clean" NN_DMC_STORAGE_OPTION NN_DMC_FILE_CONTENTS_OPTION "\n"
        "  $ " DEVMENUCOMMAND_NAME " album help\n"
        "\n"
        "  options:\n"
        "   --force      : Force copy files even if the same file exists in destination.\n"
        "   --check-empty: Check if the destination is empty before copying.\n"
        "   --nand       : Equivalent to '--storage builtin'.\n"
        "   --sd         : Equivalent to '--storage sdcard'.\n"
        "   --directory  : Alias of '--dir'.\n"
#if defined(NN_TOOL_DEVMENUCOMMANDSYSTEM)
        "   --regular    : Process only regular-album-files.\n"
        "   --extra      : Process only extra-album-files.\n"
#endif
        "\n"
        ""; // 終端

    struct SubCommand
    {
        std::string name;
        Result(*function)(bool* outValue, const Option&);
    };

//------------------------------------------------------------------------------------------------

    Result AlbumDownloadCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_Download);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecuteDownloadAction(programOption);
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result AlbumUploadCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_Upload);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecuteUploadAction(programOption);
        NN_RESULT_SUCCESS;
    }
#endif

    Result AlbumListCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_List);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecuteListAction(programOption);
        NN_RESULT_SUCCESS;
    }

    Result AlbumInfoCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_Info);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecuteInfoAction(programOption);
        NN_RESULT_SUCCESS;
    }

#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
    Result AlbumPrintEncodedApplicationIdCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_PrintEncryptApplicationId);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecutePrintEncryptedApplicationIdAction(programOption);
        NN_RESULT_SUCCESS;
    }

    Result AlbumPrintDecodedApplicationIdCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_PrintEncryptApplicationId);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecutePrintDecryptedApplicationIdAction(programOption);
        NN_RESULT_SUCCESS;
    }
#endif

    Result AlbumCleanCommand(bool* outValue, const Option& option)
    {
        album::ProgramOption programOption(album::Action_Clean);
        if (!programOption.TryParseArgument(option))
        {
            NN_LOG("Parsing options failed.\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        *outValue = album::ExecuteCleanAction(programOption);
        NN_RESULT_SUCCESS;
    }

    Result AlbumHelpCommand(bool* outValue, const Option& option)
    {
        NN_UNUSED(option);

        NN_LOG(HelpMessage);
        *outValue = true;
        NN_RESULT_SUCCESS;
    }

    const SubCommand g_SubCommands[] =
    {
        { "download",           AlbumDownloadCommand        },
#if defined NN_TOOL_DEVMENUCOMMANDSYSTEM
        { "upload",             AlbumUploadCommand          },
        { "print-encoded-applicationid", AlbumPrintEncodedApplicationIdCommand },
        { "print-decoded-applicationid", AlbumPrintDecodedApplicationIdCommand },
#endif
        { "list",               AlbumListCommand            },
        { "info",               AlbumInfoCommand            },
        { "clean",              AlbumCleanCommand           },
        { "help",               AlbumHelpCommand            },
    };

}   // namespace
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

//------------------------------------------------------------------------------------------------

Result AlbumCommand(bool* outValue, const Option& option)
{
#if !defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_UNUSED(option);
    NN_LOG("*** ALBUM command is not supported on this platform.\n");
    *outValue = false;
    NN_RESULT_SUCCESS;
#else
    if (!option.HasSubCommand())
    {
        NN_LOG(HelpMessage);
        *outValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (std::string(option.GetSubCommand()) == "--help")
    {
        return AlbumHelpCommand(outValue, option);
    }

    nn::capsrv::InitializeAlbumAccess();

    // TORIAEZU:
    // ns::InitializeForSystemUpdate() を呼び出しただけでは接続に成功して待たないため、
    // 動作上無意味な API を読んで初期化が終わるまで待つようにする
    ns::GetBackgroundNetworkUpdateState();

    // album サブコマンドの呼出し
    for (const SubCommand& subCommand : g_SubCommands)
    {
        if (subCommand.name == option.GetSubCommand())
        {
            return subCommand.function(outValue, option);
        }
    }

    NN_LOG("'%s' is not a DevMenu album command. See '" DEVMENUCOMMAND_NAME " album --help'.\n", option.GetSubCommand());
    *outValue = false;

    nn::capsrv::FinalizeAlbumAccess();
    NN_RESULT_SUCCESS;
#endif
}
