﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/nn_SdkAssert.h>

#include <acsd_horizon.h>

#include "wlan_BaseFunctions.h"
#include "wlan_MezcalType.h"
#include "wlan_VieManager.h"
#include "wlan_ClientStatusManager.h"
#include "wlan_DebugLog.h"

namespace nn { namespace wlan {

class Mezcal : public nn::wlan::WlanBaseFunctions
{
public:
#ifdef ENABLE_LOCAL_TXFLOW_CTRL
    static const int32_t  TxCountMax = 100;
#endif
    static const int FixedDesenseLevel = 22;
    Mezcal() NN_NOEXCEPT;
    ~Mezcal() NN_NOEXCEPT;

    bool InitializeDriver() NN_NOEXCEPT;
    void FinalizeDriver() NN_NOEXCEPT;
//    void BindStateMachine(StateMachine* stat);

    void InitializeEventmask(uint32_t mode) NN_NOEXCEPT;

    void SetCommonParams() NN_NOEXCEPT;
    void SetInfraParams() NN_NOEXCEPT;
    void SetLocalParams(uint32_t mode) NN_NOEXCEPT;
    void SetLocalLcsParams(uint32_t mode) NN_NOEXCEPT;
    void SetDetectParams() NN_NOEXCEPT;
    void ResetParams() NN_NOEXCEPT;
    bool ChangeWirelessMode(uint32_t mode) NN_NOEXCEPT;

    bool SetIfUpDown(uint32_t up_down) NN_NOEXCEPT;
    bool GetIfUpDown(uint32_t* up_down) NN_NOEXCEPT;
    bool SetEventMask(uint8_t mask) NN_NOEXCEPT;
    bool ClearEventMask() NN_NOEXCEPT;
    bool GetEventMask(uint8_t* pmask) NN_NOEXCEPT;
    bool SetApConfigration(WlanLocalBssConfiguration* pbssinfo) NN_NOEXCEPT;
    bool CreateAp(WlanSsidInfo* pSsidInfo) NN_NOEXCEPT;
    bool DestroyAp() NN_NOEXCEPT;
    bool SetStaticAesMode(bool mode) NN_NOEXCEPT;
    bool SetStaticAesKey(WlanStaticAesInfo* pKeyInfo) NN_NOEXCEPT;
    bool SetClientTimeoutPeriod(uint32_t time) NN_NOEXCEPT;
    bool SetBeaconLostTimeout(int time) NN_NOEXCEPT;
    bool SetTxChain(int chain) NN_NOEXCEPT;
    bool SetRxChain(int chain) NN_NOEXCEPT;
    bool SetMaxAssociationNumber(int num) NN_NOEXCEPT;

    bool ScanRequest(WlanScanParameters* scan_params) NN_NOEXCEPT;
    bool CancelScan() NN_NOEXCEPT;

    bool JoinNetworkSta(WlanConnectinoParameters* pConnectParam) NN_NOEXCEPT;
    bool JoinNetworkSpectator(WlanConnectinoParameters* pConnectParam) NN_NOEXCEPT;
    bool Disassociate() NN_NOEXCEPT;
    nn::Result Deauthenticate(WlanDisconnectInfo* deauthParam) NN_NOEXCEPT;

    bool SetSimultaneousTx(SimultaneousTxParam param) NN_NOEXCEPT;
    bool SetMacAddress(const WlanMacAddressData* mac) NN_NOEXCEPT;
    bool GetMacAddress(uint8_t macAddress[MacAddress::MacAddressSize]) NN_NOEXCEPT;
    bool AddIe(uint32_t* pOutIndex, WlanIeContainer* pInIe) NN_NOEXCEPT;
    bool DeleteIe(uint32_t ieIndex) NN_NOEXCEPT;
    bool ExecuteAcsd() NN_NOEXCEPT;
    void SetAcsdFlag(bool var) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_GeneralMutex);
        m_bExecutingAcsd = var;
        nn::os::UnlockMutex(&m_GeneralMutex);
    }
    static bool GetAcsdFlag() NN_NOEXCEPT
    {
        bool ret;
        nn::os::LockMutex(&m_GeneralMutex);
        ret = m_bExecutingAcsd;
        nn::os::UnlockMutex(&m_GeneralMutex);
        return ret;
    }

    bool GetChannel(int16_t* pOutChannel) NN_NOEXCEPT;
    bool GetAllowedChannels(WlanAllowedChannels* allowedChannel) NN_NOEXCEPT;

    void InitializeConnectionStatus() NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_ConnectionStatusMutex);

        std::memset(&m_MezcalConStat, 0, sizeof(ConnectionStatus));
        m_MezcalConStat.statusReasonCode = Dot11StatusCode_Sized16;

        nn::os::UnlockMutex(&m_ConnectionStatusMutex);
    }

    void GetConnectionStatus(ConnectionStatus* pOutStatus) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutStatus);

        nn::os::LockMutex(&m_ConnectionStatusMutex);

        *pOutStatus = m_MezcalConStat;

        nn::os::UnlockMutex(&m_ConnectionStatusMutex);
    }

    // ClientStatusManagerの持っている情報を更新する。
    // ClientStatusは相手からの接続・切断以外に上位層からの切断要求でも更新される必要があるので、ドライバーのCallbackからでもStateMachineからでも更新出来るようにしておく。
    bool UpdateClientStatus(const ClientStatus& status) NN_NOEXCEPT;
    void GetClientStatusList(ClientStatus* pOutList, uint8_t count, Bit32* pOutBitmap, bool IsClear = true) NN_NOEXCEPT;
    bool GetClientStatusByMac(ClientStatus* pOutStatus, const MacAddress& mac) NN_NOEXCEPT
    {
        return m_ClientStatusManager.GetClientStatusByMac(pOutStatus, mac);
    }

    // データ送信
    nn::Result PutFrame(nn::mbuf::Mbuf *pMbuf) NN_NOEXCEPT;

    // ActionFrame一発送信
    nn::Result PutActionFrameOneShot(const char* pData, uint16_t size, uint8_t dstMac[MacAddress::MacAddressSize],
            uint8_t bssid[MacAddress::MacAddressSize], uint32_t channel, uint32_t dwellTime) NN_NOEXCEPT;

    // ActionFrame定期送信
    bool PutActionFramePeriodically(const char* pData, uint16_t size, uint8_t bssid[MacAddress::MacAddressSize],
            uint32_t count, uint32_t interval) NN_NOEXCEPT;

    // 自身のBSSのビーコンインターバルを取得
    bool GetBeaconInterval(uint32_t* pOutInterval) NN_NOEXCEPT;

    // 定期送信ActionFrameの送信を止める
    bool CancelPutActionFramePeriodically() NN_NOEXCEPT;

    // Get RSSI
    bool GetRssiForSta(int32_t* rssi) NN_NOEXCEPT;
    bool GetRssiForAp(int32_t* rssi, const MacAddress& mac) NN_NOEXCEPT;  // RSSI of specified client

    // Enable/Disable TSF time event
    bool SetTsfTimerEventmask(bool enable) NN_NOEXCEPT;
    // Get TSF timer value
    bool GetDeltaTimeBetweenSystemAndTsf(int64_t* pOutDeltaTime) NN_NOEXCEPT;
    bool GetFwVersion(WlanIoctlResult* pOutResult) NN_NOEXCEPT;

    bool IsMatchAssocList(uint8_t macAddr[MacAddress::MacAddressSize]) NN_NOEXCEPT;

    bool EnableCoexMc2() NN_NOEXCEPT;
    bool DisableCoexMc2() NN_NOEXCEPT;
    bool GetWakeupReasonRaw(uint32_t* pOutReason, WowlWakeCount* pCounts) NN_NOEXCEPT;
    WowlWakeReason ConvertWakeupReason(uint32_t reason) NN_NOEXCEPT;
    bool ClearWakeupReason() NN_NOEXCEPT;
    bool SetupWowlParams(WlanWowlSetupParams* pParams) NN_NOEXCEPT;
    bool ClearWowlParams() NN_NOEXCEPT;
    bool ActivateWowl() NN_NOEXCEPT;
    bool DisableWowl() NN_NOEXCEPT;
    bool SetDevicePowerState(bool isFullAwake) NN_NOEXCEPT;
    void SetWakeupReasonRaw(uint32_t reason) NN_NOEXCEPT;
    void EnableWowlFeatures(uint32_t features) NN_NOEXCEPT;
    void GetWowlSleepStats(WowlSleepStats* pStats) NN_NOEXCEPT;
    bool SetCwParam(AccessCategoryIndex accessCategory, int cwmin, int cwmax) NN_NOEXCEPT;
    void SetAciMitigation(AciMitigation id) NN_NOEXCEPT;
    bool GetChannelStats(ChannelStats* pStats, size_t length, uint32_t* pCount) NN_NOEXCEPT;
    void SetDetectHomeChannel(uint32_t channel) NN_NOEXCEPT
    {
        if( channel >= WirelessChannel_1ch && channel <= WirelessChannel_165ch )
        {
            m_DetectHomeCh = channel;
        }
    }
    uint32_t GetDetectHomeChannel() NN_NOEXCEPT
    {
        return m_DetectHomeCh;
    }

    bool PutActionFramePeriodicallyForDetect(const char* pData, uint16_t size, DetectPeriodicAfCycle pattern) NN_NOEXCEPT;
    bool SetupDetectSleepParams(WlanDetectSetupParams* pParams) NN_NOEXCEPT;
    bool ClearDetectSleepParams() NN_NOEXCEPT;
    bool ActivateDetectSleep() NN_NOEXCEPT;
    bool DisableDetectSleep() NN_NOEXCEPT;
    void PullActionFrameFromDongle() NN_NOEXCEPT;
    bool GetWakeupReasonRawDetectSleep(uint32_t* pOutReason, DetectWakeCount* pCounts) NN_NOEXCEPT;
    DetectWakeReason ConvertWakeupReasonDetectSleep(uint32_t reason) NN_NOEXCEPT;
    void ClearAfCache() NN_NOEXCEPT;

    void SetActionFrameRecvMode(ActionFrameRecvMode mode) NN_NOEXCEPT;
    uint64_t GetDetectSaTotalRecvCnt() NN_NOEXCEPT
    {
        return m_DetectSaTotalRecvCnt;
    }
    void ClearDetectSaTotalRecvCnt() NN_NOEXCEPT
    {
        m_DetectSaTotalRecvCnt = 0;
    }

    int GetConnectedClientCount() NN_NOEXCEPT
    {
        return m_ClientStatusManager.GetConnectedClientCount();
    }

    /*
     * ドライバーからイベントが送られる際に呼ばれる関数
     *
     * ここでイベントデータをイベントキューにキューイングする。イベントの中味は見ない。
     * Action frameもイベントに格納されて送られてくる。
     */
    static int EventCallBackFromDriverInfraMode(void* event, void* buf) NN_NOEXCEPT;
    static int EventCallBackFromDriverLocalMode(void* event, void* buf) NN_NOEXCEPT;
    static int EventCallBackFromDriverDetectMode(void* event, void* buf) NN_NOEXCEPT;

    /*
     * ドライバーからデータ受信時に呼ばれる関数
     */
    static int RxCallBackFromDriverLocalMode(void *buf) NN_NOEXCEPT;
    static int RxCallBackFromDriverInfraMode(void *buf) NN_NOEXCEPT;
    static int RxCallBackFromDriverDetectMode(void *buf) NN_NOEXCEPT;

    /*
     * ドライバーからデータ送信完了時に呼ばれる関数
     *
     * ここで送信データの解放を行なう。
     * 引数のresultには送信が成功したか否かの結果が格納されてくる。
     */
    static int TxCompletionCallBackFromDriver(void *buf, int result) NN_NOEXCEPT;


    bool SetStaKeepAlive(int intervalms) NN_NOEXCEPT;
    void DisableFixedDesense() NN_NOEXCEPT;
    void SetMulticastList(WlanAllowedMulticastList* pList) NN_NOEXCEPT;

private:
//    bool ParseWlanNetworkInfo();
    void SetRandomMacAddress() NN_NOEXCEPT;
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    void SetUniqueMacAddress() NN_NOEXCEPT;
    void SetCountryCode() NN_NOEXCEPT;
#endif
    void DumpAcsdResults(wl_chanim_stats_t* pStats, chan_score_t* pScores) NN_NOEXCEPT;

    // Coex desense levelのセット
    void SetBtcDesenseLevel(int level) NN_NOEXCEPT;

public:
    // TORIAEZU
    static ConnectionStatus    m_MezcalConStat;
    static ClientStatusManager m_ClientStatusManager;
    static nn::os::MutexType   m_ConnectionStatusMutex;
    static bool                m_bExecutingAcsd;
    static struct ether_addr   m_OwnMacAddr;
    static nn::os::MutexType   m_GeneralMutex;
    static nn::os::MutexType   m_TsfMutex;
    static WlanTsfTimerValue   m_TsfTimerValue;
    static int64_t             m_DeltaTimeBetweenTsfAndSys;
    static nn::os::MutexType   m_TxCountMutex;
    static int32_t             m_TxCount;
    static bool                m_bWaitIfUpEvent;
    static uint32_t            m_dongleTime;     // すれちがいスリープ直前の無線チップtick値
    static nn::os::Tick        m_sysTick;        // すれちがいスリープ直前のシステムtick値

private:
    void*             m_InterfaceHandle;
    VieManager        m_vieManager;       // IEの追加・削除管理を行うオブジェクト
    bool              m_bDcsEnabled;
    uint8_t           m_DefaultEventmask[WL_EVENTING_MASK_LEN];  // 初期設定のイベントマスクを確保しておくための変数
    static uint32_t   m_wakeReasonForSet;  // 起床要因のセット時に利用される
    static uint32_t   m_wowlFeatures;
#ifdef ENABLE_WOWL_AGING_DEBUG
    static uint64_t   m_ackNum;
#endif
    /* 実装形態は要検討(とりあえず実装) */
//    static StateMachine* m_smInstance;
    bool              m_IsLcsMode;  // LCSモード中か否か
    uint32_t          m_DetectHomeCh;   // すれちがい通信時のホームチャンネル

    /* すれちがい用ActionFrame受信モード設定 */
    // 非すれちがい通信用
    const bwl_aloe_cfg_t m_detectDisableMode = {
            0,  // major
            0,  // minor
            0   // cmd
    };
    // すれちがい通信HD用
    const bwl_aloe_cfg_t m_detectHdMode = {
            DhpMajor,  // major
            DhpMiner,  // minor
            static_cast<uint8_t>(DhpCmdPeriodic | DhpCmdOneShot)   // cmd
    };
    // すれちがい通信SA用
    const bwl_aloe_cfg_t m_detectSaMode = {
            DhpMajor,  // major
            DhpMiner,  // minor
            DhpCmdPeriodic   // cmd
    };
    uint64_t         m_DetectSaTotalRecvCnt;  // すれちがいスリープ状態のときに受信したフレームの総数

    // Multicast filter用
    struct bwl_multi_filter_list {
        uint count;
        struct ether_addr ea[MulticastFilterCountMax];
    };
};

void BindStateMachine(nn::wlan::StateMachine* statm) NN_NOEXCEPT;
}}

extern "C" int wlu_set(void *wl, int cmd, void *cmdbuf, int len);  // wlコマンドを直接叩くのに使用
extern "C" int wlu_get(void *wl, int cmd, void *cmdbuf, int len);  // wlコマンドを直接叩くのに使用
extern "C" int wlu_iovar_get(void *wl, const char *iovar, void *outbuf, int len);
extern "C" int wlu_iovar_set(void *wl, const char *iovar, void *param, int paramlen);
