﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include "wlan_ClientStatusManager.h"

namespace nn { namespace wlan {

ClientStatusManager::ClientStatusManager() NN_NOEXCEPT
{
    nn::os::InitializeMutex(&m_Mutex, false, 0);
    Initialize();
}

ClientStatusManager::~ClientStatusManager() NN_NOEXCEPT
{
    nn::os::FinalizeMutex(&m_Mutex);
}

void ClientStatusManager::Initialize() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    m_UpdatedClientStatusBitMap = 0;
    for(uint8_t i = 0; i < ConnectableClientsCountMax; i++)
    {
        std::memset(&m_Status[i], 0x00, sizeof(ClientStatus));
        m_Status[i].state = ConnectionState_Disconnected;
        m_Status[i].clientMacAddress = MacAddress::CreateZeroMacAddress();
    }

    m_ConnectedClientCount = 0;

    nn::os::UnlockMutex(&m_Mutex);
}

bool ClientStatusManager::UpdateClientStatus(const ClientStatus& status) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    bool ret = false;

    // bc deauth による disconnect
    if( status.state == ConnectionState_Disconnected
            && status.clientMacAddress == MacAddress::CreateBroadcastMacAddress() )
    {
        for (uint8_t i = 0; i < ConnectableClientsCountMax; i++)
        {
            if( m_Status[i].state == ConnectionState_Connected )
            {
                m_Status[i].state = ConnectionState_Disconnected;
                m_Status[i].cause = status.cause;
                // clientMacAddress はそのまま
                m_Status[i].statusReasonCode = status.statusReasonCode;
                m_Status[i].capabilityInfo = status.capabilityInfo;
                m_Status[i].rssi = -128;
                m_Status[i].updateTick = nn::os::GetSystemTick();

                // ビットマップ更新
                EditUpdatedClientStatusBitMap(static_cast<uint32_t>(i));
            }
        }
        WLAN_LOG_DEBUG("[ClientManager]Update all status due to Master deauth\n");
        // 全台切断
        m_ConnectedClientCount = 0;
        WLAN_LOG_DEBUG("[ClientManager]Delete all clients (%d)\n", m_ConnectedClientCount);
        ret = true;
    }
    else
    {
        uint8_t emptyIndex = 0xFF;
        // index チェック
        for( uint8_t i = 0; i < ConnectableClientsCountMax; i++ )
        {
            if( m_Status[i].clientMacAddress == status.clientMacAddress )
            {
                // 古い Status をクリア
                std::memset(&m_Status[i], 0, sizeof(ClientStatus));
                // 更新
                m_Status[i] = status;
                m_Status[i].updateTick = nn::os::GetSystemTick();
                // ビットマップ更新
                EditUpdatedClientStatusBitMap(static_cast<uint32_t>(i));

                WLAN_LOG_DEBUG("[ClientManager]Update status\n");

                if( status.state == ConnectionState_Connected )
                {
                    // 1台追加
                    m_ConnectedClientCount++;
                    WLAN_LOG_DEBUG("[ClientManager]Add the client (%d)\n", m_ConnectedClientCount);
                }
                else if( ConnectionState_Disconnected )
                {
                    // 1台削除
                    m_ConnectedClientCount--;
                    WLAN_LOG_DEBUG("[ClientManager]Delete the client (%d)\n", m_ConnectedClientCount);
                }

                ret = true;
                break;
            }
            else
            {
                // 空き番を探しておく
                if( m_Status[i].state == ConnectionState_Disconnected && emptyIndex == 0xFF)
                {
                    // 空き番なので新規追加Clientであればここに追加する
                    emptyIndex = i;
                }
            }
        }
        if( ret == false && emptyIndex != 0xFF && status.state != ConnectionState_Disconnected )
        {
            // 新規追加Clientで、格納可能なので格納しておく
            m_Status[emptyIndex] = status;
            m_Status[emptyIndex].updateTick = nn::os::GetSystemTick();

            // ビットマップ更新
            EditUpdatedClientStatusBitMap(static_cast<uint32_t>(emptyIndex));

            WLAN_LOG_DEBUG("[ClientManager]Add new status\n");
            // 1台追加
            m_ConnectedClientCount++;
            WLAN_LOG_DEBUG("[ClientManager]Add new client (%d)\n", m_ConnectedClientCount);
            ret = true;
        }
    }

    nn::os::UnlockMutex(&m_Mutex);

    return ret;
}

void ClientStatusManager::GetClientStatusList(ClientStatus* pOutList, uint8_t count, Bit32* pOutBitmap, bool IsClear) NN_NOEXCEPT
{
    nn::os::LockMutex(&m_Mutex);

    NN_SDK_ASSERT_NOT_NULL(pOutList);
    NN_SDK_ASSERT_EQUAL(count, ConnectableClientsCountMax);

    for( uint8_t i = 0; i < ConnectableClientsCountMax; i++ )
    {
        pOutList[i] = m_Status[i];
    }

    *pOutBitmap = m_UpdatedClientStatusBitMap;

    if( IsClear == true )
    {
        m_UpdatedClientStatusBitMap = 0; // クリア
    }

    nn::os::UnlockMutex(&m_Mutex);
}

bool ClientStatusManager::GetClientStatusByMac(ClientStatus* pOutStatus, const MacAddress& mac) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutStatus);

    nn::os::LockMutex(&m_Mutex);

    bool ret = false;
    for( int i = 0; i < ConnectableClientsCountMax; i++ )
    {
        if( m_Status[i].clientMacAddress == mac )
        {
            std::memcpy(pOutStatus, &m_Status[i], sizeof(ClientStatus));
            ret = true;
            break;
        }
    }
    nn::os::UnlockMutex(&m_Mutex);

    return ret;
}

const uint32_t ClientStatusManager::GetClientCount() NN_NOEXCEPT
{
    uint32_t bits = m_UpdatedClientStatusBitMap;

    bits = (bits & 0x55555555) + (bits >> 1 & 0x55555555);
    bits = (bits & 0x33333333) + (bits >> 2 & 0x33333333);
    bits = (bits & 0x0f0f0f0f) + (bits >> 4 & 0x0f0f0f0f);
    bits = (bits & 0x00ff00ff) + (bits >> 8 & 0x00ff00ff);
    return (bits & 0x0000ffff) + (bits >>16 & 0x0000ffff);
}

bool ClientStatusManager::VerifyClientList(const WlanMacAddressData* list, uint32_t count) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(list);

    uint8_t lhs = 0;
    uint8_t rhs = 0;

    for( uint8_t i = 0; i < ConnectableClientsCountMax; i++ )
    {
        // Connectedのものだけ比較する
        if( m_Status[i].state == ConnectionState_Connected )
        {
            lhs++;
            for( uint32_t j = 0; j < count; j++ )
            {
                if( std::memcmp( &m_Status[i].clientMacAddress.GetMacAddressData()[0],
                                             &list[j].addr[0], MacAddress::MacAddressSize) == 0)
                {
                    WLAN_LOG_INFO("[ClientStatusManager]Corresponding status exist\n");
                    rhs++;
                    break;
                }
            }
        }
    }

    return ((lhs == rhs)? true: false);
}

}}

