﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
//#include <netinet/ether.h>
#include "wlan_Type.h"
#include "wlan_TestTxRx.h"
#include <dhd_horizon.h>
#include <bwl.h>
#include <wlioctl.h>

#include <proto/bcmevent.h>
#include <proto/802.11.h>
#include <proto/ethernet.h>
#include <bcmutils.h>
//#include <wlu_utils.h>

#include <cstdio>
#include <cstdint>
#include <cstring>
#include <cstdbool>
#include <cwctype>
#include <acsd_horizon.h>
#include <horizon_main.h>
#include <nnc/os/os_MutexApi.h>
#include <nnc/wlan/driver/wlan_DriverTypes.h>

#if 0
/* for bwl api connection type */
#define STA_BWL_MODE
#else
/* for ACSD and wpa supplicant testing */
#undef STA_BWL_MODE
#define WPA_SUPPLICANT_SCAN
#define WPA_SUPPLICANT_ADV_SCAN
#define WPA_SUPPLICANT_CONNECT
#define CONFIG_ACSD_SUPPORT
#define DCS_TEST
#define CONFIG_ACSD_CHAN_STATISTICS_DUMP
#endif
extern nnosMutexType acsd_bcm_mutex;
#define ACSD_LOCK() nnosLockMutex(&acsd_bcm_mutex)
#define ACSD_UNLOCK() nnosUnlockMutex(&acsd_bcm_mutex)
#define ETH_P_PAE 0x888E    //NOLINT(preprocessor/const)

namespace
{
    void * ifh;
    void * wpa_s = nullptr;
#if 1
    static uint8_t TEST_SSID[] = "TEST_SSID";
    static int      SIZEOF_TEST_SSID = 9;
#else
    static uint8_t TEST_SSID[] = "rctest";
    static int      SIZEOF_TEST_SSID = 6;
#endif

    nn::os::EventType event;
    nn::os::EventType recvdEvent;

    static const char *bcmerrorstrtable[] = BCMERRSTRINGTABLE;
    nn::mbuf::Mbuf *g_Mbuf;
    nn::mbuf::Mbuf *pRecvedMbuf;

    static uint8_t BROADCAST_ADDR[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    uint8_t g_senderAddr[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    uint8_t g_macAddr[6] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

    uint32_t g_BRCMenablelocal = 0;     /* 1 - enable local broadcast */
    uint32_t g_BRCMenableDatalog = 0;    /* 1 - enable tx complete dump
                                         * 2 - enable rx data dump
                                         * 3 - enable both
                                         */
    uint32_t g_BRCMenableTxpkt = 0;        /* 1 - enable tx sending tx unicast pkt
                         * 2 - enable tx broadcast pkt
                         * 3 - enables both unicast & broadcast
                         */
    uint32_t g_BRCMenableFlag = 0;        /* enable different configurations
                         * 1 - enable BCNRX_MSG
                         * 2 - enable TSF_MSG
                         */
    uint32_t g_BRCMKeyMgmt = 0;     /* 0 - NONE
                                     * 1 - OPEN - WEP
                                     * 2 - SHARE_WEP
                                     * 3 - WPAPSK TKIP
                                     * 4 - WPAPSK CCMP
                                     * 5 - WPA2PSK CCMP
                                     * 6 - WPA2PSK TKIP
                                     * 7 - WPS PBC
                                     * 8 - WPS PIN
                                     */
    static int reply_size = 2048;
    uint8_t reply[4096];
    uint32_t APMode = 0; /* 1 = AP */
}
/* g_BRCMenableFlag defines */
#define BCNRX_MSG_enabled        1    //NOLINT(preprocessor/const)
#define TSF_MSG_enabled            2    //NOLINT(preprocessor/const)

#define TEST_ENTER()    NN_LOG(">>>> Enter %s <<<<\n", __FUNCTION__)
#define TEST_LEAVE()    NN_LOG("    <<<< Leave %s <<<<\n", __FUNCTION__)

namespace nn {
namespace wlan {

void TestTxRx::Initialize()
{
TEST_ENTER();
nn::os::InitializeEvent( &event, false, nn::os::EventClearMode_AutoClear );
nn::os::InitializeEvent( &recvdEvent, false, nn::os::EventClearMode_AutoClear );
    bwl_dhd_initialize(&ifh);
TEST_LEAVE();
}

void TestTxRx::Deinitialize()
{
TEST_ENTER();
    if (wpa_s != nullptr) {
        wpa_supplicant_exit(wpa_s);
    }
    bwl_dhd_deinitialize(ifh);
    nn::os::FinalizeEvent( &event );
    nn::os::FinalizeEvent( &recvdEvent );
TEST_LEAVE();
}

void TestTxRx::BinaryDump(void* data, int len, int type)
{
    int i, j;

    if(!data)
    {
        return;
    }

    char* pData = static_cast<char*>(data);

    NN_LOG("\n%s Len[%d]----------\n", __FUNCTION__, len);

    for (i = 0; i < len; i++)
    {
        char charprint[16];
        NN_LOG(" %02x", pData[i]);
        if (type == 1 && i % 16 == 7)
        {
            NN_LOG(" ");
        }
        charprint[i % 16] = pData[i];
        if (i % 16 == 15)
        {
            NN_LOG(" ");
            for (j = 0; j < sizeof(charprint); j++)
            {
                if (j % 8 == 0)
                {
                    NN_LOG(" ");
                }
                if (type == 1)
                {
                    if (iswprint(charprint[j]))
                    {
                        NN_LOG("%c", charprint[j]);
                    }
                    else
                    {
                        NN_LOG(".");
                    }
                }
            }
            NN_LOG("\n");
        }
    }
    NN_LOG("\n----------\n");
}

void TestTxRx::ToIdle()
{
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
}

void TestTxRx::ToClass1()
{
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
}

void TestTxRx::ToApClass3()
{
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_set_inframode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_channel(ifh, 11));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
    PRINT_ERROR_IF_FAILED(bwl_create(ifh, TEST_SSID, SIZEOF_TEST_SSID));
}

void TestTxRx::WaitEvent()
{
    nn::os::WaitEvent( &event );
}

void TestTxRx::SignalEvent()
{
    NN_LOG("<WLAN_TEST> [%s(%d) ]\n",__FUNCTION__, __LINE__);
    nn::os::SignalEvent(&event);
}

void TestTxRx::WaitReceivedEvent()
{
    nn::os::WaitEvent( &recvdEvent );
}

void TestTxRx::SignalReceivedEvent()
{
    if (g_BRCMenableDatalog & 2)
    NN_LOG("<WLAN_TEST> [%s(%d) ]\n",__FUNCTION__, __LINE__);
    nn::os::SignalEvent(&recvdEvent);
}


void TestTxRx::GenerateEtherPacket(uint8_t* buf, int size)
{
    /*
    BWL_PRE_PACKED_STRUCT struct ether_header {
        uint8    ether_dhost[ETHER_ADDR_LEN];
        uint8    ether_shost[ETHER_ADDR_LEN];
        uint16    ether_type;
    } BWL_POST_PACKED_STRUCT;
    */

    struct ether_header etherHeader = {0};
    struct ether_addr broadcastAddr = {{255, 255, 255, 255, 255, 255}};

    std::memcpy(etherHeader.ether_dhost, &broadcastAddr, 6);
    etherHeader.ether_type = 0x0800;

    std::memcpy(buf, &etherHeader, sizeof(ether_header));

    for(int i = sizeof(ether_header); i < size; i++)
    {
        buf[i] = i & 0xff ;
    }
}

void TestTxRx::ApTestMode()
{
    int i = 0;
    TEST_ENTER();
    RegisterCallback();
    PRINT_ERROR_IF_FAILED(bwl_get_curetheraddr(ifh, (struct ether_addr *)&g_macAddr[0]));
    NN_LOG("%s: Test config local %d datalog %d txpkt %d\n",
            __FUNCTION__, g_BRCMenablelocal, g_BRCMenableDatalog, g_BRCMenableTxpkt );
    NN_LOG("got mac %02x:%02x:%02x-%02x:%02x:%02x\n", g_macAddr[0], g_macAddr[1],
            g_macAddr[2], g_macAddr[3], g_macAddr[4], g_macAddr[5]);
    APMode = 1;
#ifdef CONFIG_ACSD_SUPPORT
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_set_band(ifh, WLC_BAND_2G));
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_wsec(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_mpc(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_pm(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_vhtmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
    PRINT_ERROR_IF_FAILED(acsd_main());
    PRINT_ERROR_IF_FAILED(bwl_create(ifh, TEST_SSID, SIZEOF_TEST_SSID));
    PRINT_ERROR_IF_FAILED(acsd_dcs());
#else
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_set_inframode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_channel(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
    PRINT_ERROR_IF_FAILED(bwl_create(ifh, TEST_SSID, SIZEOF_TEST_SSID));
#endif /* CONFIG_ACSD_SUPPORT */
    if (g_BRCMenablelocal)
        PRINT_ERROR_IF_FAILED(bwl_set_ds3mode(ifh, 2));

    TEST_LEAVE();
}


void TestTxRx::ApTest()
{
    int i = 0;
    TEST_ENTER();
    RegisterCallback();
    PRINT_ERROR_IF_FAILED(bwl_get_curetheraddr(ifh, (struct ether_addr *)&g_macAddr[0]));
    NN_LOG("%s: Test config local %d datalog %d txpkt %d\n",
            __FUNCTION__, g_BRCMenablelocal, g_BRCMenableDatalog, g_BRCMenableTxpkt );
    NN_LOG("got mac %02x:%02x:%02x-%02x:%02x:%02x\n", g_macAddr[0], g_macAddr[1],
            g_macAddr[2], g_macAddr[3], g_macAddr[4], g_macAddr[5]);
    APMode = 1;
#ifdef CONFIG_ACSD_SUPPORT
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_set_band(ifh, WLC_BAND_2G));
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_wsec(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_mpc(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_pm(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_set_vhtmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
    PRINT_ERROR_IF_FAILED(acsd_main());
    PRINT_ERROR_IF_FAILED(bwl_create(ifh, TEST_SSID, SIZEOF_TEST_SSID));
#ifdef CONFIG_ACSD_CHAN_STATISTICS_DUMP
    {
    uchar data1[512];
    uchar data2[128];
        wl_chanim_stats_t* chanim_stats;
        chanim_stats_t *stats;
        chan_score_t *chan_score;
        int j = 0;
    void *p_data_1, *p_data_2;
        p_data_1 = (void *)data1;
        p_data_2 = (void *)data2;

        PRINT_ERROR_IF_FAILED(acsd_get_chanim(&g_macAddr[0], p_data_1, p_data_2));
        chanim_stats =  (wl_chanim_stats_t*)p_data_1;

        /* dump the channel statistics */
        NN_LOG("chanspec tx   inbss   obss   nocat   nopkt   doze     txop     "
               "goodtx  badtx   glitch   badplcp  knoise chanIdle  timestamp\n");
        for (i = 0; i < chanim_stats->count; i++) {
            stats = &chanim_stats->stats[i];
            NN_LOG("0x%4x\t", stats->chanspec);
            for (j = 0; j < CCASTATS_MAX; j++)
                NN_LOG("%d\t", stats->ccastats[j]);
            NN_LOG("%d\t%d\t%d\t%d\t%d", (stats->glitchcnt), (stats->badplcp),
                stats->bgnoise, (stats->chan_idle), (stats->timestamp));
            NN_LOG("\n");
        }

        /* dump the score of channel selection */
        chan_score = (chan_score_t *)p_data_2;
        for (i = 0; i < chan_score->count; i++) {
            NN_LOG("Chan:%#x, bss number:%d, intf score:%d, bg noise:%d\n",
            chan_score->chan_score_info[i].chan, chan_score->chan_score_info[i].bss_score,
            chan_score->chan_score_info[i].intf_score, chan_score->chan_score_info[i].bg_noise_avg);
        }
    }
#endif

#else
    PRINT_ERROR_IF_FAILED(bwl_down(ifh));
    PRINT_ERROR_IF_FAILED(bwl_set_inframode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 1));
    PRINT_ERROR_IF_FAILED(bwl_set_channel(ifh, 11));
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));
    PRINT_ERROR_IF_FAILED(bwl_create(ifh, TEST_SSID, SIZEOF_TEST_SSID));
#endif /* CONFIG_ACSD_SUPPORT */
    if (g_BRCMenablelocal)
        PRINT_ERROR_IF_FAILED(bwl_set_ds3mode(ifh, 2));

#ifdef DCS_TEST
    WaitEvent();
    WaitReceivedEvent();
    NN_LOG("Start DCS with mac %02x:%02x:%02x-%02x:%02x:%02x\n", g_senderAddr[0], g_senderAddr[1],
            g_senderAddr[2], g_senderAddr[3], g_senderAddr[4], g_senderAddr[5]);
    for(;;)
    {
        for (i = 1; i< 3; i++)
            SendData(g_senderAddr);
        NN_LOG("Send Packet...\n");
        PRINT_ERROR_IF_FAILED(acsd_dcs());
    }
#else
    WaitEvent();
    for(;;)
    {
        WaitReceivedEvent();
        if (g_BRCMenableTxpkt & 1)
            SendData(g_senderAddr);
        if (g_BRCMenableTxpkt & 2)
            SendData(BROADCAST_ADDR);
    }
#endif
    TEST_LEAVE();
}


void TestTxRx::SendData(uint8_t *dst)
{
    int len = 1024;
    uint8_t data[len];
    uint8_t ethtype[4] = {0x08, 0x00, 0x45, 0x00};
    void * p_data;
    uint8_t *pbuf;
    uint ret;

    g_Mbuf = nn::mbuf::MbufGetm(NULL, len, nn::mbuf::MbufAllocationMode_DontWait,
            nnwlandriverMbufType_TxData);
    p_data = nn::mbuf::MbufTod(g_Mbuf);
    pbuf = (uint8_t*)p_data;
    GenerateEtherPacket(data, len);
    std::memcpy(p_data, data, len);
    nn::mbuf::MbufExpand(g_Mbuf, len);

    std::memcpy(&pbuf[0], dst, 6);
    std::memcpy(&pbuf[6], g_macAddr, 6);
    std::memcpy(&pbuf[12], ethtype, 4);
    //BinaryDump(p_data, len, 1);

    ret = bwl_txpkt(ifh, g_Mbuf);
    // if the tx operation fail then need to free the mbuf here
    if (ret != BCME_OK)
        nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf *>(g_Mbuf));
}

void TestTxRx::StaTest()
{
    TEST_ENTER();
    APMode = 0;


#ifdef STA_BWL_MODE
    wl_join_params_t bcm_join;
    RegisterCallback();
    NN_LOG("%s: Test config local %d datalog %d txpkt %d\n",
            __FUNCTION__, g_BRCMenablelocal, g_BRCMenableDatalog, g_BRCMenableTxpkt );
    bcm_join.ssid.SSID_len = SIZEOF_TEST_SSID;
    memcpy(&bcm_join.ssid.SSID, &TEST_SSID, SIZEOF_TEST_SSID);
    // bwl_set_apmode -> STA
    PRINT_ERROR_IF_FAILED(bwl_set_apmode(ifh, 0));
    PRINT_ERROR_IF_FAILED(bwl_get_curetheraddr(ifh, (struct ether_addr *)&g_macAddr[0]));
    NN_LOG("got mac %02x:%02x:%02x-%02x:%02x:%02x\n", g_macAddr[0], g_macAddr[1],
            g_macAddr[2], g_macAddr[3], g_macAddr[4], g_macAddr[5]);

        // bwl_get_apmode
        {
            int apmode;
            PRINT_ERROR_IF_FAILED(bwl_get_apmode(ifh, &apmode));
            if(apmode != 0)
            {
                PRINT_ERROR();
            }
        }

    // bwl_set_inframode -> INFRA
    PRINT_ERROR_IF_FAILED(bwl_set_inframode(ifh, 1));

        // bwl_get_inframode
        {
            uint32_t inframode;
            PRINT_ERROR_IF_FAILED(bwl_get_inframode(ifh, &inframode));
            if(inframode != 1)
            {
                PRINT_ERROR();
            }
        }

    // bwl_up
    PRINT_ERROR_IF_FAILED(bwl_up(ifh));

        // bwl_isup
        {
            uint32_t isup = 0;
            PRINT_ERROR_IF_FAILED(bwl_isup(ifh, &isup));
            if(!isup)
            {
                PRINT_ERROR();
            }
        }

    PRINT_ERROR_IF_FAILED(bwl_clear_afsubtypes(ifh));

    PRINT_ERROR_IF_FAILED(bwl_add_afsubtype(ifh, 4));
    PRINT_ERROR_IF_FAILED(bwl_set_bcnlossthresh(ifh, 5));
    PRINT_ERROR_IF_FAILED(bwl_join(ifh, &bcm_join, sizeof(wlc_ssid_t), FALSE));

    if (g_BRCMenablelocal)
        PRINT_ERROR_IF_FAILED(bwl_set_ds3mode(ifh, 1));

    WaitEvent();

    /* get event mask for setting BCNRX_MSG */
    if (g_BRCMenableFlag & BCNRX_MSG_enabled)
    {
        unsigned char eventmask[WL_EVENTING_MASK_LEN];
        PRINT_ERROR_IF_FAILED(bwl_get_eventmask(ifh, eventmask));
        setbit(eventmask, WLC_E_BCNRX_MSG);
        clrbit(eventmask, WLC_E_PROBREQ_MSG);
        PRINT_ERROR_IF_FAILED(bwl_set_eventmask(ifh, eventmask));
    }

    /* get event mask for setting WLC_E_HTSFSYNC */
    if (g_BRCMenableFlag & TSF_MSG_enabled)
    {
        unsigned char eventmask[WL_EVENTING_MASK_LEN];
        PRINT_ERROR_IF_FAILED(bwl_get_eventmask(ifh, eventmask));
        setbit (eventmask,WLC_E_HTSFSYNC);
        clrbit(eventmask, WLC_E_PROBREQ_MSG);
        PRINT_ERROR_IF_FAILED(bwl_set_eventmask(ifh, eventmask));
    }

    // First Packet
    SendData(BROADCAST_ADDR);

    for(;;)
    {
    WaitReceivedEvent();
        if (g_BRCMenableTxpkt & 1)
            SendData(g_senderAddr);

        if (g_BRCMenableTxpkt & 2)
    SendData(BROADCAST_ADDR);
}
#else
    {
        char* argv[4] = {"wpa_supplicant_argument", "-iwlan0", "-Dbcm80211", "-C/data/misc/wifi/wlan0"};
        const int argc = sizeof(argv) / sizeof(char*);
        void *_wpas;
        APMode = 0;
        NN_LOG(">>>> %s::wpa_supplicant_main <<<<\n", __FUNCTION__);
        RegisterCallback();
        wpa_supplicant_main(argc, argv, &_wpas);
        wpa_s = _wpas;
        wpa_supplicant_register_notify_cb(wpa_s, &CbFunc_wpa_supp_notify);
        wpa_supplicant_register_l2_send_eapol(wpa_s, &TxEAPOL);
        NN_LOG("<<<< %s::wpa_supplicant_main - wpa_s - %p >>>>\n", __FUNCTION__, _wpas);


#ifdef WPA_SUPPLICANT_SCAN
#ifdef WPA_SUPPLICANT_ADV_SCAN
        wpa_supplicant_ctrl_scan(wpa_s, NULL, 0, NULL, 0, 120, 0, NULL);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
#else
        wpa_supplicant_ctrl_scan(wpa_s);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
//        wpa_supplicant_ctrl_scan_abort(wpa_s);
#endif
#endif

#ifdef WPA_SUPPLICANT_CONNECT
        {
            char psk[32] = "1234567890";
            char ssid[32] = "Broadcom_TEST7";
            char ssid_wep[32] = "ciscoap";
            char wep_key[32] = "1234567890";
            int wep_keyidx = 0;
            char pin[] = "91540256";
            uint16_t channel = 0;
            uint16_t num_channels = 0;

            switch(g_BRCMKeyMgmt) {
            case 0:
                // OPEN-NONE
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_NONE);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_NONE);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 1:
                // OPEN-WEP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_NONE);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_WEP);
                wpa_supplicant_ctrl_set_auth_type(wpa_s, WPA_SUPPLICANT_AUTH_OPEN);
                wpa_supplicant_ctrl_set_wep_key(wpa_s, wep_keyidx, &wep_key[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid_wep[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid_wep[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 2:
                // SHARED-WEP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_NONE);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_WEP);
                wpa_supplicant_ctrl_set_auth_type(wpa_s, WPA_SUPPLICANT_AUTH_SHARED);
                wpa_supplicant_ctrl_set_wep_key(wpa_s, wep_keyidx, &wep_key[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid_wep[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid_wep[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 3:
                // WPA-PSK TKIP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_WPA_PSK);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_TKIP);
                wpa_supplicant_ctrl_set_wsec_key(wpa_s, &psk[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 4:
                // WPA-PSK CCMP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_WPA_PSK);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_CCMP);
                wpa_supplicant_ctrl_set_wsec_key(wpa_s, &psk[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 5:
                // WPA2-PSK CCMP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_WPA2_PSK);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_CCMP);
                wpa_supplicant_ctrl_set_wsec_key(wpa_s, &psk[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 6:
                // WPA2-PSK TKIP
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_set_wpa_auth(wpa_s, WPA_SUPPLICANT_KEY_MGMT_WPA2_PSK);
                wpa_supplicant_ctrl_set_wsec(wpa_s, WPA_SUPPLICANT_WSEC_TKIP);
                wpa_supplicant_ctrl_set_wsec_key(wpa_s, &psk[0]);
#ifdef WPA_SUPPLICANT_ADV_SCAN
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0], num_channels, &channel);
#else
                wpa_supplicant_ctrl_set_ssid(wpa_s, &ssid[0]);
#endif
                wpa_supplicant_ctrl_enable_network(wpa_s);
                break;
            case 7:
                // WPS PBC
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_connect_wsc_pbc(wpa_s);
                break;
            case 8:
                // WPS PIN
                wpa_supplicant_ctrl_add_network(wpa_s);
                wpa_supplicant_ctrl_connect_wsc_pin(wpa_s, &pin[0]);
                break;
            default:
                NN_LOG("Un-recognized key_mgmt - %d.\n", g_BRCMKeyMgmt);
            }
        }
#endif
        for(;;)
        {
            WaitReceivedEvent();
            //SendData(BROADCAST_ADDR);
        }
    }
#endif

    TEST_LEAVE();
} //NOLINT(impl/function_size)

int TestTxRx::CbFunc_rxevent(void *event, void *buf)
{
    uint32_t *type = reinterpret_cast<uint32_t *>(event);
    wl_event_msg_t *ev = reinterpret_cast<wl_event_msg_t *>(event);
    void *drv_priv;
    void *evdata;
    void *pktdata;
    char *_reply = (char *) &reply[0];
    uint tsf_low;
    uint tsf_high;

    NN_LOG("<WLAN_TEST> %s ver:0x%x event:0x%x len:%d status 0x%x\n",__FUNCTION__,
        ev->version, ev->event_type, ev->datalen, ev->status);

    nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);
    pktdata = nn::mbuf::MbufTod(pMbuf);
    evdata = reinterpret_cast<void *>(
            reinterpret_cast<char *>(pktdata) + sizeof(bcm_event_t));


    /* normal event processing */
    switch(ev->event_type)
    {
    case WLC_E_LINK:
        NN_LOG("%s: WLC_E_LINK\n", __FUNCTION__);
        BinaryDump( ((char *)evdata),  ev->datalen, 1);
        break;
    case WLC_E_JOIN:
        SignalEvent();
        NN_LOG("<WLAN_TEST> [%s(%d) WLC_E_JOIN]\n",__FUNCTION__, __LINE__);
        break;

    case WLC_E_ASSOC_IND:
    case WLC_E_AUTH_IND:
        SignalEvent();
        NN_LOG("<WLAN_TEST> [%s(%d) WLC_E_JOIN]\n",__FUNCTION__, __LINE__);
        break;

    case WLC_E_ESCAN_RESULT:
        {
        NN_LOG("<WLAN_TEST> [%s(%d) ver:%d event:%d len:%d ev data 0x%x]\n",__FUNCTION__, __LINE__,
                ev->version, ev->event_type, ev->datalen, evdata);
//        BinaryDump( ((char *)evdata),  ev->datalen, 1);
//            dump_bss_info(evdata);

//            dump_bss_info(nn::mbuf::MbufTod(pMbuf));
        }
        break;

        case WLC_E_BCNRX_MSG:
        {
//        NN_LOG("<WLAN_TEST> [%s(%d) WLC_E_BCNRX_MSG]\n",__FUNCTION__, __LINE__);
//    BinaryDump( ((char *)buf+sizeof(wl_event_msg_t)),  ev->datalen, 1);
        }
        break;

        case WLC_E_PROBREQ_MSG:
//    NN_LOG("<WLAN_TEST> [%s(%d) WLC_E_PROBREQ_MSG]\n",__FUNCTION__, __LINE__);
        break;

        case WLC_E_HTSFSYNC:
        /*NN_LOG("<WLAN_TEST> [%s(%d) ver:%d event:%d len:%d ev data 0x%x]\n",__FUNCTION__, __LINE__,
                ev->version, ev->event_type, ev->datalen, evdata);
        BinaryDump( ((char *)evdata),  ev->datalen, 1); */
        tsf_low = ((uint *)evdata)[0];
        tsf_high = ((uint *)evdata)[1];
        NN_LOG("tsf_high - %08X tsf_low - %08X\n", tsf_high, tsf_low);
        break;

    default:
        NN_LOG("<WLAN_TEST> [%s(%d) ver:%d event:%d len:%d evdata 0x%x\n",__FUNCTION__, __LINE__,
                ev->version, ev->event_type, ev->datalen, evdata);
        break;
    }

#ifndef STA_BWL_MODE
    if (APMode == 0) {
        if (wpa_s) {
            drv_priv = wpa_supplicant_get_drv_priv(wpa_s);
            _bcm_event_handle_read(drv_priv, NULL, reinterpret_cast<char *>(pktdata) + sizeof(ether_header), 0);
        }
    }
#endif /* !STA_BWL_MODE */

#ifdef CONFIG_ACSD_SUPPORT
    if (APMode) {
        NN_LOG("<WLAN_TEST> get event:: type - %d. status - %d.\n", ev->event_type, ev->status);
        switch (ev->event_type)
        {
            case WLC_E_ESCAN_RESULT:
                NN_LOG("<WLAN_TEST> Update scan results through acsd_scan_results()\n");
                ACSD_LOCK();
                acsd_scan_results(pktdata);
                ACSD_UNLOCK();
                if (ev->status == 0x0) {
                    NN_LOG("<WLAN_TEST> RECV_SCAN_RESULT_STATUS_SUCCESS\n");
                }
                break;
            case WLC_E_DCS_REQUEST:
            case WLC_E_SCAN_COMPLETE:
            case WLC_E_PKTDELAY_IND:
            case WLC_E_TXFAIL_THRESH:
                NN_LOG("event type:%d, calling acsd_event_handler() for handling the event ...\n", ev->event_type);
            acsd_event_recv((unsigned char*)pktdata, (sizeof(bcm_event_t) + ev->datalen + 2) );
                NN_LOG("event type:%d, calling acsd_event_handler() handling event done...\n", ev->event_type);
                break;
            default:
                break;
        }
    }
#endif /* CONFIG_ACSD_SUPPORT */

    nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf *>(buf));
    return 0;
}

int TestTxRx::CbFunc_txcompletion(void *buf , int result)
{
    nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);

    if (g_BRCMenableDatalog & 1) {
    NN_LOG("<WLAN_TEST> [%s(%d)] result=%d\n",__FUNCTION__, __LINE__, result);
        BinaryDump(buf, (pMbuf->_len > 256 ? 256 : pMbuf->_len) , 1);
    }

        nn::mbuf::MbufFreem(reinterpret_cast<nn::mbuf::Mbuf *>(buf));

    return 0;
}

int TestTxRx::CbFunc_rxdata(void *buf)
{
    nn::mbuf::Mbuf *pMbuf = reinterpret_cast<nn::mbuf::Mbuf *>(buf);

    if (g_BRCMenableDatalog & 2) {
    NN_LOG("<WLAN_TEST> [%s(%d) protcol=%d top=0x%x len=%d]\n",__FUNCTION__, __LINE__,
            pMbuf->_protocol, pMbuf->_top, pMbuf->_len);

    BinaryDump(nn::mbuf::MbufTod(pMbuf), nn::mbuf::MbufLength(pMbuf, NULL) , 1);
    }
    uint8_t *receivedData = reinterpret_cast<uint8_t*>(nn::mbuf::MbufTod(pMbuf));

#ifndef STA_BWL_MODE
    {
        NN_LOG("%s:: Ether_Type - %x.\n", __func__, pMbuf->_protocol);
        if (pMbuf->_protocol == ETH_P_PAE)
            wpa_supplicant_rx_eapol_queue_pkt(wpa_s, &receivedData[6], (uint8_t *)(nn::mbuf::MbufTod(pMbuf)) + sizeof(ether_header), pMbuf->_len - sizeof(ether_header));
    }
#endif
    std::memcpy(g_senderAddr, &receivedData[6], 6);

    nn::mbuf::MbufFreem(pMbuf);

    SignalReceivedEvent();

    return 0;
}

int TestTxRx::CbFunc_wpa_supp_notify(void *ctx, int level, int type, const char *buf, int len)
{
    wl_event_msg_t *event = (wl_event_msg_t *)(buf);
    uint32_t event_type = event->event_type;


    switch (type) {
        case WPA_NOTIFY_SCAN_RESULTS:
        {
            NN_LOG("<WLAN_TEST> WPA_NOTIFY_SCAN_RESULTS\n");
        }
            break;
        case WPA_NOTIFY_CONNECTED:
            NN_LOG("<WLAN_TEST> WPA_NOTIFY_CONNECTED\n");
            if (event_type == WLC_E_ASSOC) {
                NN_LOG("<WPA_SUPP_TEST>WLC_E_ASSOC received\n");
                NN_LOG("<WPA_SUPP_TEST>STATUS = %d. ASSOC_STATUS = %d.\n", event->status, event->reason);
            } else
                NN_LOG("ERR:: in-correct wl_event_msg_t %d.\n", event_type);
            break;
        case WPA_NOTIFY_DISCONNECTED:
            NN_LOG("<WLAN_TEST> WPA_NOTIFY_DISCONNECTED\n");
            if ((event_type == WLC_E_DISASSOC) ||
                (event_type == WLC_E_DISASSOC_IND)) {
                NN_LOG("<WPA_SUPP_TEST>WLC_E_DISASSOC received\n");
                NN_LOG("<WPA_SUPP_TEST>STATUS = %d. DISASSOC_REASON = %d.\n", event->status, event->reason);
            } else if ((event_type == WLC_E_DEAUTH) ||
                       (event_type == WLC_E_DEAUTH_IND)) {
                NN_LOG("<WPA_SUPP_TEST>WLC_E_DEAUTH received\n");
                NN_LOG("<WPA_SUPP_TEST>STATUS = %d. DEAUTH_REASON = %d.\n", event->status, event->reason);
            } else
                NN_LOG("ERR:: in-correct wl_event_msg_t\n");
            break;
        case WPS_NOTIFY_SUCCESS:
            NN_LOG("<WLAN_TEST> WPS_NOTIFY_SUCCESS\n");
            break;
        default:
            NN_LOG("<WLAN_TEST> Un-recognized notify event - %d.\n", type);
    }

    return 0;
}

int TestTxRx::TxEAPOL(void *buf)
{
    NN_LOG("<WLAN_TEST> %s:: enter.\n", __func__);
    dhd_start_xmit(buf);
    return 0;
}

void TestTxRx::RegisterCallback()
{
    dhd_register_txcompletion_cb(ifh, &CbFunc_txcompletion);
    dhd_register_rxdata_cb(ifh, &CbFunc_rxdata);
    dhd_register_event_cb(ifh, &CbFunc_rxevent);

}
#if 1

char *
TestTxRx::local_wl_ether_etoa(const struct ether_addr *n)
{
    static char etoa_buf[ETHER_ADDR_LEN * 3];
    char *c = etoa_buf;
    int i;

    for (i = 0; i < ETHER_ADDR_LEN; i++) {
        if (i)
            *c++ = ':';
        c += sprintf(c, "%02X", n->octet[i] & 0xff);
    }
    return etoa_buf;
}

void
TestTxRx::dump_bss_info(void *wbi)
{
#define SSID_FMT_BUF_LEN (4*32+1)    //NOLINT(preprocessor/const)
    wl_scan_results_t *scanresult = reinterpret_cast<wl_scan_results_t *>(wbi);
    wl_bss_info_t *bi = scanresult->bss_info;
    char ssidbuf[SSID_FMT_BUF_LEN];
    char chspec_str[CHANSPEC_STR_LEN];
    wl_bss_info_107_t *old_bi;
    int mcs_idx = 0, start_idx = 0;
    bool start_idx_valid = FALSE;
    uint16 capability;

    /* Convert version 107 to 109 */
    if ((bi->version) == LEGACY_WL_BSS_INFO_VERSION) {
        old_bi = (wl_bss_info_107_t *)bi;
        bi->chanspec = CH20MHZ_CHSPEC(old_bi->channel);
        bi->ie_length = old_bi->ie_length;
        bi->ie_offset = sizeof(wl_bss_info_107_t);
    } else {
        /* do endian swap and format conversion for chanspec if we have
         * not created it from legacy bi above
         */
        //bi->chanspec = wl_chspec32_from_driver(bi->chanspec);
    }

    //wl_format_ssid(ssidbuf, bi->SSID, bi->SSID_len);


    NN_LOG("SSID: \"%s\"\n", bi->SSID);

    NN_LOG("Mode: %x\t", bi->capability);
    NN_LOG("RSSI: %d dBm\t", (int16)((bi->RSSI)));

    /*
     * SNR has valid value in only 109 version.
     * So print SNR for 109 version only.
     */
    if ((bi->version) == WL_BSS_INFO_VERSION) {
        NN_LOG("SNR: %d dB\t", (int16)((bi->SNR)));
    }

    NN_LOG("noise: %d dBm\t", bi->phy_noise);
    if (bi->flags) {
        uint16 flags = (bi->flags);
        NN_LOG("Flags: ");
        if (flags & WL_BSS_FLAGS_FROM_BEACON)
            NN_LOG("FromBcn ");
        if (flags & WL_BSS_FLAGS_FROM_CACHE)
            NN_LOG("Cached ");
        if (flags & WL_BSS_FLAGS_RSSI_ONCHANNEL)
            NN_LOG("RSSI on-channel ");
        NN_LOG("\t");
    }
    NN_LOG("Channel: %X\n", bi->chanspec);

    NN_LOG("BSSID: %s\t", local_wl_ether_etoa(&bi->BSSID));

    NN_LOG("Capability: ");
    capability = (bi->capability);
    if (capability & DOT11_CAP_ESS)
        NN_LOG("ESS ");
    if (capability & DOT11_CAP_IBSS)
        NN_LOG("IBSS ");
    if (capability & DOT11_CAP_POLLABLE)
        NN_LOG("Pollable ");
    if (capability & DOT11_CAP_POLL_RQ)
        NN_LOG("PollReq ");
    if (capability & DOT11_CAP_PRIVACY)
        NN_LOG("WEP ");
    if (capability & DOT11_CAP_SHORT)
        NN_LOG("ShortPre ");
    if (capability & DOT11_CAP_PBCC)
        NN_LOG("PBCC ");
    if (capability & DOT11_CAP_AGILITY)
        NN_LOG("Agility ");
    if (capability & DOT11_CAP_SHORTSLOT)
        NN_LOG("ShortSlot ");
    if (capability & DOT11_CAP_RRM)
        NN_LOG("RRM ");
    if (capability & DOT11_CAP_CCK_OFDM)
        NN_LOG("CCK-OFDM ");
    NN_LOG("\n");

    NN_LOG("Supported Rates: 0x%x count %d", bi->rateset.rates, bi->rateset.count);
    NN_LOG("\n");
/*
    if ((bi->ie_length))
        wl_dump_wpa_rsn_ies((uint8 *)(((uint8 *)bi) + (bi->ie_offset)),
                            (bi->ie_length));
*/
    if ((bi->version) != LEGACY_WL_BSS_INFO_VERSION && bi->n_cap) {
        if (bi->vht_cap)
            NN_LOG("VHT Capable:\n");
        else
            NN_LOG("HT Capable:\n");
        if (CHSPEC_IS8080(bi->chanspec)) {
/*
            NN_LOG("\tChanspec: 5GHz channel %d-%d 80+80MHz (0x%x)\n",
             wf_chspec_primary80_channel(bi->chanspec),
             wf_chspec_secondary80_channel(bi->chanspec),
             bi->chanspec);
*/
        }
        else {
            NN_LOG("\tChanspec: %sGHz channel %d %dMHz (0x%x)\n",
                CHSPEC_IS2G(bi->chanspec)?"2.4":"5", CHSPEC_CHANNEL(bi->chanspec),
                (CHSPEC_IS160(bi->chanspec) ?
                160:(CHSPEC_IS80(bi->chanspec) ?
                80 : (CHSPEC_IS40(bi->chanspec) ?
                40 : (CHSPEC_IS20(bi->chanspec) ? 20 : 10)))),
                bi->chanspec);
        }
        NN_LOG("\tPrimary channel: %d\n", bi->ctl_ch);
        NN_LOG("\tHT Capabilities: ");
        if ((bi->nbss_cap) & HT_CAP_40MHZ)
            NN_LOG("40Mhz ");
        if ((bi->nbss_cap) & HT_CAP_SHORT_GI_20)
            NN_LOG("SGI20 ");
        if ((bi->nbss_cap) & HT_CAP_SHORT_GI_40)
            NN_LOG("SGI40 ");
        NN_LOG("\n\tSupported HT MCS :");
        for (mcs_idx = 0; mcs_idx < (MCSSET_LEN * 8); mcs_idx++) {
            if (isset(bi->basic_mcs, mcs_idx) && !start_idx_valid) {
                NN_LOG(" %d", mcs_idx);
                start_idx = mcs_idx;
                start_idx_valid = TRUE;
            }

            if (!isset(bi->basic_mcs, mcs_idx) && start_idx_valid) {
                if ((mcs_idx - start_idx) > 1)
                    NN_LOG("-%d", (mcs_idx - 1));
                start_idx_valid = FALSE;

            }
        }
        NN_LOG("\n");

        if (bi->vht_cap) {
            int i;
            uint mcs, rx_mcs, prop_mcs = VHT_PROP_MCS_MAP_NONE;
            char mcs_str[16], rx_mcs_str[16];

            if (bi->vht_mcsmap) {
                NN_LOG("\tNegotiated VHT MCS:\n");
                for (i = 1; i <= VHT_CAP_MCS_MAP_NSS_MAX; i++) {
                    mcs = VHT_MCS_MAP_GET_MCS_PER_SS(i, (bi->vht_mcsmap));

                    if ((bi->length) >= (OFFSETOF(wl_bss_info_t,
                        vht_mcsmap_prop) + (bi->ie_length) +
                        sizeof(uint16))) {
                        prop_mcs = VHT_MCS_MAP_GET_MCS_PER_SS(i,
                            (bi->vht_mcsmap_prop));
                    }
                    strcpy(mcs_str,
                        (mcs == VHT_CAP_MCS_MAP_0_9 ? "0-9 " :
                        (mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8 " :
                        (mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7 " :
                         " -- ")))
                         );
                    if (prop_mcs != VHT_PROP_MCS_MAP_NONE)
                        strcpy(mcs_str,
                            (mcs == VHT_CAP_MCS_MAP_0_9 ? "0-11      " :
                            (mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8, 10-11" :
                            (mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7, 10-11" :
                             "    --    ")))
                             );

                    if (mcs != VHT_CAP_MCS_MAP_NONE) {
                        NN_LOG("\t\tNSS%d : %s \n", i,
                            mcs_str);
                    }
                }
            } else {
                NN_LOG("\tSupported VHT MCS:\n");
                for (i = 1; i <= VHT_CAP_MCS_MAP_NSS_MAX; i++) {
                    mcs = VHT_MCS_MAP_GET_MCS_PER_SS(i,
                        (bi->vht_txmcsmap));

                    rx_mcs = VHT_MCS_MAP_GET_MCS_PER_SS(i,
                        (bi->vht_rxmcsmap));

                    if ((bi->length) >= (OFFSETOF(wl_bss_info_t,
                        vht_txmcsmap_prop) + (bi->ie_length) +
                        sizeof(uint16))) {
                        prop_mcs = VHT_MCS_MAP_GET_MCS_PER_SS(i,
                            (bi->vht_txmcsmap_prop));
                    }

                    strcpy(mcs_str,
                        (mcs == VHT_CAP_MCS_MAP_0_9 ? "0-9 " :
                        (mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8 " :
                        (mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7 " : " -- "))));
                    if (prop_mcs != VHT_PROP_MCS_MAP_NONE)
                        strcpy(mcs_str,
                            (mcs == VHT_CAP_MCS_MAP_0_9 ? "0-11      " :
                            (mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8, 10-11" :
                            (mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7, 10-11" :
                             "    --    ")))
                            );

                    strcpy(rx_mcs_str ,
                        (rx_mcs == VHT_CAP_MCS_MAP_0_9 ? "0-9 " :
                        (rx_mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8 " :
                        (rx_mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7 " : " -- "))));
                    if (prop_mcs != VHT_PROP_MCS_MAP_NONE)
                        strcpy(rx_mcs_str,
                            (rx_mcs == VHT_CAP_MCS_MAP_0_9 ? "0-11      " :
                            (rx_mcs == VHT_CAP_MCS_MAP_0_8 ? "0-8, 10-11" :
                            (rx_mcs == VHT_CAP_MCS_MAP_0_7 ? "0-7, 10-11" :
                             "    --    ")))
                             );

                    if ((mcs != VHT_CAP_MCS_MAP_NONE) ||
                        (rx_mcs != VHT_CAP_MCS_MAP_NONE)) {
                        NN_LOG("\t\tNSS%d Tx: %s  Rx: %s\n", i,
                            mcs_str, rx_mcs_str);
                    }
                }
            }
        }
       //bi->chanspec = wl_chspec32_to_driver(bi->chanspec);
    }
/*
    if ((bi->ie_length))
    {
        wl_dump_wps((uint8 *)(((uint8 *)bi) + (bi->ie_offset)),
            (bi->ie_length));
    }
*/
    if ((bi->flags) & WL_BSS_FLAGS_HS20) {
        NN_LOG("Hotspot 2.0 capable\n");
    }
/*
    if (bcm_find_vs_ie((uint8 *)(((uint8 *)bi) + (bi->ie_offset)),
        (bi->ie_length),
        (uint8 *)WFA_OUI, WFA_OUI_LEN, WFA_OUI_TYPE_OSEN) != NULL) {
        NN_LOG("OSEN supported\n");
    }
*/
    NN_LOG("\n");
} //NOLINT(impl/function_size)
#endif


}
}

