﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/fatal/fatal_Api.h>
#include <nn/fs/fs_PriorityPrivate.h>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

// for vi
#include "../../Libraries/visrv/visrv_Log.h"
#include "../../Libraries/visrv/service/visrv_DriverConnection.h"
#include "../../Libraries/visrv/visrv_InitializeForSystemProcess.h"
#include "../../Libraries/visrv/visrv_ServerManager.h"
#include "../../Libraries/visrv/visrv_MemoryManagement.h"
#include "../../Libraries/visrv/master/visrv_Lib.h"
#include "../../Libraries/visrv/native/visrv_SurfaceComposerClientHolder.h"
#include "../../Libraries/visrv/native/visrv_SyncpointWaiter.h"
#include "../../Libraries/visrv/vic/visrv_VicTaskWorker.h"

// for cec
#include <nn/cec/cec_Server.h>

// for hdcp
#include <nn/hdcp/impl/hdcp_Lib.h>

// for caps:sc, caps:ss
#include "../../../../Iris/Sources/Libraries/capsrv/server/visrv_InitializeForScreenShotServer.h"

// for nvmm
#include <nn/mmnv/server/mmnv_RequestHipcServer.h>

//#define NN_VI_ENABLE_PRINT_MEMORY_USAGE 1


//----------------------------
// MemoryManagement
//----------------------------
namespace {
    // Add work around to NSBG-6386: double the amount of memory 'donated'.
    static const size_t DonateMemorySize = 2 * 2 * 128 * 1024; // TORIAEZU
    static const size_t HeapMemorySize   = 2 * 2 * 256 * 1024; // TORIAEZU

    NN_ALIGNAS(4096) char g_DonateMemory[DonateMemorySize];
    NN_ALIGNAS(4096) char g_HeapMemory[HeapMemorySize];

    nn::mem::StandardAllocator g_Allocator;


    void ReportMemoryUsage() NN_NOEXCEPT
    {
#ifdef NN_VI_ENABLE_PRINT_MEMORY_USAGE
        static size_t g_AllocatedSizeMax = 0;
        size_t s = HeapMemorySize - g_Allocator.GetTotalFreeSize();
        g_AllocatedSizeMax = std::max(g_AllocatedSizeMax, s);
        NN_SDK_LOG("[visrv][mem] usage: %lld (max %lld) B\n", static_cast<int64_t>(s), static_cast<int64_t>(g_AllocatedSizeMax));
#endif
    }

    void* AllocateGraphicsMemory(size_t size, size_t alignment, void*) NN_NOEXCEPT
    {
        return aligned_alloc(alignment, size);
    }
    void FreeGraphicsMemory(void* p, void*) NN_NOEXCEPT
    {
        free(p);
    }
    void* ReallocateGraphicsMemory(void* p, size_t newSize, void*) NN_NOEXCEPT
    {
        return realloc(p, newSize);
    }

    void SetupMemory() NN_NOEXCEPT
    {
        g_Allocator.Initialize(g_HeapMemory, HeapMemorySize);
        nv::SetGraphicsAllocator(AllocateGraphicsMemory, FreeGraphicsMemory, ReallocateGraphicsMemory, nullptr);
        // Initialize nv with system process port name
        nv::SetGraphicsServiceName("nvdrv:s");
        nv::InitializeGraphics(g_DonateMemory, DonateMemorySize);
    }

    void UpdateSystemConfig() NN_NOEXCEPT
    {
        nn::settings::system::TvSettings    settings;
        nn::settings::system::GetTvSettings(&settings);
        if(settings.hdmiContentType != nn::settings::system::HdmiContentType_Game)
        {
                settings.hdmiContentType = nn::settings::system::HdmiContentType_Game;
                nn::settings::system::SetTvSettings(settings);
        }
    }

}

extern "C" void* malloc(size_t size)
{
    auto p = g_Allocator.Allocate(size);
    ReportMemoryUsage();
    return p;
}

extern "C" void free(void* p)
{
    if(p)
    {
        g_Allocator.Free(p);
    }
    ReportMemoryUsage();
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = malloc(sum);
    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    void* q = g_Allocator.Reallocate(p, newSize);
    ReportMemoryUsage();
    return q;
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    void* p = g_Allocator.Allocate(size, alignment);
    ReportMemoryUsage();
    return p;
}

void* operator new(size_t size)
{
    return malloc(size);
}

void* operator new(size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void operator delete(void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

void operator delete(void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    free(ptr);
}

void* operator new[](size_t size)
{
    return malloc(size);
}

void* operator new[](size_t size, const std::nothrow_t&) NN_NOEXCEPT
{
    return malloc(size);
}

void operator delete[](void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

void operator delete[](void* ptr, const std::nothrow_t&) NN_NOEXCEPT
{
    free(ptr);
}

//----------------------------
// Threading
//----------------------------

namespace {
    static const size_t SyncpointThreadStackSize                  = 32 * 1024;
    static const size_t VicWorkerThreadStackSize                  = 16 * 1024;
    static const size_t CecMainStackSize                          =  8 * 1024;
    static const size_t CecCancelStackSize = 8 * 1024;
    static const size_t CapsrvScreenShotControlIpcThreadStackSize =  8 * 1024;
    static const size_t CapsrvScreenShotIpcThreadStackSize        =  4 * 1024;
    static const size_t CapsrvScreenShotWorkerThreadStackSize     = 16 * 1024;
    static const size_t MmnvServerThreadStackSize                 =  4 * 1024;

    NN_ALIGNAS(4096) char g_SyncpointThreadStackSize[SyncpointThreadStackSize];
    NN_ALIGNAS(4096) char g_VicWorkerThreadStackSize[VicWorkerThreadStackSize];
    NN_ALIGNAS(4096) char g_CecMainStack[CecMainStackSize];
    NN_ALIGNAS(4096) char g_CecCancelStack[CecCancelStackSize];
    NN_ALIGNAS(4096) char g_CapsrvScreenShotControlIpcThreadStack[CapsrvScreenShotControlIpcThreadStackSize];
    NN_ALIGNAS(4096) char g_CapsrvScreenShotControlRealtimeIpcThreadStack[CapsrvScreenShotControlIpcThreadStackSize];
    NN_ALIGNAS(4096) char g_CapsrvScreenShotIpcThreadStack       [CapsrvScreenShotIpcThreadStackSize];
    NN_ALIGNAS(4096) char g_CapsrvScreenShotWorkerThreadStack    [CapsrvScreenShotWorkerThreadStackSize];
    NN_ALIGNAS(4096) char g_MmnvServerThreadStack                [MmnvServerThreadStackSize];

    nn::os::ThreadType g_SyncpointThread;
    nn::os::ThreadType g_VicWorkerThread;
    nn::os::ThreadType g_CecMainThread;
    nn::os::ThreadType g_CecCancelThread;
    nn::os::ThreadType g_CapsrvScreenShotControlIpcThread;
    nn::os::ThreadType g_CapsrvScreenShotControlRealtimeIpcThread;
    nn::os::ThreadType g_CapsrvScreenShotIpcThread;
    nn::os::ThreadType g_CapsrvScreenShotWorkerThread;
    nn::os::ThreadType g_MmnvServerThread;

    void SyncpointThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting syncpoint waiter\n");
        nn::visrv::native::g_SyncpointWaiter.Run();
    }

    void VicWorkerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting vic worker\n");
        nn::visrv::vic::g_VicTaskWorker.Run();
    }

    void cecMainThreadProcedure(void* pVoidArg) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting CEC Service\n");
        nn::cec::server::LoopCecServer();
    }

    void cecCancelThreadProcedure(void* pVoidArg) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting CEC Cancel Service\n");
        nn::cec::server::LoopCecCancelServer();
    }

    void CapsrvScreenShotControlIpcThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting ScreenShotControl Service\n");
        nn::capsrv::server::ScreenShotControlServerThreadFunction(nullptr);
    }

    void CapsrvScreenShotControlRealtimeIpcThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting ScreenShotControl Service\n");
        nn::capsrv::server::ScreenShotControlRealtimeServerThreadFunction(nullptr);
    }

    void CapsrvScreenShotIpcThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting ScreenShot Service\n");
        nn::capsrv::server::ScreenShotServerThreadFunction(nullptr);
    }

    void CapsrvScreenShotWorkerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting screen shot worker\n");
        nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Background);
        nn::capsrv::server::ScreenShotWorkerThreadFunction(nullptr);
    }

    void MmnvServerThreadFunction(void*) NN_NOEXCEPT
    {
        NN_VISRV_LOG("Starting multimedia Service\n");
        nn::mmnv::server::InitializeRequestServer();
        nn::mmnv::server::LoopRequestServer();
    }
}


//----------------------------
// Main
//----------------------------

extern "C" void nninitStartup() NN_NOEXCEPT
{
    // NOTICE:
    //   VI process' process heap is managed by nn::visrv::ProcessHeapBlockManager.
    //   Don't call nn::os::SetMemoryHeapSize() or nn::os::AllocateMemoryBlock() directly.
    //
    //   The os heap should be used only for development or testing purpose.
    //   You can use g_Allocator instead of os heap.
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
    // do nothing
}

extern "C" void nnMain() NN_NOEXCEPT
{
    // INFO: Fatal 発生時にエラーレポートの記録のみを行うように
    nn::fatal::SetFatalPolicy(nn::fatal::FatalPolicy::FatalPolicy_OnlyErrorReport);

    NN_VISRV_LOG("Starting process\n");
    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_PRIORITY(vi, Main));
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(vi, Main));

    SetupMemory();

    UpdateSystemConfig();

    nn::os::CreateThread(&g_SyncpointThread, SyncpointThreadFunction, nullptr,
        g_SyncpointThreadStackSize, sizeof(g_SyncpointThreadStackSize), NN_SYSTEM_THREAD_PRIORITY(vi, Syncpoint));
    nn::os::SetThreadNamePointer(&g_SyncpointThread, NN_SYSTEM_THREAD_NAME(vi, Syncpoint));

    nn::os::CreateThread(&g_VicWorkerThread, VicWorkerThreadFunction, nullptr,
        g_VicWorkerThreadStackSize, sizeof(g_VicWorkerThreadStackSize), NN_SYSTEM_THREAD_PRIORITY(vi, VicWorker));
    nn::os::SetThreadNamePointer(&g_VicWorkerThread, NN_SYSTEM_THREAD_NAME(vi, VicWorker));

    nn::os::CreateThread(&g_CecMainThread, cecMainThreadProcedure, &g_CecMainThread,
        g_CecMainStack, CecMainStackSize, NN_SYSTEM_THREAD_PRIORITY(cec, Main));
    nn::os::SetThreadNamePointer(&g_CecMainThread, NN_SYSTEM_THREAD_NAME(cec, Main));
    nn::os::CreateThread(&g_CecCancelThread, cecCancelThreadProcedure, &g_CecCancelThread,
                         g_CecCancelStack, CecCancelStackSize, NN_SYSTEM_THREAD_PRIORITY(cec, CancelThread));
    nn::os::SetThreadNamePointer(&g_CecCancelThread, NN_SYSTEM_THREAD_NAME(cec, CancelThread));

    nn::os::CreateThread(&g_CapsrvScreenShotControlIpcThread, CapsrvScreenShotControlIpcThreadFunction, nullptr,
        g_CapsrvScreenShotControlIpcThreadStack, sizeof(g_CapsrvScreenShotControlIpcThreadStack), NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotControlIpcServer));
    nn::os::SetThreadNamePointer(&g_CapsrvScreenShotControlIpcThread, NN_SYSTEM_THREAD_NAME(capsrv, ScreenShotControlIpcServer));

    nn::os::CreateThread(&g_CapsrvScreenShotControlRealtimeIpcThread, CapsrvScreenShotControlRealtimeIpcThreadFunction, nullptr,
        g_CapsrvScreenShotControlRealtimeIpcThreadStack, sizeof(g_CapsrvScreenShotControlRealtimeIpcThreadStack), NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotWorkerBoosted));
    nn::os::SetThreadNamePointer(&g_CapsrvScreenShotControlRealtimeIpcThread, NN_SYSTEM_THREAD_NAME(capsrv, ScreenShotWorkerBoosted));

    nn::os::CreateThread(&g_CapsrvScreenShotIpcThread, CapsrvScreenShotIpcThreadFunction, nullptr,
        g_CapsrvScreenShotIpcThreadStack, sizeof(g_CapsrvScreenShotIpcThreadStack), NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotIpcServer));
    nn::os::SetThreadNamePointer(&g_CapsrvScreenShotIpcThread, NN_SYSTEM_THREAD_NAME(capsrv, ScreenShotIpcServer));

    nn::os::CreateThread(&g_CapsrvScreenShotWorkerThread, CapsrvScreenShotWorkerThreadFunction, nullptr,
        g_CapsrvScreenShotWorkerThreadStack, sizeof(g_CapsrvScreenShotWorkerThreadStack), NN_SYSTEM_THREAD_PRIORITY(capsrv, ScreenShotWorker));
    nn::os::SetThreadNamePointer(&g_CapsrvScreenShotWorkerThread, NN_SYSTEM_THREAD_NAME(capsrv, ScreenShotWorker));

    nn::os::CreateThread(&g_MmnvServerThread, MmnvServerThreadFunction, nullptr,
        &g_MmnvServerThreadStack, MmnvServerThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(capsrv, MmnvServer));
    nn::os::SetThreadNamePointer(&g_MmnvServerThread, NN_SYSTEM_THREAD_NAME(capsrv, MmnvServer));

    nn::visrv::service::InitializeDriverConnection();
    nn::visrv::native::InitializeSurfaceComposerClient();

    nn::hdcp::impl::Initialize();

    nn::visrv::InitializeForSystemProcess();
    nn::cec::server::InitializeCecServer();
    nn::capsrv::server::InitializeForScreenShotServer(&g_CapsrvScreenShotWorkerThread);
    nn::cec::server::InitializeCecCancelServer();

    NN_VISRV_LOG("Start Service\n");
    nn::os::StartThread(&g_SyncpointThread);
    nn::os::StartThread(&g_VicWorkerThread);
    nn::os::StartThread(&g_CecMainThread);
    nn::os::StartThread(&g_CecCancelThread);
    nn::os::StartThread(&g_CapsrvScreenShotWorkerThread);
    nn::os::StartThread(&g_CapsrvScreenShotIpcThread);
    nn::os::StartThread(&g_CapsrvScreenShotControlIpcThread);
    nn::os::StartThread(&g_CapsrvScreenShotControlRealtimeIpcThread);
    nn::os::StartThread(&g_MmnvServerThread);

    // Run VI main loop
    nn::visrv::g_ServerManager.Run();

    nn::cec::server::RequestStopCecCancelServer();
    nn::cec::server::RequestStopCecServer();
    nn::capsrv::server::FinalizeForScreenShotServer();
    nn::cec::server::FinalizeCecCancelServer();
    nn::cec::server::FinalizeCecServer();

    nn::hdcp::impl::Finalize();

    nn::os::WaitThread(&g_MmnvServerThread);
    nn::os::DestroyThread(&g_MmnvServerThread);
    nn::os::WaitThread(&g_CapsrvScreenShotControlRealtimeIpcThread);
    nn::os::DestroyThread(&g_CapsrvScreenShotControlRealtimeIpcThread);
    nn::os::WaitThread(&g_CapsrvScreenShotControlIpcThread);
    nn::os::DestroyThread(&g_CapsrvScreenShotControlIpcThread);
    nn::os::WaitThread(&g_CapsrvScreenShotControlIpcThread);
    nn::os::DestroyThread(&g_CapsrvScreenShotIpcThread);
    nn::os::WaitThread(&g_CapsrvScreenShotWorkerThread);
    nn::os::DestroyThread(&g_CapsrvScreenShotWorkerThread);
    nn::os::WaitThread(&g_CecMainThread);
    nn::os::DestroyThread(&g_CecMainThread);
    nn::os::WaitThread(&g_CecCancelThread);
    nn::os::DestroyThread(&g_CecCancelThread);
    nn::os::WaitThread(&g_VicWorkerThread);
    nn::os::DestroyThread(&g_VicWorkerThread);
    nn::os::WaitThread(&g_SyncpointThread);
    nn::os::DestroyThread(&g_SyncpointThread);

    nn::visrv::FinalizeForSystemProcess();

    NN_VISRV_LOG("Exiting process\n");
}
