﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <cstring>
#include <cctype>
#include "shell_Console.h"
#include "shell_CommandParser.h"

namespace nn { namespace shell {

CommandParser::CommandParser(char* pBuffer, const char* pCommandLine) NN_NOEXCEPT
    : m_pSrc(pCommandLine), m_pDst(pBuffer), m_HasNext(true)
{
    NN_SDK_ASSERT(pBuffer != NULL);
    NN_SDK_ASSERT(pCommandLine != NULL);
}

// 次の文字を返すが読み取りポインタは進めない
int CommandParser::Peek() NN_NOEXCEPT
{
    return static_cast<unsigned char>(*m_pSrc);
}

// 現在の文字を引数の文字として使用せず、読み取りポインタを次の文字へ進める
void CommandParser::SkipAndNext() NN_NOEXCEPT
{
    m_pSrc++;
}

// 現在の文字を引数の文字として使用して、読み取りポインタを次の文字へ進める
void CommandParser::UseAndNext() NN_NOEXCEPT
{
    *m_pDst = *m_pSrc;
    m_pDst ++;
    m_pSrc ++;
}

// 引数の開始を記録（引数の最初の文字を指している状態で呼ぶ）
CommandParser::ReadResult CommandParser::BeginArg() NN_NOEXCEPT
{
    if (m_pOutCommand->argc >= MaxCommandArgumentCount)
    {
        return ReadResult_TooManyArguments;
    }
    if( m_pOutCommand->pCommandLine == nullptr )
    {
        m_pOutCommand->pCommandLine = m_pSrc;
    }

    m_pOutCommand->argv[m_pOutCommand->argc] = m_pDst;
    m_pOutCommand->pArgumentPositions[m_pOutCommand->argc] = m_pSrc;
    return ReadResult_Ok;
}

// 引数の終了を記録（引数の最後の文字の直後を指している状態で呼ぶ）
void CommandParser::EndArg() NN_NOEXCEPT
{
    // ここで引数終端の '\0' を書き込みたいが、次に読む場所（m_pSrcが指している場所）の可能性があるため不可
    // 引数終端の位置を保存しておく
    m_pArgEnd[m_pOutCommand->argc] = m_pDst;

    m_pDst ++;
    m_pOutCommand->argc++;
}

// 引数終端の '\0' を書き込む
void CommandParser::NullTerminateArgs() NN_NOEXCEPT
{
    for (int i = 0; i < m_pOutCommand->argc; i ++)
    {
        *m_pArgEnd[i] = '\0';
    }
}


// pCommandLine に与えられた文字列から、コマンドを1つ読み取ります。
// コマンドは、';' で区切られた各部分文字列です。
//
// 各コマンドを空白で区切り、各部分文字列を0終端にして、
// それらの先頭へのポインタを pArgV に、部分文字列の数を pArgC の指す先に格納します。
// ただし、二重引用符で囲われた部分は区切りません。また、二重引用符は取り除かれます。
CommandParser::ReadResult CommandParser::ReadCommand(InputCommand* pOutCommand) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutCommand != NULL);
    NN_SDK_ASSERT(m_HasNext);

    if (!m_HasNext)
    {
        return ReadResult_Empty;
    }

    m_pOutCommand = pOutCommand;
    m_pOutCommand->argc = 0;
    m_pOutCommand->pCommandLine = nullptr;
    m_pOutCommand->commandLineBytes = 0;
    m_HasNext = false;

    while (true)
    {
        const int c = Peek();
        if (c == '\0')
        {
            if( m_pOutCommand->pCommandLine != nullptr )
            {
                m_pOutCommand->commandLineBytes = m_pSrc - m_pOutCommand->pCommandLine;
            }
            break;
        }
        else if (c == ';')
        {
            if( m_pOutCommand->pCommandLine != nullptr )
            {
                m_pOutCommand->commandLineBytes = m_pSrc - m_pOutCommand->pCommandLine;
            }
            SkipAndNext();
            m_HasNext = true;
            break;
        }
        else if (std::isspace(c))
        {
            SkipAndNext();
            continue;
        }
        else if (c == '"')
        {
            ReadResult result;
            result = BeginArg();
            if (result != ReadResult_Ok)
            {
                return result;
            }
            result = ReadQuotedToken();
            if (result != ReadResult_Ok)
            {
                return result;
            }
            EndArg();
        }
        else
        {
            ReadResult result;
            result = BeginArg();
            if (result != ReadResult_Ok)
            {
                return result;
            }
            result = ReadToken();
            if (result != ReadResult_Ok)
            {
                return result;
            }
            EndArg();
        }
    }

    NullTerminateArgs();
    return ReadResult_Ok;
}

// 二重引用符で囲まれたトークンを読み込み、引数とする
CommandParser::ReadResult CommandParser::ReadQuotedToken() NN_NOEXCEPT
{
    NN_SDK_ASSERT(Peek() == '"');
    SkipAndNext();

    while (true)
    {
        const char c = Peek();
        if (c == '\0')
        {
            return ReadResult_UnmatchedQuote;
        }
        else if (c == '"')
        {
            SkipAndNext();
            return ReadResult_Ok;
        }
        else
        {
            UseAndNext();
            continue;
        }
    }
}

// 空白または ';' で区切られた文字列を読み込み、引数とする
CommandParser::ReadResult CommandParser::ReadToken() NN_NOEXCEPT
{
    while (true)
    {
        const int c = Peek();
        if (c == '\0' || std::isspace(c) || c == ';')
        {
            return ReadResult_Ok;
        }
        else
        {
            UseAndNext();
            continue;
        }
    }
}

}}  // namespace nn { namespace shell {
