﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_DmntType.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/nn_Result.h>

#include <nn/result/result_HandlingUtility.h>

#include "profiler_UniqueSortedList.h"
#include "profiler_ThreadListItem.h"

namespace nn { namespace profiler {

typedef UniqueSortedList<::nn::Bit64, ::nn::profiler::ThreadListItem> ThreadList;

enum ProfilingMode : uint32_t
{
    ProfilingMode_Disabled      = 0,
    ProfilingMode_OutOfProcess  = (1 << 0),
    ProfilingMode_InProcess     = (1 << 1),
};
inline ProfilingMode operator|(ProfilingMode a, ProfilingMode b)
{
    return static_cast<ProfilingMode>(static_cast<std::underlying_type<ProfilingMode>::type>(a) | static_cast<std::underlying_type<ProfilingMode>::type>(b));
}
inline ProfilingMode& operator|=(ProfilingMode& a, ProfilingMode b)
{
    return (a = a | b);
}

void UpdateSingleThreadInfo(ThreadListItem* threadItem);
void DumpTargetApplicationInfo();

class TargetApplication
{
    NN_DISALLOW_COPY(TargetApplication);

public:
    static void Initialize();
    static void Finalize();

    static nn::Result Launch();
    static nn::Result ScheduleAttachOnLaunch(nn::ncm::ProgramId programId);
    static nn::Result AttachOnLaunch(nn::ncm::ProgramId programId);
    static nn::Result Attach(nn::ncm::ProgramId programId);
    static nn::Result Attach(nn::Bit64 processId);
    static nn::Result RegisterLibrary(uint64_t processId);
    static nn::Result UnregisterLibrary(uint64_t processId);
    static TargetApplication* GetCurrent();

private:
    static nn::Result AttachImpl(nn::Bit64 processId);

public:
    nn::Result Kill();
    nn::Result Close();

    bool IsAttached() const;
    bool IsLibraryInitialized() const;
    bool IsAttachedToLibrary() const;
    inline nn::Bit64 GetLibraryProcessId() const { return m_libraryPid; }

    uint32_t GetCoreMask();
    void SetCoreMask(uint32_t mask);
    int GetCoreCount();

    bool FindModuleName(uintptr_t baseaddr, uintptr_t endaddr, char* pOutName);
    bool FindModuleBuildId(uintptr_t baseaddr, uintptr_t endaddr, void* gnuBuildId);

    void GetApplicationName(const char*& pName, size_t& pLength) const;

    nn::Result GetProcessMemoryInfo(int *pOutCount, nn::svc::MemoryInfo *pOutMemoryInfo, int arrayCount);
    void FindCodeRegions();
    uintptr_t GetMinCodeAddress() const;
    uintptr_t GetMaxCodeAddress() const;
    int GetCodeRegionCount() const;

    bool Is64Bit() const;
    size_t GetPointerSize() const;

    nn::Result ReadMemory(uintptr_t buffer, uintptr_t address, size_t size) const;
    nn::Result WriteMemory(uintptr_t buffer, uintptr_t address, size_t size);
    nn::Result GetThreadContext(nn::svc::ThreadContext *context, nn::os::ThreadId threadId, uint32_t flags);
    nn::Result RequestBreak();
    nn::Result QueryMemory(nn::svc::MemoryInfo* pMemInfo, nn::svc::PageInfo* pPageInfo, uintptr_t address);
    nn::Result IsValidCodeAddress(uintptr_t address);

    bool GetStackStartFromThreadId(nn::os::ThreadId thread, uintptr_t* stackBase, uintptr_t sp) const;
    ThreadList* GetThreadList();
    void UpdateThreadInfo();

    uint32_t GetSdkVersion() const;
    void SetSdkVersion(uint32_t version);

    ProfilingMode GetProfilingMode() const;
    nn::Result SetProfilingMode(ProfilingMode mode);
    ProfilingMode GetAllAvailableProfilingModes() const;

    void StartProfiling(int64_t waitTime);
    void StopProfiling();

    void ChangeWatchThreadAffinity(int core);
    void DumpTimers() const;

    nn::svc::Handle GetDebugHandle();
    int64_t GetWaitTime();

protected:
    bool IsOutOfProcessAvailable() const;
    bool IsInProcessAvailable() const;

private:
    static TargetApplication* s_CurrentApplication;


private:
    nn::Bit64 m_processId;
    nn::Bit64 m_libraryPid;

    uint32_t m_sdkVersion;

    uint32_t m_CoreMask;
    int m_CoreCount;

    uint64_t m_minCodeAddress;
    uint64_t m_maxCodeAddress;
    uint64_t m_minStaticCodeAddress;
    uint64_t m_maxStaticCodeAddress;
    int m_codeRegionCount;
    int m_staticCodeRegionCount;

    ProfilingMode m_profilingMode;

    ThreadList m_threadList;
    bool m_isAttached;

private:
    TargetApplication();
    ~TargetApplication();

    void FindStaticCodeRegions();
    void FindDynamicCodeRegions();

    void AvailableProfilingModesChanged();
};


}} // nn::profiler
