﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma GCC diagnostic ignored "-Wsign-conversion"
#include <nn/os/os_ReaderWriterLock.h>
NN_PRAGMA_POP_WARNINGS

#include "profiler_Memory.h"


namespace nn { namespace profiler {


template<typename K, typename V>
class UniqueSortedList
{
public:
    static const size_t DefaultListSize = 64;

private:
    struct ListIndices
    {
        K key;
        V* value;
    };

private:
    static void* Allocate(size_t size)
    {
        return Memory::GetInstance()->Allocate(size);
    }

    static void Deallocate(void* ptr)
    {
        Memory::GetInstance()->Free(ptr);
    }

public:
    UniqueSortedList() : UniqueSortedList(DefaultListSize) {}
    NN_IMPLICIT UniqueSortedList(size_t count);
    ~UniqueSortedList();

    void Clear();

    int Count() const;

    V* Insert(K key);
    void Remove(K key);

    V* GetByKey(K key) const;
    V* GetByIndex(int index) const;

    /**
     *  @brief Acquires a read lock on the list.
     */
    void ReadLock() { nn::os::AcquireReadLock(&m_lock); }

    /**
     *  @brief Releases a read lock from the list.
     */
    void ReadUnlock() { nn::os::ReleaseReadLock(&m_lock); }

protected:
    void WriteLock() { nn::os::AcquireWriteLock(&m_lock); }
    void WriteUnlock() { nn::os::ReleaseWriteLock(&m_lock); }

private:
    int GetIndex(ListIndices* list, K key) const;

    nn::os::ReaderWriterLockType m_lock;

    ListIndices* m_list;
    size_t m_listSize;
    size_t m_listCount;
};


template<typename K, typename V>
UniqueSortedList<K, V>::UniqueSortedList(size_t count)
{
    const size_t sizeBytes = sizeof(ListIndices) * count;
    {
        m_list = reinterpret_cast<ListIndices*>(Allocate(sizeBytes));
        memset(m_list, 0, sizeBytes);
        m_listSize = count;
        m_listCount = 0;
    }
    {
        memset(&m_lock, 0, sizeof(m_lock));
        nn::os::InitializeReaderWriterLock(&m_lock);
    }
}


template<typename K, typename V>
UniqueSortedList<K, V>::~UniqueSortedList()
{
    Clear();
    nn::os::FinalizeReaderWriterLock(&m_lock);
    Deallocate(m_list);
    m_list = nullptr;
}


/**
 *  @brief Clears the list contents.
 *
 *  @details
 *  This function will clear all of the contents of the list, calling the destructors of
 *  any Value objects, and then free the memory for that Value object.
 */
template<typename K, typename V>
void UniqueSortedList<K, V>::Clear()
{
    WriteLock();
    for (int i = 0; i < Count(); ++i)
    {
        m_list[i].value->~V();
        Deallocate(m_list[i].value);
    }
    m_listCount = 0;
    memset(m_list, 0, sizeof(ListIndices) * m_listSize);
    WriteUnlock();
}


/**
 *  @brief Gets the number of Values in the list.
 *
 *  @return The number of Values in the list.
 */
template<typename K, typename V>
int UniqueSortedList<K, V>::Count() const
{
    return static_cast<int>(m_listCount);
}


/**
 *  @brief Inserts a Value with the given Key into the list.
 *
 *  @param [in] key The Key of Value that will be inserted.
 *
 *  @return The Value for the given Key.
 *
 *  @details
 *  This function will first check if the given Key is already in the list.
 *  If it is, it will return back the Value for that Key instead of inserting a new value.
 *  If it is not, a new entry will be made into the list, memory allocated for the Value object,
 *  and the initializer for that object called with the Key provided to the Value constructor.
 */
template<typename K, typename V>
V* UniqueSortedList<K, V>::Insert(K key)
{
    V* value = GetByKey(key);
    if (value != nullptr)
    {
        return value;
    }

    WriteLock();

    // Check if we need to expand the size of the list.
    // Note: The list size can never shrink, only grow.
    if (m_listSize <= m_listCount)
    {
        m_listSize *= 2;
        ListIndices* newList = reinterpret_cast<ListIndices*>(Allocate(m_listSize * sizeof(ListIndices)));
        memcpy(newList, m_list, m_listCount * sizeof(ListIndices));
        auto oldList = m_list;
        m_list = newList;
        Deallocate(oldList);
    }

    {
        // Find the index that we will be inserting at.
        // There is a small chance that we will find the item already inserted.
        // If this happens, do not add it again.
        int i = 0;
        for (; i < Count(); ++i)
        {
            auto testItem = m_list[i].key;
            if (key == testItem)
            {
                value = m_list[i].value;
                break;
            }
            else if (key < testItem || testItem == 0)
            {
                break;
            }
        }

        if (value == nullptr)
        {
            if (Count() > i)
            {
                // We are not inserting at the end of the list.
                // Move the list back.
                memmove(&m_list[i + 1], &m_list[i], sizeof(ListIndices) * size_t(Count() - i));
            }

            void* mem = Allocate(sizeof(V));
            value = ::new (mem) V(key);

            m_list[i].key = key;
            m_list[i].value = value;
            ++m_listCount;
        }
    }
    WriteUnlock();

    return value;
}


/**
*  @brief Removes a Value with the given Key into the list.
*
*  @param [in] key The Key of Value that will be removed.
*
*  @return The Value for the given Key.
*/
template<typename K, typename V>
void UniqueSortedList<K, V>::Remove(K key)
{
    V* value = nullptr;

    WriteLock();
    {
        // Find the index that we will be removing at.
        // There is a small chance that we will not find the item.
        // If this happens, removal is already complete.
        int index = GetIndex(m_list, key);
        if (index >= 0)
        {
            value = GetByIndex(index);

            if (index < Count() - 1)
            {
                int i = index;
                // We are not remove at the end of the list.
                // Move the list forward.
                memmove(&m_list[i], &m_list[i + 1], sizeof(ListIndices) * size_t(Count() - i - 1));
            }

            value->~V();
            Deallocate(value);
            --m_listCount;
        }
    }
    WriteUnlock();
}


/**
 *  @brief Gets the Value with the specified Key.
 *
 *  @param [in] key The Key of the Value to obtained.
 *
 *  @return The Value with the specified Key.
 *
 *  @details
 *  In general, this function should not be called directly.
 *  Instead, make use of the Insert() function as that will also result in the Key/Value being added
 *  to the list.
 *
 *  The goal of this function is to be as fast as possible for a lookup.
 */
template<typename K, typename V>
V* UniqueSortedList<K, V>::GetByKey(K key) const
{
    V* value = nullptr;

    // After much debate, the read locks in this function have been disabled.
    // There are a couple of potential issues with not having them, but the ones we could think of were deemed
    // insignificant enough compared to the gains of not locking.
    /* Known potential issues:
         1. False negatives.
            Because the list is sorted and we are looking up keys using a binary search, it is possible that
            another thread will write a key to the list while this is searching. This could push the item
            we are looking for outside of the current bounds. This will result in the item not being found
            even though it is in the list; resulting in a nullptr being returned.
            For the most part, this Get() function is not called directly, instead having everything routed
            through the call to Insert(). The Insert() function internally ensures that no key is stored
            in the list twice.
         2. Continued searching after a clear.
            If the list is cleared while a search is active, it is possible that a previously valid bound
            will no longer be valid. The clear will also result in memory being freed and zeroed. The zeroed
            memory poses no threat as we never attempt to dereference the Value data, instead just returning
            the pointer to that data. So the Get() function will return nullptr, the same as if nothing was
            found in the list. The caller of Get() will need to be sure to cope with this issue properly.
            For the freed issue, it is possible to end up with a stale pointer. There is no great way to
            avoid this, however, it should generally not cause a problem as if the data is wrong, it should
            not affect operations too badly.
    */

    // Make a local copy of this to avoid potential issues where the list is destroyed due to
    // the profiler being finalized while another thread on another core is in the middle of searching
    // through the list.
    // If ReadLock/ReadUnlock is re-enabled, this check can be removed.
    ListIndices* list = m_list;
    if (list == nullptr) { return value; }

    //ReadLock();
    int index = GetIndex(list, key);
    if (index >= 0)
    {
        value = list[index].value;
    }
    //ReadUnlock();

    return value;
}


/**
 *  @brief Gets the item at the specified index.
 *
 *  @param [in] index The index of the Value to obtain.
 *
 *  @return The Value at the specified @c index.
 *
 *  @details
 *  The general thought is that this function will only be called while attempting to walk the list.
 *  As such, ReadLock() should be called before walking the list and ReadUnlock() called afterwards.
 *  This will ensure that nothing is added to the list while walking it.
 */
template<typename K, typename V>
V* UniqueSortedList<K, V>::GetByIndex(int index) const
{
    V* value = nullptr;

    //ReadLock();
    if (index < Count())
    {
        value = m_list[index].value;
    }
    //ReadUnlock();
    return value;
}


template<typename K, typename V>
int UniqueSortedList<K, V>::GetIndex(ListIndices* list, K key) const
{
    int lowerBound = 0;
    int upperBound = Count() - 1;
    while (lowerBound <= upperBound)
    {
        int index = ((upperBound + lowerBound) / 2);
        ListIndices& listItem = list[index];
        auto testThread = listItem.key;
        if (key == testThread)
        {
            return index;
        }
        else if (key < testThread)
        {
            upperBound = index - 1;
        }
        else if (key > testThread)
        {
            lowerBound = index + 1;
        }
    }
    return -1;
}


}} // nn::profiler
