﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>

namespace nn { namespace profiler {

class StringTable
{
public:
    static int GetVersion() NN_NOEXCEPT;

public:
    StringTable() NN_NOEXCEPT;
    ~StringTable() NN_NOEXCEPT;

    uintptr_t FindString(const char* str) const NN_NOEXCEPT;
    uintptr_t AddString(const char* str) NN_NOEXCEPT;
    void Clear() NN_NOEXCEPT;

    static const size_t MaxStrLen = 254;
    static const size_t TableSize = 128 * 1024;

    inline uint32_t GetOffset(uintptr_t value) const NN_NOEXCEPT
    {
        if (value == UINTPTR_MAX) { return static_cast<uint32_t>(value); }
        uintptr_t table = reinterpret_cast<uintptr_t>(m_table);
        return static_cast<uint32_t>(value - table);
    }

    int GetStringCount() const NN_NOEXCEPT;

    void* GetTableAddress() const NN_NOEXCEPT
    {
        return m_table;
    }

    size_t GetUsedTableSize() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_current >= m_table);
        return static_cast<size_t>(m_current - m_table);
    }

    size_t GetTotalSize() const NN_NOEXCEPT
    {
        return m_size;
    }

    void OverwriteWithCopy(const char* table, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(size <= m_size);
        memcpy(m_table, table, size);
        m_current = m_table + size;
    }

protected:
    uintptr_t FindString(const char* str, size_t len, size_t hash) const NN_NOEXCEPT;

private:
    char* m_table;
    size_t m_size;
    std::atomic<char*> m_current;
};

}} // namespace nn::profiler
