﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/pcm/driver/pcm.h>
#include <nn/pcm/driver/pcm_Suspend.h>
#include <nn/pcm/pcm_ServiceName.h>
#include <nn/pcm/pcm_IManager.sfdl.h>
#include <nn/pcm/pcm_ManagerImpl.h>
#include <nn/pcm/pcm_ShimInternal.h>

#include "nvdbgsvc_pscUtil.h"

namespace nn { namespace nvdbgsvc {

namespace {

// サーバー側から発行する全セッションの Max
// TORIAEZU : そんなに大量にはいらないはず
const int PcmSessionCountMax = 8;

nn::sf::UnmanagedServiceObject<nn::pcm::IManager, nn::pcm::ManagerImpl> g_PcmManager;

// TORIAEZU : 今はマネージャーは 1つだけ
class PcmServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<PcmSessionCountMax, 1>
{};

PcmServerManager g_PcmServerManager;

const size_t PcmStackSize = 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_PcmServerStack[PcmStackSize];
nn::os::ThreadType g_PcmServerThread;

// スリープ等の通知を受け取るオブジェクト
nn::psc::PmModule g_PmModule;

void LoopPcmServer(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::psc::PmState    state         = nn::psc::PmState_Unknown;
    nn::psc::PmState    previousState = nn::psc::PmState_Unknown;
    nn::psc::PmFlagSet  flags;
    static const detail::ServerThreadParam<PcmServerManager> threadParam =
    {
        &g_PmModule,
        &g_PcmServerManager,
        detail::StateEdge_MinimumAwakeToSleepReady
    };

    bool ipcEnable = true;

    while (NN_STATIC_CONDITION(true))
    {
        if (ipcEnable)
        {
            // IPC 許可状態 (IPC ループ)
            detail::IpcLoop<PcmServerManager, nn::psc::PmModuleId_Pcm>(&previousState, &state, &flags, &threadParam);
        }
        else
        {
            // IPC 禁止状態 (PSC のイベントのみ待ち受ける)
            g_PmModule.GetEventPointer()->Wait();
            g_PmModule.GetEventPointer()->Clear();

            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));
        }

        auto stateEdge = detail::GetStateEdge(previousState, state);
        switch (stateEdge)
        {
        case detail::StateEdge_MinimumAwakeToSleepReady:
            NN_SDK_LOG("[nvdbgsvc] Pcm suspend\n");
            nn::pcm::driver::SuspendAllMeasuringPoints();
            ipcEnable = false;
            break;

        case detail::StateEdge_EssentialServicesAwakeToMinimumAwake:
            NN_SDK_LOG("[nvdbgsvc] Pcm resume\n");
            nn::pcm::driver::ResumeAllMeasuringPoints();
            ipcEnable = true;
            break;

        case detail::StateEdge_SleepReadyToEssentialServicesSleepReady:
        case detail::StateEdge_EssentialServicesSleepReadyToEssentialServicesAwake:
            ipcEnable = false;
            break;

        default:
            ipcEnable = true;
            break;;
        }

        g_PmModule.Acknowledge(state, nn::ResultSuccess());
        previousState = state;
    }
}

}

void StartPcmServer() NN_NOEXCEPT
{
    // PCM ドライバライブラリおよびサーバの初期化
    nn::pcm::driver::Initialize();
    nn::pcm::InitializeWith(g_PcmManager.GetShared());

    // サービス名の登録とポートの初期化
    // 現在はどのプロセスからアクセスしても 1つのマネージャーにアクセスされる
    g_PcmServerManager.RegisterObjectForPort(nn::pcm::GetInternalManager(), PcmSessionCountMax, nn::pcm::ServiceName);

    // PSC へのモジュール登録
    auto result = g_PmModule.Initialize(
        nn::psc::PmModuleId_Pcm,
        detail::PcmDependencies,
        sizeof(detail::PcmDependencies) / sizeof(detail::PcmDependencies[0]),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_PcmServerManager.Start();

    // Pcm 用サーバ処理スレッドの作成
    result = nn::os::CreateThread(&g_PcmServerThread,
                                    LoopPcmServer,
                                    nullptr,
                                    g_PcmServerStack,
                                    sizeof(g_PcmServerStack),
                                    NN_SYSTEM_THREAD_PRIORITY(pcm, IpcServer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_PcmServerThread, NN_SYSTEM_THREAD_NAME(pcm, IpcServer));

    nn::os::StartThread(&g_PcmServerThread);
    NN_SDK_LOG("[nvdbgsvc] Start PCM Server\n");
}

void RequestStopPcmServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_PcmServerManager.RequestStop();
}

void WaitAndFinalizePcmServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_PcmServerThread);
    nn::os::DestroyThread(&g_PcmServerThread);

    nn::pcm::driver::Finalize();
}

}}
