﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/psc.h>
#include <nn/bgtc.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "npns_SystemEventHandlerThread.h"
#include "npns_Common.h"
#include "npns_Instance.h"
#include "npns_ClientThread.h"
#include "npns_StateMachineThread.h"

namespace nn{ namespace npns{

int32_t GetSleepPeriodicInterval()
{
    int32_t interval;
#ifdef NN_NPNS_TEST_PERIODIC_INTERVAL_SECONDS
    interval = NN_NPNS_TEST_PERIODIC_INTERVAL_SECONDS;
#else
    interval = g_Daemon.GetController().GetLastBrokerCheckingIntervalFromServer();
#endif
    if (interval == 0)
    {
        if (nn::settings::fwdbg::GetSettingsItemValue(
            &interval, sizeof(interval), "npns", "sleep_periodic_interval"
        ) != sizeof(interval))
        {
            NN_NPNS_WARN("fwdbg::GetSettingsItemValue failed.");
        }

    }
    return interval;
}

int32_t GetSleepMaxTryCount()
{
    int32_t count;
#ifdef NN_NPNS_TEST_TRY_MAX_COUNT_ON_HALFAWAKE
    count = NN_NPNS_TEST_TRY_MAX_COUNT_ON_HALFAWAKE;
#else
    count = g_Daemon.GetController().GetLastKeepSessionTryCountFromServer();
#endif
    if (count == 0)
    {
        if (nn::settings::fwdbg::GetSettingsItemValue(
            &count, sizeof(count), "npns", "sleep_max_try_count"
        ) != sizeof(count))
        {
            NN_NPNS_WARN("fwdbg::GetSettingsItemValue failed.");
        }
    }
    return count;
}

SystemEventHandlerThread::SystemEventHandlerThread()
    : ThreadBase(NN_NPNS_THREAD_CONFIG(SystemEventHandler))
    , m_CurrentRetryCountOnHalfAwake(0)
    , m_CurrentPeriodicInterval(0)
{
}

SystemEventHandlerThread::~SystemEventHandlerThread()
{
}

Result SystemEventHandlerThread::Initialize()
{
    const psc::PmModuleId pmDependencies[] = {
        psc::PmModuleId_Socket,
        psc::PmModuleId_Bgtc,
        // スリープ突入時にはファイルを触らない
        // psc::PmModuleId_Fs,
        psc::PmModuleId_Nifm,
    };

    Result result = PowerStateEventHandler::Initialize(
        psc::PmModuleId_Npns,
        pmDependencies, sizeof(pmDependencies) / sizeof(pmDependencies[0])
    );
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = ThreadBase::Initialize();
    if (result.IsFailure())
    {
        PowerStateEventHandler::Finalize();
        return result;
    }

    return ResultSuccess();
}

void SystemEventHandlerThread::Finalize()
{
    RequestExit();
    ThreadBase::Finalize();

    PowerStateEventHandler::Finalize();
}

void SystemEventHandlerThread::ThreadBody()
{
    Result result;
    os::SystemEvent& eventPowerState = PowerStateEventHandler::GetEvent();
    os::SystemEvent& eventTrigger  = bgtc::GetTriggerEvent();
    os::SystemEvent& eventSchedule = bgtc::GetScheduleEvent();
    os::Tick tickHalfAwakeStarted = os::Tick(0);
    while (NN_STATIC_CONDITION(true))
    {
        int index = os::WaitAny(
            eventPowerState.GetBase(),
            eventTrigger.GetBase(),
            eventSchedule.GetBase()
        );
        if (IsExiting())
        {
            break;
        }

        // 半起床共通処理
        if (bgtc::IsInHalfAwake() && (index == 1 || index == 2))
        {
            NN_NPNS_RECORD_EVENT(Halfawake);

            if (tickHalfAwakeStarted.GetInt64Value() == 0)
            {
                tickHalfAwakeStarted = os::GetSystemTick();
            }
        }

        switch (index)
        {
        case 0:
            if (tickHalfAwakeStarted.GetInt64Value() > 0)
            {
                int32_t duration = std::max<int32_t>(
                    0, static_cast<int32_t>((os::GetSystemTick() - tickHalfAwakeStarted).ToTimeSpan().GetSeconds())
                );
                NN_NPNS_RECORD_ACCUMULATED_VALUE(DurationHalfawake, duration);
                tickHalfAwakeStarted = os::Tick(0);
            }

            eventPowerState.Clear();
            PowerStateEventHandler::Dispatch();
            break;

        case 1: // eventTrigger : 他モジュールの相乗り or WoWL トリガ等により接続の確認を行う場合
            eventTrigger.Clear();

            // ScheduleEvent を優先して処理する
            if (!eventSchedule.TimedWait(TimeSpan::FromMilliSeconds(100)))
            {
                OnReceiveTriggerEvent(&eventPowerState);
                break;
            }
            else
            {
                NN_FALL_THROUGH;
            }

        case 2: // eventSchedule: スケジュールにより半起床になった or 半起床中に長時間のタスクが続いた場合
            eventSchedule.Clear();
            eventTrigger.Clear(); // TriggerEvent を再処理してしまうのを防止
            OnReceiveScheduleEvent(&eventPowerState);
            NN_NPNS_RECORD_EVENT(HalfawakeScheduled);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void SystemEventHandlerThread::OnReceivePowerStateEvent(psc::PmState stateLast, psc::PmState stateTo, psc::PmFlagSet flagset)
{
    ClientThread& client   = g_Daemon.GetClientThread();
    StateMachineThread& sm = g_Daemon.GetStateMachineThread();

    switch (stateTo)
    {
    case psc::PmState_FullAwake:
        sm.OnFullAwakeEnter();

        // FullAwake 中は bgtc は使わないのでスケジュール設定を解除
        bgtc::UnscheduleTask();
        NN_NPNS_RECORD_EVENT(Fullawake);
        break;

    case psc::PmState_MinimumAwake:
        if (IsTrrigeredFromFullAwake())
        {
            sm.OnFullAwakeLeave();
        }
        break;

    case psc::PmState_SleepReady:
    {
        // 明示的な切断を行う前が一番情報が揃っているので状況を保存しておく
        const bool bIsEnabledRealTimePushDuringSleep = Controller::IsEnabledRealTimePushDuringSleep();

        // スリープ遷移時に接続を維持できるかどうかを判定する
        // NOTE: 毎回切断しても良いがサーバの負荷を下げるため、維持できる場合は維持できるようにしている
        const bool bShouldKeepSession =
            // 接続の維持ができる電源状態にあるか？
            bIsEnabledRealTimePushDuringSleep
            // Proxy 接続の場合は WSP 無効化のため一旦切断する必要がある
            && !client.IsUsingProxy()
            // USB-Ether の場合はスリープ遷移で接続が切れてしまうので、綺麗に切断しておく
            && !bgtc::WillDisconnectNetworkWhenEnteringSleep();

        // 判定に応じて、半起床に備えた状態にするか一旦切断するか処理を行う
        sm.OnSleepEnter(bShouldKeepSession);

        if (IsTrrigeredFromFullAwake())
        {
            UpdateSchedule();

#if NN_NPNS_ENABLE_WOWLAN
            // OnSleepEnter の結果、半起床に備えた常時接続状態に遷移しなかった場合は即起きの予約をする
            if (bIsEnabledRealTimePushDuringSleep && sm.GetCurrentState() != State_ConnectedOnHalfAwake)
            {
                NN_NPNS_INFO("Leaving from full-awake and scheduled entring to half-awake immediately in order to establish a XMPP session.\n");
                bgtc::NotifyTaskStarting();
            }

            NN_NPNS_INFO("Will attempt connection %d times during %d seconds.\n", GetSleepMaxTryCount(), GetSleepPeriodicInterval());
            m_CurrentRetryCountOnHalfAwake = 0;
#endif
        }
        else if (IsPeriodicIntervalChanged())
        {
            NN_NPNS_INFO("Periodic interval has been changed %d -> %dseconds.\n", m_CurrentPeriodicInterval, GetSleepPeriodicInterval());
            UpdateSchedule();
        }
        NN_NPNS_RECORD_EVENT(Sleep);
#if NN_NPNS_ENABLE_LOG_VERBOSE
        g_Daemon.GetStatistics().ShowAll();
#endif
        break;
    }
    case psc::PmState_EssentialServicesSleepReady:
    case psc::PmState_EssentialServicesAwake:
        break;

    case psc::PmState_ShutdownReady:
        sm.OnShutdown();
        client.OnShutdown();
        break;

    default:
        break;
    }
}

void SystemEventHandlerThread::OnReceiveTriggerEvent(os::SystemEvent * pCancelEvent)
{
    // 全起床中は自律動作に任せる
    if (!bgtc::IsInHalfAwake())
    {
        return;
    }

    Result result;
    if (Controller::IsEnabledRealTimePushDuringSleep())
    {
        const int32_t maxRetryCount = GetSleepMaxTryCount() - 1;
        if (m_CurrentRetryCountOnHalfAwake <= maxRetryCount)
        {
            NN_NPNS_INFO("HalfAwake triggered by non-scheduled events. (online = %s, retry processed = %d/%d)\n",
                g_Daemon.GetController().IsOnline() ? "yes" : "no",
                m_CurrentRetryCountOnHalfAwake, maxRetryCount);
            bool bIsRetryProcessed = false;
            const bool PerformPingPong = false;
            // AP 切断イベントもスケジューリングイベントも全てひっくるめて処理
            g_Daemon.GetStateMachineThread().OnHalfAwakeEnter(pCancelEvent, PerformPingPong, &bIsRetryProcessed);
            if (bIsRetryProcessed)
            {
                NN_NPNS_INFO("Recovered XMPP session.\n");
                ++m_CurrentRetryCountOnHalfAwake;
                NN_NPNS_RECORD_EVENT(HalfawakeSessionRecovered);
            }
        }
        else
        {
            NN_NPNS_WARN("HalfAwake triggered, but exeeded retry count(%d).\n", maxRetryCount);
        }
    }
#if defined(NN_DETAIL_ENABLE_SDK_STRUCTURED_LOG)
    else
    {

        bgtc::Interval interval;
        result = bgtc::GetScheduledTaskInterval(&interval);
        if (result.IsSuccess() && interval > 0)
        {
            NN_NPNS_INFO("HalfAwake triggered, but next arrival check is scheduled after %d sec.\n", interval);
        }
    }
#endif
}

void SystemEventHandlerThread::OnReceiveScheduleEvent(os::SystemEvent* pCancelEvent)
{
    // 全起床中は自律動作に任せる
    if (!bgtc::IsInHalfAwake())
    {
        return;
    }

    m_CurrentRetryCountOnHalfAwake = 0;
    const bool PerformPingPong = true;
    g_Daemon.GetStateMachineThread().OnHalfAwakeEnter(pCancelEvent, PerformPingPong);
}

void SystemEventHandlerThread::RequestExit()
{
    PowerStateEventHandler::GetEvent().Signal();
    ThreadBase::RequestExit();
}

void SystemEventHandlerThread::UpdateSchedule()
{
    bgtc::Interval periodicInterval = GetSleepPeriodicInterval();
    // 起床予定時刻を予想しやすくするため周期タイマを使う
    bgtc::SchedulePeriodicTask(periodicInterval, periodicInterval);
    m_CurrentPeriodicInterval = periodicInterval;
}

bool SystemEventHandlerThread::IsPeriodicIntervalChanged() const
{
    return m_CurrentPeriodicInterval != GetSleepPeriodicInterval();
}

}
}
