﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/bgtc.h>
#include <cstdlib>

#include "npns_Common.h"
#include "npns_StateMachine.h"

namespace nn {
namespace npns {

namespace
{
    template <typename T>
    bool GetSettingsFwdbgValue(T* pOut, const char* keyString)
    {
        size_t bytes = nn::settings::fwdbg::GetSettingsItemValue(pOut, sizeof(T), "npns", keyString);
        return(bytes == sizeof(T));
    }
}

StateMachine::StateMachine(Controller& controller, nn::os::Event& eventInterrupt)
    : m_Controller(controller)
    , m_eventInterrupt(eventInterrupt)
    , m_BackoffWait(0)
    , m_SerialErrorCount(0)
    , m_bUserDriven(false)
    , m_bError(false)
    , m_bAutonomy(true)
    , m_bEulaAcceptedOnce(false)
{
    m_bAutonomy = IsBackgroundProcessingEnabled();
}

StateMachine::~StateMachine()
{
}

Result StateMachine::Initialize()
{
    Result result = StateMachineBase::Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    result = m_Controller.Initialize();
    NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

    return result;
}

void StateMachine::Finalize()
{
    m_Controller.Finalize();

    StateMachineBase::Finalize();
}

bool StateMachine::IsBackgroundProcessingEnabled()
{
#ifdef NN_NPNS_ENABLE_AUTONOMY
# if NN_NPNS_ENABLE_AUTONOMY
    return true;
# else
    return false;
# endif
#else
# ifndef NN_BUILD_CONFIG_OS_WIN
    bool isEnabled = true;
    GetSettingsFwdbgValue(&isEnabled, "background_processing");
    return isEnabled;
# else
    return true;
# endif
#endif
}

bool StateMachine::IsEulaAccepted()
{
#if NN_NPNS_ENABLE_EULA_CHECK
    if (m_bEulaAcceptedOnce)
    {
        // 一度 EULA 通過したら以降のチェックはスキップ
        return true;
    }

    bool bAccepted = false;

    settings::system::RegionCode regionCode;
    settings::system::GetRegionCode(&regionCode);

    settings::system::EulaVersion* pEulaVers = new settings::system::EulaVersion[settings::system::EulaVersionCountMax];
    if (pEulaVers)
    {
        int count = settings::system::GetEulaVersions(pEulaVers, settings::system::EulaVersionCountMax);
        for (int i = 0; i < count; ++i)
        {
            settings::system::EulaVersion& eulaVer = pEulaVers[i];
            if (eulaVer.regionCode == regionCode)
            {
                bAccepted = true;
                break;
            }
        }
        delete [] pEulaVers;
    }
    if (bAccepted)
    {
        NN_NPNS_TRACE("Eula was accepted.\n");
        m_bEulaAcceptedOnce = true;
    }
    return bAccepted;
#else
    return true;
#endif
}

TimeSpan StateMachine::GetSleepProcessingTimeout()
{
    int32_t timeout;
#ifdef NN_NPNS_TEST_SLEEP_PROCESSING_TIMEOUT_SECONDS
    timeout = NN_NPNS_TEST_SLEEP_PROCESSING_TIMEOUT_SECONDS;
#else
    GetSettingsFwdbgValue(&timeout, "sleep_processing_timeout");
#endif
    return TimeSpan::FromSeconds(timeout);
}

bool StateMachine::IsTargetState(State state) const
{
    switch (state)
    {
    case State_Online:
    case State_Connected:
    case State_Suspend:
    case State_Exit:
    case State_Idle:
    case State_ConnectedOnHalfAwake:
        return true;

    case State_Initial:
    case State_AwaitingOnline:
    case State_NoValidJid:
    case State_RequestingJid:
    case State_ReadyToConnect:
    case State_Connecting:
    case State_BackoffWaiting:
    case State_ShutingDown:
    case State_CheckingArrival:
    default:
        return false;
    }
}

Result StateMachine::OnTransit()
{
    Result result;

    const State stateTarget = GetTargetState();

#ifndef NN_SWITCH_DISABLE_DEBUG_PRINT
    if (GetCurrentState() != GetTargetState())
    {
        NN_NPNS_INFO("Changing: Current=[%s], Target=[%s]\n",
            GetStateString(GetCurrentState()), GetStateString(GetTargetState()));
    }
    else
    {
        NN_NPNS_INFO("Staying: Current=[%s]\n", GetStateString(GetCurrentState()));
    }
#endif

    switch (GetCurrentState())
    {
    case State_Initial:
        m_eventInterrupt.TimedWait(TimeSpan::FromSeconds(10));
        result = SetCurrentState(State_Idle);
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
        break;
/*
        switch (stateTarget)
        {
        case State_Online:
        case State_Connected:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
        default:
            NN_SDK_ASSERT(false);
        }
        break;
*/
    case State_Idle:
        switch (stateTarget)
        {
        case State_Online:
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            if (IsEulaAccepted())
            {
                result = SetCurrentState(State_AwaitingOnline);
            }
            else
            {
                if (stateTarget != State_ConnectedOnHalfAwake)
                {
                    result = SetCurrentState(State_AwaitingEulaAccepted);
                }
                else
                {
                    result = ResultEulaIsNotAccepted();
                }
            }
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Suspend:
            result = SetCurrentState(State_Suspend);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Exit:
            result = SetCurrentState(State_ShutingDown);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Idle:
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_AwaitingEulaAccepted:
        switch (stateTarget)
        {
        case State_Online:
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            while (!IsEulaAccepted())
            {
                result = WaitAnyEvents(nn::TimeSpan::FromSeconds(30));
                if (result.IsFailure())
                {
                    return result;
                }
            }
            result = SetCurrentState(State_Idle);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Idle:
        case State_Suspend:
        case State_Exit:
            result = SetCurrentState(State_Idle);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;
        default:
            NN_SDK_ASSERT(false);
        }
        break;



    case State_AwaitingOnline:
        switch (stateTarget)
        {
        case State_Online:
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            NN_NPNS_INFO("Waiting network available...\n");
            result = m_Controller.WaitForOnline(&m_eventInterrupt, m_bUserDriven);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);

            NN_NPNS_RECORD_EVENT(Online);
            break;

        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Idle);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_Online:
        switch (stateTarget)
        {
        case State_Online:
            if (!m_Controller.IsOnline())
            {
                return ResultDisconnected();
            }
            break;

        case State_Connected:
        case State_ConnectedOnHalfAwake:
            if (!m_Controller.IsOnline())
            {
                return ResultDisconnected();
            }
            else if (m_Controller.HasValidJid())
            {
                result = SetCurrentState(State_ReadyToConnect);
                NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            }
            else
            {
                result = SetCurrentState(State_NoValidJid);
                NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            }
            break;

        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Idle);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_NoValidJid:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            result = SetCurrentState(State_RequestingJid);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_RequestingJid:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            result = SetCurrentState(State_ReadyToConnect);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_ReadyToConnect:
        switch (stateTarget)
        {
        case State_Connected:
            result = SetCurrentState(State_Connecting);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_ConnectedOnHalfAwake:
            result = SetCurrentState(State_CheckingArrival);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_Connecting:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            NN_NPNS_INFO("Waiting connection...\n");
            while (m_Controller.XmppIsConnectionInProgress())
            {
                result = WaitAnyEvents();
                if (result.IsFailure())
                {
                    m_Controller.XmppDisconnect();
                    return result;
                }
            }

            if (m_Controller.XmppIsConnected())
            {
                NN_NPNS_RECORD_EVENT(Connect);
                if (bgtc::IsInHalfAwake())
                {
                    NN_NPNS_RECORD_EVENT(ConnectHalfawake);
                }
                m_Controller.XmppSetupConnectionForSleep();
                result = SetCurrentState(stateTarget);
            }
            else
            {
                NN_NPNS_RECORD_EVENT(ConnectError);
                result = m_Controller.XmppGetLastError();
            }
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_Connected:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            if (!m_Controller.XmppIsConnected()
                || !m_Controller.IsOnline())
            {
                NN_NPNS_RECORD_EVENT(ConnectionLost);
                if (!m_Controller.IsOnline())
                {
                    NN_NPNS_RECORD_EVENT(ConnectionLostOffline);
                }
                NN_NPNS_TRACE("Detected disconnect. XmppIsConnected: %d, IsOnline: %d\n"
                    , m_Controller.XmppIsConnected(), m_Controller.IsOnline());
                return ResultDisconnected();
            }
            if (stateTarget == State_ConnectedOnHalfAwake)
            {
                result = SetCurrentState(State_ConnectedOnHalfAwake);
            }
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;
    case State_ConnectedOnHalfAwake:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
            if (!m_Controller.XmppIsConnected()
                || !m_Controller.IsOnline())
            {
                NN_NPNS_RECORD_EVENT(ConnectionLost);
                if (!m_Controller.IsOnline())
                {
                    NN_NPNS_RECORD_EVENT(ConnectionLostOffline);
                }
                NN_NPNS_TRACE("Detected disconnect. XmppIsConnected: %d, IsOnline: %d\n"
                    , m_Controller.XmppIsConnected(), m_Controller.IsOnline());
                return ResultDisconnected();
            }
            if (stateTarget == State_Connected)
            {
                result = SetCurrentState(State_Connected);
            }
            break;

        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    case State_BackoffWaiting:
        if (m_Controller.IsOnline())
        {
            // 自ら NIFM に接続要求を出していなければ接続状態にはならないので、
            // すでに EULA チェックはスキップしているはず
            result = SetCurrentState(State_Online);
        }
        else
        {
            result = SetCurrentState(State_Idle);
        }
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
        break;

    case State_Suspend:
        switch (stateTarget)
        {
        case State_Online:
        case State_Connected:
        case State_ConnectedOnHalfAwake:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Idle);
            break;

        case State_Suspend:
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;
    case State_ShutingDown:
        SetCurrentState(State_Exit);
        return ResultSuccess();

    case State_Exit:
        return ResultSuccess();

    case State_CheckingArrival:
        switch (stateTarget)
        {
        case State_Connected:
        case State_ConnectedOnHalfAwake:
        {
            bool bIsArrived = true;
            NN_NPNS_RECORD_EVENT(CheckArrival);
            result = m_Controller.CheckMessageArrived(&bIsArrived, &m_eventInterrupt);
            if (result.IsFailure())
            {
                NN_NPNS_RECORD_EVENT(CheckArrivalError);
                return result;
            }

            if (!bIsArrived && !m_Controller.IsEnabledRealTimePushDuringSleep())
            {
                // 新着なし、かつ間欠起動のみの動作状態ならここで処理を止める
                return ResultNoMessageArrived();
            }

            result = SetCurrentState(State_Connecting);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;
        }
        case State_Online:
        case State_Suspend:
        case State_Exit:
        case State_Idle:
            result = SetCurrentState(State_Online);
            NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
            break;

        default:
            NN_SDK_ASSERT(false);
        }
        break;

    default:
        NN_SDK_ASSERT(false);
    }

    return ResultSuccess();
} // NOLINT(impl/function_size)

Result StateMachine::OnEntering(State stateTo)
{
    NN_NPNS_TRACE("OnEntering: [%s] => [%s]\n", GetStateString(GetCurrentState()), GetStateString(stateTo));
    Result result;
    switch (stateTo)
    {
    case State_Initial:
        break;

    case State_Idle:
        if (GetCurrentState() != State_Initial && GetCurrentState() != State_Suspend)
        {
            m_Controller.CancelOnlineRequest();
        }
        break;

    case State_AwaitingEulaAccepted:
        break;

    case State_AwaitingOnline:
        if (GetCurrentState() == State_Idle)
        {
            m_Controller.SubmitOnlineRequest();
        }
        break;

    case State_Online:
        break;

    case State_NoValidJid:
        break;

    case State_RequestingJid:
        result = m_Controller.RequestJid();
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
        break;

    case State_ReadyToConnect:
        break;

    case State_Connecting:
        if (GetTargetState() == State_ConnectedOnHalfAwake)
        {
            m_Controller.XmppSetStatus(Client::Status_Inactive);
        }
        else
        {
            m_Controller.XmppSetStatus(Client::Status_Active);
        }
        result = m_Controller.XmppConnect();
        NN_NPNS_DETAIL_RETURN_IF_FAILED(result);
        break;

    case State_Connected:
        if (GetCurrentState() == State_ConnectedOnHalfAwake)
        {
            m_Controller.XmppSetStatus(Client::Status_Active);
        }

        if (m_Controller.XmppIsUsingProxy())
        {
            // 間欠起動でない && プロキシ接続時 の場合 WSP 有効化
            bool isPowerModeNormal = true; // TODO 電力モードを settings から取得
            m_Controller.XmppEnableWhiteSpacePing(&m_eventInterrupt, isPowerModeNormal);
        }

        m_Controller.RequestPlayReportSubmissionIfRequired();
        break;

    case State_ConnectedOnHalfAwake:
        if (GetCurrentState() == State_Connected)
        {
            m_Controller.XmppSetStatus(Client::Status_Inactive);
            // Proxy 使用時は State_ConnectedOnHalfAwake 前に一旦切断されているので WSP が有効になっていることはない
        }
        break;

    case State_BackoffWaiting:
        break;

    case State_Suspend:
    case State_ShutingDown:
    case State_Exit:
    case State_CheckingArrival:
        break;
#if 0
    case State_Initial:
    case State_Idle:
    case State_AwaitingOnline:
    case State_Online:
    case State_NoValidJid:
    case State_RequestingJid:
    case State_ReadyToConnect:
    case State_Connecting:
    case State_Connected:
    case State_Disconnecting:
    case State_BackoffWaiting:
    case State_Suspend:
    case State_ShutingDown:
    case State_Exit:
        break;
#endif
    default:
        NN_SDK_ASSERT(false);
    }
    return ResultSuccess();
}

Result StateMachine::OnLeaving(State stateFrom, State stateTo)
{
    NN_NPNS_TRACE("OnLeaving: [%s] => [%s]\n", GetStateString(stateFrom), GetStateString(stateTo));
    switch (stateFrom)
    {
    case State_Initial:
        break;

    case State_Idle:
        break;

    case State_AwaitingEulaAccepted:
        break;

    case State_AwaitingOnline:
        break;

    case State_Online:
        break;

    case State_NoValidJid:
    case State_RequestingJid:
    case State_ReadyToConnect:
    case State_Connecting:
        if (stateTo != State_Connected && stateTo != State_ConnectedOnHalfAwake)
        {
            m_Controller.XmppDisconnect();
        }
        break;

    case State_Connected:
    case State_ConnectedOnHalfAwake:
        if (stateTo != State_Connected && stateTo != State_ConnectedOnHalfAwake)
        {
            m_Controller.XmppDisconnect();
            if (m_Controller.XmppIsConnected())
            {
                WaitAnyEvents(nn::TimeSpan::FromSeconds(2));
            }
        }
        break;

    case State_BackoffWaiting:
        if (m_bUserDriven)
        {
            NN_NPNS_INFO("Skipping backoff wait because the current request is drived by user.\n");
        }
        else
        {
            NN_NPNS_INFO("Waiting %d seconds to retry...\n", m_BackoffWait.GetSeconds());
            WaitAnyEvents(m_BackoffWait);
            NN_NPNS_INFO("Start retry...\n");
        }
        break;

    case State_Suspend:
    case State_ShutingDown:
    case State_Exit:
    case State_CheckingArrival:
        break;
    default:
        NN_SDK_ASSERT(false);
    }
    return ResultSuccess();
}

void StateMachine::SetTargetState(State stateTarget, bool bUser)
{
    NN_NPNS_INFO("New target state [%s]\n", GetStateString(stateTarget));
    m_bUserDriven = bUser;
    m_bError = false;
    StateMachineBase::SetTargetState(stateTarget);
}

void StateMachine::OnTransitionComplete()
{
    NN_NPNS_INFO("Transition complete [%s]\n", GetStateString(GetCurrentState()));

    m_bUserDriven = false;

    if (GetCurrentState() == State_Connected || GetCurrentState() == State_ConnectedOnHalfAwake)
    {
        m_SerialErrorCount = 0;
    }

    if (GetCurrentState() != State_Exit && m_bAutonomy)
    {
        if (bgtc::IsInHalfAwake())
        {
            if (GetCurrentState() == State_Online)
            {
                SetTargetState(State_ConnectedOnHalfAwake, false);
            }
        }
        else
        {
            if (GetCurrentState() != State_Connected)
            {
                SetTargetState(State_Connected, false);
            }
        }
    }
}

bool StateMachine::OnTransitionFailure(Result result)
{
    NN_NPNS_INFO("Transition failure current=[%s], target=[%s], result=0x%08x\n",
        GetStateString(GetCurrentState()), GetStateString(GetTargetState()), result.GetInnerValueForDebug());

    m_stateLastTarget = GetTargetState();

    switch (GetCurrentState())
    {
    case State_AwaitingEulaAccepted:
        NN_SDK_ASSERT(
            ResultInterruptByRequest::Includes(result) ||
            ResultCommandTimeOut::Includes(result)
            , "%08x", result.GetInnerValueForDebug());
        // retry
        return false;

    case State_NoValidJid:
        break;

    case State_Connecting:
        if (result <= ResultAuthenticationFailed())
        {
            m_Controller.InvalidateJid();
            SetCurrentState(State_NoValidJid);
            if (m_bUserDriven)
            {
                // retry
                m_bUserDriven = false;
                return true;
            }
        }
        break;
    default:
        break;
    }

    if (IsAutonomyEnabled())
    {
        CalculateBackoffWait();

        SetCurrentState(State_BackoffWaiting);
        m_bUserDriven = false;
        m_eventInterrupt.Clear();
    }

    if (m_Controller.IsOnline())
    {
        SetTargetState(State_Online, false);
    }
    else
    {
        SetTargetState(State_Idle, false);
    }
    return false;
}

void StateMachine::CalculateBackoffWait()
{
    if (m_bUserDriven)
    {
        m_SerialErrorCount = 0;
    }

#if NN_NPNS_ENABLE_BACKOFF_ON_ERROR
    int64_t backoffSecondsBase = NN_NPNS_BACKOFF_SECONDS_INITIAL;

    if (m_SerialErrorCount++ == 0)
    {
        m_BackoffWait = TimeSpan::FromSeconds(backoffSecondsBase);
    }
    else
    {
        int randomValue;
        nn::os::GenerateRandomBytes(&randomValue, sizeof(randomValue));
        if (randomValue < 0) { randomValue *= -1; }
        int32_t incrementBase  = NN_NPNS_BACKOFF_SECONDS_INCREMENT * m_SerialErrorCount;
        int32_t incrementValue = static_cast<int>(randomValue % (incrementBase / 2)) + (incrementBase / 2);
        int32_t effectiveValue = std::min<int32_t>(static_cast<int32_t>(m_BackoffWait.GetSeconds()) + incrementValue, NN_NPNS_BACKOFF_SECONDS_MAX);
        NN_NPNS_TRACE("Backoff wait is changed: %d -> %d\n", m_BackoffWait.GetSeconds(), effectiveValue);
        m_BackoffWait = TimeSpan::FromSeconds(effectiveValue);
    }
#else
    m_BackoffWait = TimeSpan::FromSeconds(backoffSecondsBase);
#endif

    // server-kick で受け取った backoff 秒数を最小値とする
    m_BackoffWait = std::max(m_BackoffWait, m_Controller.GetBackoffWaitFromServer());
}

const char* StateMachine::GetStateString(State state) const
{
    const char* ppStrings[] =
    {
        "State_Initial",
        "State_Idle",
        "State_AwaitingOnline",
        "State_Online",
        "State_NoValidJid",
        "State_RequestingJid",
        "State_ReadyToConnect",
        "State_Connecting",
        "State_Connected",
        "State_BackoffWaiting",
        "State_Suspend",
        "State_ShutingDown",
        "State_Exit",
        "State_AwaitingEulaAccepted",
        "State_CheckingArrival",
        "State_ConnectedOnHalfAwake"
    };
    int index = static_cast<int>(state);
    const int count = sizeof(ppStrings) / sizeof(ppStrings[0]);
    if (index < 0 || index >= count)
    {
        return "unknown";
    }
    return ppStrings[index];
}

Result StateMachine::WaitAnyEvents()
{
    return m_Controller.WaitAnyEvents(&m_eventInterrupt);
}

Result StateMachine::WaitAnyEvents(const TimeSpan & time)
{
    return m_Controller.WaitAnyEvents(&m_eventInterrupt, time);
}

bool StateMachine::SetAutonomyEnabled(bool bEnable)
{
    bool bPrevious = m_bAutonomy;
    m_bAutonomy = bEnable;
    if (!bPrevious && bEnable)
    {
        m_eventInterrupt.Signal();
    }
    return bPrevious;
}



}
}
