﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/nfc/server/nfc_CreateUserManagerByDfc.h> // for nfc IUserManager
#include <nn/nfc/server/nfc_CreateSystemManagerByDfc.h> // for nfc ISystemManager
#include <nn/nfc/detail/nfc_Port.h>
#include <nn/nfp/server/nfp_CreateUserManagerByDfc.h> // for nfp IUserManager
#include <nn/nfp/server/nfp_CreateSystemManagerByDfc.h> // for nfp ISystemManager
#include <nn/nfp/server/nfp_CreateDebugManagerByDfc.h> // for nfp IDebugManager
#include <nn/nfp/detail/nfp_Port.h>
#include <nn/nfc/am/server/nfc_CreateAmManagerByDfc.h> // for nfc am IAmManager
#include <nn/nfc/am/detail/nfc_AmPort.h>
#include <nn/nfc/mifare/server/nfc_MifareCreateUserManagerByDfc.h> // for IUserManager
#include <nn/nfc/mifare/detail/nfc_MifarePort.h>

namespace {

enum PortIndex
{
    PortIndex_NfcUser,
    PortIndex_NfcSystem,
    PortIndex_NfpUser,
    PortIndex_NfpSystem,
    PortIndex_NfpDebug,
    PortIndex_NfcAm,
    PortIndex_NfcMifare,
    PortIndex_Count
};

const int HipcServerThreadCount = 4;
const size_t HipcServerThreadStackSize = 8192;
NN_ALIGNAS(4096) char g_HipcServerThreadStacks[HipcServerThreadCount][HipcServerThreadStackSize];
nn::os::ThreadType  g_HipcServerThreads[HipcServerThreadCount];

static const int ClientCountMax = 4; //同時利用するアプリケーションの最大数
static const int TotalSessionCount = (nn::nfc::detail::UserSessionCount + nn::nfp::detail::UserSessionCount) * ClientCountMax + nn::nfc::am::detail::AmSessionCount; //(例) (nn::nfc の nfc on/off と 最高 4 devices を同時利用 + nn::nfp の 最高 4 devices を同時利用) * 同時利用するアプリケーションの最大数 + nfc:am

struct MyServerOption
{
    static const size_t PointerTransferBufferSize = 4 * 1024;
    static const int SubDomainCountMax = 1 + 2 * ClientCountMax; // (例) nfc:am + (nfc:user + nfp:user) * 同時利用するアプリケーションの最大数
    static const int ObjectInSubDomainCountMax = 1 + 1 + 2 * ClientCountMax;// (例) 1 + nfc:am + (nfc:user + nfp:user) * 同時利用するアプリケーションの最大数
};

class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count, MyServerOption>
{
};

std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager* g_pMyServerManager;

void LoopHipcServerThreadFunction(void* arg)
{
    MyServerManager* pManager = reinterpret_cast<MyServerManager*>(arg);

    // IPCリクエスト処理開始
    pManager->LoopAuto();
}

} // anonymous

void InitializeServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!g_pMyServerManager, "Server is already initialized.\n");

    g_pMyServerManager = new (&g_MyServerManagerStorage) MyServerManager;

    nn::Result result;
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfc::server::CreateUserManagerByDfc(), ClientCountMax, nn::nfc::detail::UserPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfc::server::CreateSystemManagerByDfc(), ClientCountMax, nn::nfc::detail::SystemPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfp::server::CreateUserManagerByDfc(), ClientCountMax, nn::nfp::detail::UserPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfp::server::CreateSystemManagerByDfc(), ClientCountMax, nn::nfp::detail::SystemPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfp::server::CreateDebugManagerByDfc(), ClientCountMax, nn::nfp::detail::DebugPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfc::am::server::CreateAmManagerByDfc(), 1, nn::nfc::am::detail::AmPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = g_pMyServerManager->RegisterObjectForPort(
        nn::nfc::mifare::server::CreateUserManagerByDfc(), ClientCountMax, nn::nfc::mifare::detail::UserPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    g_pMyServerManager->Start();

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::Result result = nn::os::CreateThread(&g_HipcServerThreads[i],
                                                 LoopHipcServerThreadFunction,
                                                 g_pMyServerManager,
                                                 g_HipcServerThreadStacks[i],
                                                 HipcServerThreadStackSize,
                                                 NN_SYSTEM_THREAD_PRIORITY(nfc, HipcServer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::os::SetThreadNamePointer(&g_HipcServerThreads[i], NN_SYSTEM_THREAD_NAME(nfc, HipcServer));
    }
}

void LoopServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager, "Server is not initialized.\n");

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::os::StartThread( &g_HipcServerThreads[i] );
    }

    g_pMyServerManager->LoopAuto(); // RequestStop() が呼ばれるまで返らない
}

void FinalizeServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager, "Server is not initialized.\n");

    for(int i = 0; i < HipcServerThreadCount; ++i)
    {
        nn::os::WaitThread(&g_HipcServerThreads[i]);
        nn::os::DestroyThread(&g_HipcServerThreads[i]);
    }

    g_pMyServerManager->~MyServerManager();
    g_pMyServerManager = nullptr;
}
