﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>

#include <nn/ns/ns_DevelopApi.h>

#include <nn/err/err_Result.h>
#include <nn/fatal/fatal_ApiPrivate.h>

#include <nn/creport/creport_Result.h>
#include "creport_CrashReport.h"

namespace nn { namespace creport {

namespace
{

Bit64 ConvertAsciiDecimalToBit64(const char* decimal) NN_NOEXCEPT
{
    Bit64 value = 0;
    for (int i=0; decimal[i] && i<20; ++i) // 2^64 は 10 進数で 20 桁なのでそれ以上見ない
    {
        if (decimal[i] >= '0' && decimal[i] <= '9')
        {
            value = value * 10 + decimal[i] - '0';
        }
        else
        {
            break;
        }
    }
    return value;
}

} // ~namespace <anonymous>

}}

extern "C" void nninitStartup()
{
}
extern "C" void nndiagStartup()
{
}

nn::creport::CrashReport g_Creport;

extern "C" void nnMain()
{
    // 引数からクラッシュした pid を取得
    int numArgs = nn::os::GetHostArgc();
    char** args = nn::os::GetHostArgv();
    for (int i=0; i<numArgs; ++i)
    {
        NN_SDK_LOG("[creport] args[%d] = [%s]\n", i, args[i]);
    }

    if (numArgs < 2)
    {
        return;
    }

    for (int i=0; i<numArgs; ++i)
    {
        if (args[i] == nullptr)
        {
            return;
        }
    }

    nn::os::ProcessId processId;
    bool              isDetailed;

    processId.value        = nn::creport::ConvertAsciiDecimalToBit64( args[0] );
    isDetailed             = (args[1][0] == '1') ? true : false;

    //
    // クラッシュレポートを作成
    //
    if (g_Creport.Attach(processId.value, isDetailed))
    {
        g_Creport.CollectDebugEvent();

        if (g_Creport.IsApplication())
        {
            g_Creport.CollectDyingMessage();
        }

        g_Creport.CreateErrorReport();
        g_Creport.Detach();
    }

    if (g_Creport.GetResult() <= nn::creport::ResultInvalidResult())
    {
        // JIT Debug に入っていなかった可能性が高いのでプロセスを Terminate せずに終了
        return;
    }

    // ns にプロセスの終了を依頼
    if (nn::ns::InitializeForDevelop().IsSuccess())
    {
        nn::ns::TerminateProcess(processId);
    }

    // 最後に fatal に飛ぶ
    // アプリの場合は fatal に飛ばしてはいけない
    if (g_Creport.IsApplication() == false && g_Creport.IsUserBreak() == false)
    {
        nn::fatal::ThrowFatalWithoutErrorReport(g_Creport.GetResult());
    }
}
