﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet.h>

#include <mutex>

#include "btm_AppletManager.h"

#define NN_BTM_SCOPED_LOCK(Mutex) \
        std::lock_guard<decltype(Mutex)> NN_MAKE_TEMPORARY_NAME(locker)(Mutex)

namespace nn { namespace btm {

    AppletManager::AppletManager()
    {
        m_AppletResourceUserEntryMutex = NN_OS_SDK_RECURSIVE_MUTEX_INITIALIZER();

        for (auto appletResourceUser : m_AppletResourceUserEntry)
        {
            appletResourceUser.id.aruid     = nn::applet::AppletResourceUserId::GetInvalidId();
            appletResourceUser.id.appletId  = static_cast<uint32_t>(nn::applet::AppletId_None);
            appletResourceUser.isRegistered = false;
        }

        m_InFocusedAppletResourceUser.aruid    = nn::applet::AppletResourceUserId::GetInvalidId();
        m_InFocusedAppletResourceUser.appletId = static_cast<uint32_t>(nn::applet::AppletId_None);
    }

    AppletManager::~AppletManager()
    {
    }

    bool AppletManager::RegisterAppletResourceUser(const nn::applet::AppletResourceUserId& aruid,
                                                   uint32_t appletId) NN_NOEXCEPT
    {
        NN_BTM_SCOPED_LOCK(m_AppletResourceUserEntryMutex);

        bool isRegisterSucceeded = false;

        int userIndex = IsAppletResourceUserIdRegistered(aruid);

        if (userIndex >= 0)
        {
            isRegisterSucceeded = true;
        }
        else
        {
            for (auto& user : m_AppletResourceUserEntry)
            {
                if (!user.isRegistered)
                {
                    user.isRegistered   = true;
                    user.id.aruid       = aruid;
                    user.id.appletId    = appletId;

                    isRegisterSucceeded = true;

                    break;
                }
            }
        }

        return isRegisterSucceeded;
    }

    bool AppletManager::UnregisterAppletResourceUser(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
    {
        NN_BTM_SCOPED_LOCK(m_AppletResourceUserEntryMutex);

        int userIndex = IsAppletResourceUserIdRegistered(aruid);
        bool isUnregisterSucceeded = false;

        if (userIndex >= 0)
        {
            m_AppletResourceUserEntry[userIndex].isRegistered = false;
            isUnregisterSucceeded = true;
        }

        return isUnregisterSucceeded;
    }

    bool AppletManager::SetInFocusedAppletResourceUser(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
    {
        NN_BTM_SCOPED_LOCK(m_AppletResourceUserEntryMutex);

        int userIndex = IsAppletResourceUserIdRegistered(aruid);
        bool isSetSucceeded = false;

        if (userIndex >= 0)
        {
            m_InFocusedAppletResourceUser.aruid    = m_AppletResourceUserEntry[userIndex].id.aruid;
            m_InFocusedAppletResourceUser.appletId = m_AppletResourceUserEntry[userIndex].id.appletId;

            isSetSucceeded = true;
        }

        return isSetSucceeded;
    }

    AppletManager::AppletResourceUser AppletManager::GetInFocusedAppletResourceUser() NN_NOEXCEPT
    {
        NN_BTM_SCOPED_LOCK(m_AppletResourceUserEntryMutex);

        return m_InFocusedAppletResourceUser;
    }

    int AppletManager::IsAppletResourceUserIdRegistered(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
    {
        NN_BTM_SCOPED_LOCK(m_AppletResourceUserEntryMutex);

        int userIndex = -1;

        for (int i = 0; i < NN_ARRAY_SIZE(m_AppletResourceUserEntry); ++i)
        {
            if (m_AppletResourceUserEntry[i].isRegistered && m_AppletResourceUserEntry[i].id.aruid == aruid)
            {
                userIndex = i;
                break;
            }
        }

        return userIndex;
    }
}}  // namespace nn::btm
