﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "boot2_SigloBoot.h"

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/fs/fs_Application.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Mount.h>
#include <nn/htc.h>
#include <nn/htc/htc_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_LocationResolver.h>
#include <nn/nn_SdkText.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <cstring>
#include <algorithm>
#include <nn/pm/pm_ShellApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringView.h>
#include <nn/nn_Abort.h>

#if !defined(NN_SIGLO_BOOT_LIST_FILE)
#error "NN_SIGLO_BOOT_LIST_FILE が定義されていません。"
#endif

namespace nn { namespace boot2 {

    namespace
    {
        const char SigloBootListPath[] = "C:/siglo_boot/list.txt";
        const char SigloBootListPathEnvironmentVariableName[] = "SIGLO_BOOT_LIST_PATH";
        const char SigloBootListDirectoryEnvironmentVariableName[] = "SIGLO_BOOT_LIST_DIR";

        const int LaunchFlags = pm::LaunchProgramFlags_NotifyException
                              | pm::LaunchProgramFlags_DisableAslr;

        char                        g_ListBuffer[256 * 1024];
        nn::lr::LocationResolver    g_LocationResolver;

        void InitializeLibraries()
        {
            nn::Result ret;

            // htc 初期化
            htc::Initialize();

            // lr 初期化
            nn::lr::Initialize();
            NN_ABORT_UNLESS(nn::lr::OpenLocationResolver(&g_LocationResolver, ncm::StorageId::Host).IsSuccess());
        }

        void FinalizeLibraries()
        {
            lr::Finalize();
            htc::Finalize();
        }

        // htc 経由で環境変数を読み出す
        Result ReadEnvironmentVariable(char* pOutBuffer, size_t bufferSize, const char* envName)
        {
            nn::Result ret;

            for(;;)
            {
                size_t valueSize;
                ret = htc::GetEnvironmentVariable(&valueSize, pOutBuffer, bufferSize, envName);
                if( ! (ret <= htc::ResultConnectionFailure()) )
                {
                    break;
                }

                NN_SDK_LOG(NN_TEXT("[siglo_boot] Waiting for connection from TargetManager...\n"));
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            if( ret.IsSuccess() )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Use environment variable %s.\n"), envName);
                NN_SDK_LOG(NN_TEXT("[siglo_boot] %s = '%s'\n"), envName, pOutBuffer);
            }
            else
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Failed to get environment variable %s.\n"), envName);
            }

            return ret;
        }

        struct ContentMetaBuffer
        {
            std::unique_ptr<Bit8> bufferManager;
            const Bit8* buffer;
            size_t size;
        };

        const int MountNameLength = fs::MountNameLengthMax + 1;

        void MakeUniqueMountName(char* mountName, size_t bufferSize) NN_NOEXCEPT
        {
            static std::atomic_uint s_Count;
            util::SNPrintf(mountName, bufferSize, "aput%08x", s_Count.fetch_add(1));
        }

        bool IsMatchExtension(const char* str, const char* extension) NN_NOEXCEPT
        {
            return (util::string_view(str).substr(strlen(str) - strlen(extension))
                == util::string_view(extension));
        }

        Result SearchContentMeta(bool* outValue, lr::Path* outPath, const char* pPath) NN_NOEXCEPT
        {
            const char* contentMetaExtension = ".cnmt.nca";

            char mountName[MountNameLength];
            MakeUniqueMountName(mountName, MountNameLength);

            NN_RESULT_DO(fs::MountApplicationPackage(mountName, pPath));
            NN_UTIL_SCOPE_EXIT{ fs::Unmount(mountName); };

            char rootDirectoryPath[MountNameLength + 2];
            util::SNPrintf(rootDirectoryPath, NN_ARRAY_SIZE(rootDirectoryPath), "%s:/", mountName);
            fs::DirectoryHandle directory;
            NN_RESULT_DO(fs::OpenDirectory(&directory, rootDirectoryPath, fs::OpenDirectoryMode_File));
            NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

            while(NN_STATIC_CONDITION(true))
            {
                fs::DirectoryEntry entry;
                int64_t outRead;
                NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directory, 1));
                if (outRead == 0)
                {
                    break;
                }

                if(IsMatchExtension(entry.name, contentMetaExtension))
                {
                    auto length = util::SNPrintf(outPath->string, sizeof(outPath->string), "%s/%s", pPath, entry.name);
                    NN_ABORT_UNLESS(length + 1 < sizeof(outPath->string));
                    if (entry.directoryEntryType == fs::DirectoryEntryType_Directory)
                    {
                        outPath->string[length] = '/';
                        outPath->string[length + 1] = '\0';
                    }
                    *outValue = true;
                    NN_RESULT_SUCCESS;
                }
            }

            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        Result ReadContentMeta(::std::unique_ptr<Bit8>* outBuffer, size_t* outValue, const char* pPath) NN_NOEXCEPT
        {
            char mountMetaName[MountNameLength];
            MakeUniqueMountName(mountMetaName, MountNameLength);

            NN_RESULT_DO(fs::MountContent(mountMetaName, pPath, fs::ContentType::ContentType_Meta));
            NN_UTIL_SCOPE_EXIT{ fs::Unmount(mountMetaName); };

            char rootDirectoryPath[MountNameLength + 2];
            util::SNPrintf(rootDirectoryPath, NN_ARRAY_SIZE(rootDirectoryPath), "%s:/", mountMetaName);
            fs::DirectoryHandle directory;
            NN_RESULT_DO(fs::OpenDirectory(&directory, rootDirectoryPath, fs::OpenDirectoryMode_File));
            NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(directory); };

            fs::DirectoryEntry entry;
            int64_t outRead;
            NN_RESULT_DO(fs::ReadDirectory(&outRead, &entry, directory, 1));
            NN_RESULT_THROW_UNLESS(outRead == 1, fs::ResultPathNotFound());

            fs::FileHandle file;
            lr::Path path;
            util::SNPrintf(path.string, sizeof(path.string), "%s%s", rootDirectoryPath, entry.name);
            // Open .cnmt file
            NN_RESULT_DO(fs::OpenFile(&file, path.string, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
            size_t metaSize = static_cast<size_t>(fileSize);
            ::std::unique_ptr<Bit8> meta(new Bit8[metaSize]);
            NN_ABORT_UNLESS_NOT_NULL(meta);

            size_t readSize;
            NN_RESULT_DO(fs::ReadFile(&readSize, file, 0, meta.get(), metaSize));

            *outBuffer = ::std::move(meta);
            *outValue = readSize;

            NN_RESULT_SUCCESS;
        }

        void GetContentMetaBuffer(ContentMetaBuffer* outBuffer, const char* pPath) NN_NOEXCEPT
        {
            bool hasContentMeta;
            lr::Path path;
            NN_ABORT_UNLESS_RESULT_SUCCESS(SearchContentMeta(&hasContentMeta, &path, pPath));

            NN_ABORT_UNLESS(hasContentMeta);
            NN_ABORT_UNLESS_RESULT_SUCCESS(ReadContentMeta(&outBuffer->bufferManager, &outBuffer->size, path.string));
            outBuffer->buffer = outBuffer->bufferManager.get();
        }

        void GetNcaPathFromNsp(lr::Path* outPath, ContentMetaBuffer* pMeta) NN_NOEXCEPT
        {
            auto metaReader = ncm::PackagedContentMetaReader(pMeta->buffer, pMeta->size);

            for(int i = 0; i < metaReader.CountContent(); i++)
            {
                auto contentInfo = metaReader.GetContentInfo(i);

                if (contentInfo->GetType() == ncm::ContentType::Program)
                {
                    lr::Path path;
                    util::SNPrintf(path.string, sizeof(path.string), "%s/%s.nca", outPath->string, ncm::GetContentIdString(contentInfo->GetId()).data);
                    util::SNPrintf(outPath->string, sizeof(outPath->string), "%s", path.string);

                    return;
                }
            }

            NN_ABORT();
        }

        void ProcessPathLine(uint64_t programIdValue, const char* pLine, const char* pLineEnd)
        {
            size_t lineLength = pLineEnd - pLine;

            // TORIAEZU: "@Host:/" をパスの先頭に付与
            const char HostMountName[] = "@Host:/";

            nn::lr::Path path;

            if( std::strlen(HostMountName) + lineLength + 1 < sizeof(path.string) )
            {
                nn::ncm::ProgramId programId = { programIdValue };

                std::memset(path.string, 0, sizeof(path.string));
                std::memcpy(path.string, HostMountName, std::strlen(HostMountName));
                std::memcpy(path.string + std::strlen(HostMountName), pLine, lineLength);

                // nsp の場合は末尾に nca のパスを付与
                if( IsMatchExtension(path.string, ".nsp") )
                {
                    ContentMetaBuffer meta;
                    GetContentMetaBuffer(&meta, path.string);

                    GetNcaPathFromNsp(&path, &meta);
                }

                g_LocationResolver.RedirectProgramPath(programId, path);

                NN_SDK_LOG(NN_TEXT("[siglo_boot] run '%s'\n"), path.string);

                nn::os::ProcessId processId;
                nn::Result result = nn::pm::LaunchProgram(&processId, ncm::MakeProgramLocation(ncm::StorageId::Host, programId), LaunchFlags);
                if( result.IsFailure() )
                {
                    NN_SDK_LOG(NN_TEXT("[siglo_boot] Failed to run '%s'(0x%08x). Skip it.\n"), path.string, result.GetInnerValueForDebug());
                }
            }
            else
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Skip '%.256s ...' because of too long path.\n"), pLine);
            }
        }

        void ProcessRelativePathLine(uint64_t programIdValue, const char* pLine, const char* pLineEnd, const char* pListPath)
        {
            size_t lineLength = pLineEnd - pLine;

            // TORIAEZU: "@Host:/" をパスの先頭に付与
            const char HostMountName[] = "@Host:/";

            nn::lr::Path path;

            if( ! (std::strlen(HostMountName) + std::strlen(pListPath) + 1 < sizeof(path.string)) )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Skip '%.256s ...' because of too long list path.\n"), pLine);
            }

            std::memset(path.string, 0, sizeof(path.string));
            std::memcpy(path.string, HostMountName, std::strlen(HostMountName));
            std::memcpy(path.string + std::strlen(HostMountName), pListPath, std::strlen(pListPath));

            auto pLastSlash     = std::strrchr(path.string, '/');
            auto pLastBaskSlash = std::strrchr(path.string, '\\');

            char* pLastPos = std::max(pLastSlash, pLastBaskSlash);

            // HostMountName に / があるので NULL にはならない
            NN_SDK_ASSERT( pLastPos != NULL );

            auto pStringEnd = pLastPos + 1;
            *pStringEnd = '\0';

            if( ! ((pStringEnd - path.string) + lineLength + 1 < sizeof(path.string)) )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Skip '%.256s ...' because of too long path.\n"), pLine);
            }

            std::memcpy(pStringEnd, pLine, lineLength);
            *(pStringEnd + lineLength) = '\0';

            nn::ncm::ProgramId programId = { programIdValue };
            g_LocationResolver.RedirectProgramPath(programId, path);

            NN_SDK_LOG(NN_TEXT("[siglo_boot] run '%s'\n"), path.string);

            nn::os::ProcessId processId;
            nn::Result result = nn::pm::LaunchProgram(&processId, ncm::MakeProgramLocation(ncm::StorageId::Host, programId), LaunchFlags);
            if( result.IsFailure() )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Failed to run '%s'(0x%08x). Skip it.\n"), path.string, result.GetInnerValueForDebug());
            }
        }

        uint64_t ParseHexString(const char* pBegin, const char* pEnd)
        {
            uint64_t v = 0;
            auto pPos = pBegin;

            while( pPos < pEnd )
            {
                auto c = *pPos++;

                if( ('a' <= c) && (c <= 'f') )
                {
                    v <<= 4;
                    v |= (c - 'a' + 10);
                }
                else if( ('A' <= c) && (c <= 'F') )
                {
                    v <<= 4;
                    v |= (c - 'A' + 10);
                }
                else if( ('0' <= c) && (c <= '9') )
                {
                    v <<= 4;
                    v |= (c - '0');
                }
                else
                {
                    break;
                }
            }

            return v;
        }

        void ProcessNLine(const char* pLine, const char* pLineEnd)
        {
            auto pPos = pLine + 2;

            // 先頭の空白類を読み飛ばす
            while( (pPos < pLineEnd) && (0 <= *pPos) && (*pPos <= ' ') )
            {
                pPos++;
            }

            // 0x なら読み飛ばす
            if( (pPos + 2 <= pLineEnd) && (std::memcmp(pPos, "0x", 2) == 0) )
            {
                pPos += 2;
            }

            // 16 進文字列をパース
            uint64_t id = ParseHexString(pPos, pLineEnd);

            os::ProcessId processId;
            ncm::ProgramId programId = { id };
            auto location = ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, programId);
            nn::Result result = nn::pm::LaunchProgram(&processId, location, LaunchFlags);
            if( result.IsFailure() )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Failed to run '%016llx'(0x%08x). Skip it.\n"), id, result.GetInnerValueForDebug());
            }
        }

        void ProcessSigloBoot()
        {
            nn::Result ret;
            const size_t bufferSize = 256;
            char envValue[bufferSize];
            char listPath[bufferSize];

            NN_SDK_LOG(NN_TEXT("[siglo_boot] begin\n"));

            InitializeLibraries();

            if (ReadEnvironmentVariable(envValue, sizeof(envValue), SigloBootListDirectoryEnvironmentVariableName).IsSuccess())
            {
                nn::util::SNPrintf(listPath, sizeof(listPath), "%s\\%s", envValue, NN_SIGLO_BOOT_LIST_FILE);
            }
            else if (ReadEnvironmentVariable(envValue, sizeof(envValue), SigloBootListPathEnvironmentVariableName).IsSuccess())
            {
                std::strncpy(listPath, envValue, sizeof(listPath));
            }
            else {
                std::strncpy(listPath, SigloBootListPath, sizeof(listPath));
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Use default path '%s'.\n"), SigloBootListPath);
            }

            // ホストをマウント
            ret = nn::fs::MountHostRoot();
            if( ret.IsFailure() )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] fs::MountHostRoot failed.\n"));
                return;
            }

            // リストの取得
            nn::fs::FileHandle f;
            ret = nn::fs::OpenFile(&f, listPath, nn::fs::OpenMode_Read);

            if( ret.IsFailure() )
            {
                NN_SDK_LOG(NN_TEXT("[siglo_boot] Not found '%s'. Skip it.\n"), listPath);
            }
            else
            {
                // リストの読み取り
                size_t fileSize;
                ret = nn::fs::ReadFile(&fileSize, f, 0, g_ListBuffer, sizeof(g_ListBuffer) - 1);
                if( ret.IsFailure() )
                {
                    NN_SDK_LOG(NN_TEXT("[siglo_boot] Failed to read '%s'. Skip it.\n"), listPath);
                    fileSize = 0;
                }
                nn::fs::CloseFile(f);

                g_ListBuffer[fileSize] = '\0';


                // リストのパース
                const char* pBegin = g_ListBuffer;
                uint64_t programIdValue = 0xF000000200000000ull;

                // UTF-8 BOM をスキップ
                if (pBegin[0] == static_cast<char>(0xef) &&
                    pBegin[1] == static_cast<char>(0xbb) &&
                    pBegin[2] == static_cast<char>(0xbf))
                {
                    pBegin += 3;
                }

                for(;;)
                {
                    // 次の行末を探す
                    const char* pFound = std::strchr(pBegin, '\n');
                    if( pFound == NULL )
                    {
                        pFound = strchr(pBegin, '\0');
                    }

                    // 改行が \r\n なら行末の位置を調整する
                    const char* pEnd = pFound;
                    if( (pFound != g_ListBuffer) && (*(pFound - 1) == '\r') )
                    {
                        pEnd = pFound - 1;
                    }

                    // 行末が空白なら行末の位置を調整する
                    while( (pBegin < pEnd) && ('\0' <= *(pEnd - 1)) && (*(pEnd - 1) <= ' ') )
                    {
                        pEnd--;
                    }

                    // 短すぎる行は無視
                    // '#' から始まる行はコメントアウト
                    // それ以外の行をパスとして扱う
                    if( pEnd - pBegin >= 2 && pBegin[0] != '#' )
                    {
                        // /^[a-zA-Z]:/ ならパスとして起動
                        if( (  ( ('A' <= *pBegin) && (*pBegin <= 'Z') )
                            || ( ('a' <= *pBegin) && (*pBegin <= 'z') ) )
                         && (*(pBegin + 1) == ':') )
                        {
                            ProcessPathLine(programIdValue, pBegin, pEnd);
                        }
                        // /^n / ならプログラム ID として内蔵ストレージから起動
                        else if( std::memcmp(pBegin, "n ", 2) == 0 )
                        {
                            ProcessNLine(pBegin, pEnd);
                        }
                        // /\.nca$/ なら相対パスとして起動
                        else if( pEnd - pBegin >= 4 )
                        {
                            if( std::memcmp(pEnd - 4, ".nca", 4) == 0 )
                            {
                                ProcessRelativePathLine(programIdValue, pBegin, pEnd, listPath);
                            }
                        }
                    }

                    if( *pFound == '\0' )
                    {
                        break;
                    }

                    pBegin = pFound + 1;
                    programIdValue++;
                }
            }

            FinalizeLibraries();

            NN_SDK_LOG(NN_TEXT("[siglo_boot] end.\n"));
        }

    }   // anonymous namespace


    void SigloBoot()
    {
        ProcessSigloBoot();
    }

}}  // namespace nn::boot2

