﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// Workaround: gcc の cstdio で snprintf が定義されていないので C ヘッダを使う
#include <stdio.h> // NOLINT(others/c_header)

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/util/util_ScopeExit.h>

#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_Mount.h>

#include "boot_NandCheck.h"

namespace {

void PrintEntries(const char* directoryPath, uint64_t& totalSize)
{
    NN_SDK_LOG("%s:\n", directoryPath);

    // 最初にディレクトリに含まれるファイル一覧を表示します
    {
        nn::fs::DirectoryHandle directoryHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directoryHandle);
        };

        int64_t entryCount = 1;
        nn::fs::DirectoryEntry entry;

        while (entryCount)
        {
            // エラーを返すことは無いのでハンドリング不要です。
            nn::fs::ReadDirectory(&entryCount, &entry, directoryHandle, 1);

            if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
            {
                //NN_SDK_LOG("%12lld  %s\n", entry.fileSize, entry.name);
                NN_SDK_LOG("    %-50s %12lld\n", entry.name, entry.fileSize);
                totalSize += entry.fileSize;
            }
        }
    }

    NN_SDK_LOG("\n");

    // 次に子ディレクトリに対して再帰的に走査します
    {
        nn::fs::DirectoryHandle directoryHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directoryHandle);
        };

        int64_t entryCount = 1;
        nn::fs::DirectoryEntry entry;

        while (entryCount)
        {
            // エラーを返すことは無いのでハンドリング不要です。
            nn::fs::ReadDirectory(&entryCount, &entry, directoryHandle, 1);

            if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                const int MaxDirectoryNameLength = 300;

                char childDirectoryPath[MaxDirectoryNameLength];
                snprintf(childDirectoryPath, sizeof(childDirectoryPath), "%s%s/", directoryPath, entry.name);

                // 子ディレクトリに対して再帰的に関数を呼び出す
                PrintEntries(childDirectoryPath, totalSize);
            }
        }
    }
}

}

namespace nn { namespace boot {

void PrintFilesOnSystemPartition()
{
    static const char SystemMountName[] = "system";
    uint64_t totalSize = 0;

    //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountBis(SystemMountName, nn::fs::BisPartitionId::SystemProperPartition));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountBis(SystemMountName, nn::fs::BisPartitionId::System));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(SystemMountName);
    };

    // system 以下の全エントリをログに表示
    PrintEntries("system:/", totalSize);

    // teamcity 用の表示
    NN_SDK_LOG("##teamcity[buildStatisticValue key='System partition usage' value='%lld']\n", totalSize);
}

}} // namespace nn::boot
