﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/pinmux/driver/pinmux.h>
#include <nn/pinmux/driver/pinmux_InitialConfig.h>

#include <nn/gpio/driver/gpio_InitialConfig.h>

#if defined(ENABLE_SECONDARY_UART_LOG)
#include <nne/pinmux/pinmux.h>
#endif

#include <nn/pm/pm_ShellApi.h>

#include <nn/fs/fs_MemoryManagement.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "boot_FanEnable.h"
#include "boot_ChangeVoltage.h"
#include "boot_BootImage.h"
#include "boot_BootReason.h"
#include "boot_BatteryChargeChecker.h"
#include "boot_ClockChecker.h"
#include "boot_ClockOutControl.h"
#include "boot_SplAccessor.h"
#include "boot_SplashScreen.h"
#include "boot_Display.h"
#include "boot_VolumeButton.h"
#if defined(ENABLE_NAND_USAGE_CHECK_UART_LOG)
#include "boot_NandCheck.h"
#endif

namespace {

// ログを出して見る限り、最大で 1640byte 利用していた
// そこに少し余裕を持たせて 4KB に
uint8_t g_HeapMemory[4 << 10];
nn::lmem::HeapHandle g_HeapHandle;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

}

extern "C" void nninitStartup()
{
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}

//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(boot, Main));

    // GPIO の電圧の変更(1.8V へ)
    nn::boot::ChangeGpioVoltageTo1_8v();

    // GPIO ピンの初期一括設定 (Pinmux より先に設定すること)
    nn::gpio::driver::SetInitialGpioConfig();

    g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), nn::lmem::CreationOption_NoOption);
    nn::fs::SetAllocator(Allocate, Deallocate);

    // 想定外のクロックで再起動することがあるので起動要因のクリアより前におこなう
    nn::boot::CheckClock();

    nn::boot::DetectBootReason();

    // 電池残量チェック
    // XXX: 電池僅少時の画面表示で pinmux をいじる場合があるため、やむを得ず pinmux の初期設定前に実行する
    auto hardwareType = nn::boot::GetHardwareType();
    if (hardwareType != nn::spl::HardwareType_Copper)
    {
        // コーポレートロゴなどの表示
        nn::boot::ShowSplashScreen();

        nn::boot::CheckBatteryCharge();
    }

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#if defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
    // Vol up/down のどちらかが押されていたらメンテナンスモードの入り損ないと判断してシャットダウン
    if (nn::boot::IsAnyVolumeButtonPressed())
    {
        nn::boot::ShutdownSystem();
    }

    // safemode プロセスで絵を出すための準備
    nn::boot::SetupDisplayForSafeMode();
#endif
#endif

    // pinmux の初期一括設定
    nn::pinmux::driver::Initialize();
    nn::pinmux::driver::SetInitialConfig();
    nn::pinmux::driver::SetInitialDrivePadConfig();
    nn::pinmux::driver::Finalize();

    // Wake Pin の一括設定
    nn::gpio::driver::SetInitialWakePinConfig();

    // EDEV でログを出す際の実装分岐
#if defined(ENABLE_SECONDARY_UART_LOG)
#if defined NN_BUILD_CONFIG_HARDWARE_NX

    nn::pinmux::driver::PinmuxSession session;

    nn::pinmux::driver::Initialize();
    nn::pinmux::driver::OpenSession(&session, nn::pinmux::AssignablePinGroupName_ExtConUTx);
    nn::pinmux::driver::SetPinAssignment(&session, nn::pinmux::PinAssignment_ExtConTxUart);
    nn::pinmux::driver::CloseSession(&session);
    nn::pinmux::driver::Finalize();

    NN_SDK_LOG("[boot] Enable Secondary Uart Log.\n");
#else
#error ENABLE_SECONDARY_UART_LOG is not supported
#endif // NN_BUILD_CONFIG_HARDWARE_NX
#endif // ENABLE_SECONDARY_UART_LOG

    if (hardwareType != nn::spl::HardwareType_Copper)
    {
        // 外部デバイスへのクロック供給の開始　(NX では Audio CODEC 向け)
        nn::boot::SetInitialClockOutSetting();
    }

    // FAN の電源 ON
    nn::boot::SetFanPowerEnabled();

    // BootImages の更新
    nn::boot::RepairBootImagesAndRebootIfNeeded(hardwareType);

    // pm に boot の完了を通知
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pm::InitializeForShell());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pm::NotifyBootFinished());
    //NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pm::FinalizeForShell()); // not implemented

#if defined(ENABLE_NAND_USAGE_CHECK_UART_LOG)
    // ファイルサイズ計測
    nn::boot::PrintFilesOnSystemPartition();
#endif

#if defined(NN_BOOT_BUILD_TYPE_SAFE_MODE)
    NN_SDK_LOG("[boot] safe boot finished.\n");
#endif

    NN_SDK_LOG("[boot] boot process finished.\n");
}

//-----------------------------------------------------------------------------
