﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>
#include <nn/dd.h>

namespace nn {
namespace boot {

namespace
{
const nn::dd::PhysicalAddress   PmcPhysicalAddress   = 0x07000e400ull;

const uintptr_t OffsetOfApbdevPmcPwrDet0    = 0x48;
const uintptr_t OffsetOfApbdevPmcPwrDetVal0 = 0xe4;

// 3.3V / 1.8 V 切り替え可能な IO ブロック
// NX ではすべて 1.8 V で使用する
const int SpiHvRailAoIoBit   = 23;
const int GpioRailAoIoBit    = 21;
const int AudioHvRailAoIoBit = 18;
const int Sdmmc3RailAoIoBit  = 13;
//const int Sdmmc1RailAoIoBit  = 12; // SDMMC1 は動的制御をするのでここでは触らない

const Bit32 RailIoMask = 1 << SpiHvRailAoIoBit |
                         1 << GpioRailAoIoBit  |
                         1 << AudioHvRailAoIoBit |
                         1 << Sdmmc3RailAoIoBit;


// 設定した後の待ち時間
const nn::TimeSpan WaitTime = nn::TimeSpan::FromMicroSeconds(100);

inline void WriteMasked32( dd::PhysicalAddress address, Bit32 value, Bit32 mask)
{
    dd::ReadModifyWriteIoRegister(address, value, mask);
}

}

// 3.3 V の Capability を持つ GPIO の電圧を 1.8V に変更します。
void ChangeGpioVoltageTo1_8v() NN_NOEXCEPT
{
    dd::PhysicalAddress apbDevPmcPwrDetAddress      = PmcPhysicalAddress + OffsetOfApbdevPmcPwrDet0;
    dd::PhysicalAddress apbDevPmcPwrDetValAddress   = PmcPhysicalAddress + OffsetOfApbdevPmcPwrDetVal0;

    // すべて 1　をセット (ソフトウェアでコントロール可にする)
    WriteMasked32(apbDevPmcPwrDetAddress,
                                          1 << SpiHvRailAoIoBit |
                                          1 << GpioRailAoIoBit |
                                          1 << AudioHvRailAoIoBit |
                                          1 << Sdmmc3RailAoIoBit,
                                          RailIoMask);

    // すべて 0 をセット (1.8 V に切り替え)
    WriteMasked32(apbDevPmcPwrDetValAddress,
                                          0 << SpiHvRailAoIoBit |
                                          0 << GpioRailAoIoBit |
                                          0 << AudioHvRailAoIoBit |
                                          0 << Sdmmc3RailAoIoBit,
                                          RailIoMask);

    nn::os::SleepThread(WaitTime);
}

}}  // namespace nn::boot
