﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "bluetooth_InternalTypes.h"


namespace nn {
namespace bluetooth {


/**
 * @brief Represents the standard Bluetooth interface.
 */
class BluetoothLeInterfaceC
{
    NN_DISALLOW_COPY(BluetoothLeInterfaceC);
    NN_DISALLOW_MOVE(BluetoothLeInterfaceC);

public:
    BluetoothLeInterfaceC()                NN_NOEXCEPT : m_BluetoothLeInterface() {}

    BluetoothLeInterface& GetBluetoothLeInterface() NN_NOEXCEPT
    {
        return m_BluetoothLeInterface;
    };

    /*
     * Opens the interface and provides the callback routines
     * to the implemenation of this interface.
     */
    static BluetoothLeStatus InitializeBle(BluetoothLeCallbacks* callbacks);

    /* Initialize bluetooth low energy services */
    static BluetoothLeStatus EnableBle();

    /* Disables bluetooth low energy */
    static BluetoothLeStatus DisableBle();

    /* Finalize bluetooth low energy */
    static void CleanupBle(void);

    /* Set BLE discoverable/connectable */
    static BluetoothLeStatus SetLeVisibility(bool discoverable, bool connectable);

    /* Set BLE connection parameters */
    static BluetoothLeStatus SetLeConnectionParameter(LeConnectionParams connectionParams);

    /* Set BLE default connection parameters */
    static BluetoothLeStatus SetLeDefaultConnectionParameter(LeConnectionParams connectionParams);

    /* Set BLE advertisement data */
    static BluetoothLeStatus SetLeAdvertiseData(LeAdvertiseData advertiseData);

    /* Set BLE advertisement parameters */
    static BluetoothLeStatus SetLeAdvertiseParameter(Btbdaddr *pBdAddr, uint16_t minAdvInt, uint16_t maxAdvInt);

    /* Start BLE discovery */
    static BluetoothLeStatus StartLeScan();

    /* Cancel discovery */
    static BluetoothLeStatus StopLeScan();

    /* Set BLE Scan parameters */
    static BluetoothLeStatus SetLeScanParameter(uint16_t scanInterval, uint16_t scanWindow);

    /* Add Scan Filter Condition */
    static BluetoothLeStatus AddLeScanFilterCondition(const BleAdvertiseFilter* filter);

    /* Delete Scan Filter Condition */
    static BluetoothLeStatus DeleteLeScanFilterCondition(const BleAdvertiseFilter* filter);

    /* Delete a specific Scan Filter */
    static BluetoothLeStatus DeleteLeScanFilter(uint8_t filterIndex);

    /* Clear all the BLE scan filters */
    static BluetoothLeStatus ClearLeScanFilters();

    /* Enable BLE Scan Filter */
    static BluetoothLeStatus EnableLeScanFilter(bool enable);

    /* Register BLE GATT client */
    static BluetoothLeStatus RegisterLeClient(GattAttributeUuid uuid);

    /* Unregister BLE GATT client */
    static BluetoothLeStatus UnregisterLeClient(uint8_t clientIf);

    /* Register BLE GATT client */
    static BluetoothLeStatus UnregisterLeClientAll();

    /* Connect to a BLE GATT server */
    static BluetoothLeStatus LeClientConnect(uint8_t clientIf, Btbdaddr *pBdAddr, bool isDirect);

    /* Cancel Connection request tot a BLE GATT server */
    static BluetoothLeStatus LeClientCancelConnection(uint8_t clientIf, Btbdaddr* pBdAddr, bool isDirect);

    /* Disconnect from a BLE GATT server */
    static BluetoothLeStatus LeClientDisconnect(uint32_t connId);

    /* Get the attributes of a device */
    static BluetoothLeStatus LeClientGetAttributes(uint32_t connId, Btbdaddr *pBdAddr);

    /* Discover a BLE GATT service */
    static BluetoothLeStatus LeClientDiscoverService(uint32_t connId, GattAttributeUuid filterUuid);

    /* Read a BLE GATT characteristic from a service */
    static BluetoothLeStatus LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType);

    /* Read a BLE GATT descriptor from a service */
    static BluetoothLeStatus LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType);

    /* Write a value to a BLE GATT service characteristic */
    static BluetoothLeStatus LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse);

    /* Write a value to a BLE GATT service descriptor */
    static BluetoothLeStatus LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType);

    /* Register to receive notifications for a given characteristic from a service */
    static BluetoothLeStatus LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId);

    /* Deregister a previous request for notifications */
    static BluetoothLeStatus LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId);

    /* Configure MTU */
    static BluetoothLeStatus LeClientConfigureMtu(uint32_t connId, uint16_t mtu);

    /* Register BLE GATT server */
    static BluetoothLeStatus RegisterLeServer(GattAttributeUuid uuid);

    /* Deregister BLE GATT server */
    static BluetoothLeStatus UnregisterLeServer(uint8_t serverIf);

    /* Connect to a BLE GATT client */
    static BluetoothLeStatus LeServerConnect(uint8_t serverIf, Btbdaddr *pBdAddr, bool isDirect);

    /* Disconnect from a BLE GATT client */
    static BluetoothLeStatus LeServerDisconnect(uint8_t serverIf, Btbdaddr *pBdAddr);

    /* Initialize a BLE GATT server service */
    static BluetoothLeStatus CreateLeService(uint8_t serverIf, GattAttributeUuid uuid, uint8_t attrNum, bool isPrimaryService);

    /* Start a local BLE GATT server service */
    static BluetoothLeStatus StartLeService(uint8_t serverIf, GattAttributeUuid uuid);

    /* Add a BLE GATT characteristic to a service */
    static BluetoothLeStatus AddLeCharacteristic(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties);

    /* Add a BLE GATT descriptor to a service */
    static BluetoothLeStatus AddLeDescriptor(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid descUuid, uint16_t permissions);

    /* Set attribute value for response */
    static BluetoothLeStatus SetLeResponse(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const uint8_t* pData, uint16_t len);

    /* Send indication/notification to client */
    static BluetoothLeStatus LeSendIndication(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t len, const uint8_t* pData, bool confirm);

    /* Get the first characteristic from a service */
    static BluetoothLeStatus LeGetFirstCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter);

    /* Get the subsequent characteristic from a service */
    static BluetoothLeStatus LeGetNextCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter);

    /* Find the first descriptor of the characteristic on the given server from cache */
    static BluetoothLeStatus LeGetFirstDescriptor(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter);

    /* Find the subsequent descriptor of the characteristic on the given server from cache */
    static BluetoothLeStatus LeGetNextDescriptor(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter);

    /* Register GATT data path to be informed via BLE Core Event */
    static BluetoothLeStatus RegisterLeCoreDataPath(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE Core Event */
    static BluetoothLeStatus UnregisterLeCoreDataPath(const GattAttributeUuid& uuid);

    /* Register GATT data path to be informed via BLE HID Event */
    static BluetoothLeStatus RegisterLeHidDataPath(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE HID Event */
    static BluetoothLeStatus UnregisterLeHidDataPath(const GattAttributeUuid& uuid);

    /* Register GATT data path to be informed via BLE Event */
    static BluetoothLeStatus RegisterLeDataPath(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE Event */
    static BluetoothLeStatus UnregisterLeDataPath(const GattAttributeUuid& uuid);

private:
    BluetoothLeInterface m_BluetoothLeInterface;
};

/*******************************************************************************
 **
 ** Function         LeConvertBsaStatus
 **
 ** Description      Convert result from tBSA_STATUS to BluetoothLeStatus
 **
 ** Parameters       resultFromBsa: tBSA_STATUS to be converted
 **
 ** Returns          BluetoothLeStatus result
 **
 *******************************************************************************/
BluetoothLeStatus LeConvertBsaStatus(tBSA_STATUS resultFromBsa);

} // bluetooth
} // nn


