﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------------*
  Copyright (C) 2008 The Android Open Source Project
  Modified by Nintendo

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/bluetooth/bluetooth_TypesHal.h>
#include <nn/bluetooth/bluetooth_Types.h>
#ifdef __cplusplus
#include <nn/settings/system/settings_BluetoothDevices.h>
#endif


// For tracing the interface to the upper layer (btm/hid, or the sample client)
#if 1
#    define BTHAL_IF_DEBUG(format, ...) (void*)0
#else
#    define BTHAL_IF_DEBUG(format, ...) NN_SDK_LOG("BTHAL_IF: %s: " format "\n", __func__, ##__VA_ARGS__)
#endif

// For tracing the interface to the lower layer (BSA)
#if 1
#    define BTHAL_BSA_DEBUG(format, ...) (void*)0
#else
#    define BTHAL_BSA_DEBUG(format, ...) NN_SDK_LOG("BTHAL_SA: %s: " format "\n", __func__, ##__VA_ARGS__)
#endif

#if 1
#    define BTHAL_IF_BLE_DEBUG(format, ...) (void*)0
#else
#    define BTHAL_IF_BLE_DEBUG(format, ...) NN_SDK_LOG("BTHAL_IF_BLE %s: " format "\n", __func__, ##__VA_ARGS__)
#endif

// For tracing callback events from BSA
//#define BTHAL_CB_DEBUG




#ifdef __cplusplus
using namespace nn::bluetooth;
extern "C"
{
#endif // ifdef __cplusplus

#define BT_DEFAULT_PIN_CODE                 "0000"

enum {BT_DEFAULT_PIN_LEN=4};

/* The Bluetooth Hardware Module ID */
#define BT_HARDWARE_MODULE_ID               "bluetooth"
#define BT_STACK_MODULE_ID                  "bluetooth"
#define BT_STACK_TEST_MODULE_ID             "bluetooth_test"

/* Bluetooth profile interface IDs */
#define BT_PROFILE_HANDSFREE_ID             "handsfree"
#define BT_PROFILE_HANDSFREE_CLIENT_ID      "handsfree_client"
#define BT_PROFILE_ADVANCED_AUDIO_ID        "a2dp"
#define BT_PROFILE_ADVANCED_AUDIO_SINK_ID   "a2dp_sink"
#define BT_PROFILE_HEALTH_ID                "health"
#define BT_PROFILE_SOCKETS_ID               "socket"
#define BT_PROFILE_HIDHOST_ID               "hidhost"
#define BT_PROFILE_PAN_ID                   "pan"
#define BT_PROFILE_MAP_CLIENT_ID            "map_client"
#define BT_PROFILE_GATT_ID                  "gatt"
#define BT_PROFILE_AV_RC_ID                 "avrcp"
#define BT_PROFILE_AV_RC_CTRL_ID            "avrcp_ctrl"
#define BT_PROFILE_EXTENSION                "extension"
#define BT_PROFILE_LOW_ENERGY               "low_energy"
enum {BT_MAX_NUM_UUIDS=32};

/* Bluetooth Interface callbacks */

/* Bluetooth Enable/Disable Callback. */
typedef void (*AdapterStateChangedCallbackFuncType)(BluetoothState state);

/* GET/SET Adapter Properties callback */
typedef void (*AdapterPropertiesCallbackFuncType)(BluetoothIfStatus status,
                                               int NumProperties,
                                               BluetoothProperty *properties);

/* GET/SET Remote Device Properties callback */
typedef void (*RemoteDevicePropertiesCallbackFuncType)(BluetoothIfStatus status,
                                                       Btbdaddr *pBdAddr,
                                                       int NumProperties,
                                                       BluetoothProperty *properties);

/* New device discovered callback */
/* If EIR data is not present, then BD_NAME and RSSI shall be NULL and -1
 * respectively */
typedef void (*DeviceFoundCallbackFuncType)(int NumProperties,
                                         BluetoothProperty *properties);

/* Discovery state changed callback */
typedef void (*DiscoveryStateChangedCallbackFuncType)(BluetoothDiscoveryState state);

/* Bluetooth Legacy PinKey Request callback */
typedef void (*PinRequestCallbackFuncType)(Btbdaddr *pBdAddr,
                                        Btbdname *pBdName, ClassOfDevice *pCod);

/* Bluetooth SSP Request callback */
typedef void (*SspRequestCallbackFuncType)(Btbdaddr *pBdAddr,
                                     Btbdname *pBdName,
                                     ClassOfDevice *pCod,
                                     BluetoothSspVariant PairingVariant,
                                     uint32_t pass_key);

/* Bluetooth Bond state changed callback */
/* Invoked in response to create_bond, cancel_bond or remove_bond */
typedef void (*BondStateChangedCallbackFuncType)(BluetoothIfStatus status,
                                            Btbdaddr *pBdAddr,
                                            BluetoothBondState state);

/* Bluetooth SetReport status callback */
typedef void (*SetReportStatusCallbackFuncType)(Btbdaddr *pBdAddr,
                                                BluetoothHhStatus status );

/* Bluetooth GetReport status callback */
typedef void (*GetReportStatusCallbackFuncType)(Btbdaddr *pBdAddr,
                                                BluetoothHhStatus status,
                                                uint16_t length,
                                                uint8_t *pData );

/* Bluetooth ACL connection state changed callback */
typedef void (*AclStateChangedCallbackFuncType)(BluetoothIfStatus status, Btbdaddr *pBdAddr,
                                            BluetoothAclState state);


/* Thread Associate/Disassociate Callback */
/* Callback that is invoked by the callback thread to allow upper layer to attach/detach */
typedef void (*CallbackThreadEventFuncType)(BluetoothCallbackThreadEvent evt);

/* Bluetooth Test Mode Callback */
/* Receive any HCI event from controller. Must be in DUT Mode for this callback to be received */
typedef void (*DutModeReceivedCallbackFuncType)(uint16_t opcode, uint8_t *buffer, uint8_t len);

/* LE Test mode callbacks */
typedef void (*LEtestModeCallbackFuncType)(BluetoothIfStatus status, uint16_t NumPackets);

/* Callback invoked when energy details are obtained */
typedef void (*EnergyInfoCallbackFuncType)(BluetoothActivityEnergyInfo *pEnergyInfo);

/* Callback invoked when a fatal error occurs */
typedef void (*FatalErrorCallbackFuncType)();

/* Bluetooth DM callback structure. */
typedef struct {
    /* set to sizeof(BluetoothCallbacks) */
    size_t size;
    AdapterStateChangedCallbackFuncType      AdapterStateChangedCallback;
    AdapterPropertiesCallbackFuncType        AdapterPropertiesCallback;
    RemoteDevicePropertiesCallbackFuncType   RemoteDevicePropertiesCallback;
    DeviceFoundCallbackFuncType              DeviceFoundCallback;
    DiscoveryStateChangedCallbackFuncType    DiscoveryStateChangedCallback;
    PinRequestCallbackFuncType               PinRequestCallback;
    SspRequestCallbackFuncType               SspRequestCallback;
    BondStateChangedCallbackFuncType         BondStateChangedCallback;
    SetReportStatusCallbackFuncType          SetReportStatusCallback;
    GetReportStatusCallbackFuncType          GetReportStatusCallback;
    AclStateChangedCallbackFuncType          AclStateChangedCallback;
    CallbackThreadEventFuncType              CallbackThreadEventCallback;
    DutModeReceivedCallbackFuncType          DutModeReceivedCallback;
    LEtestModeCallbackFuncType               LEtestModeCallback;
    EnergyInfoCallbackFuncType               EnergyInfoCallback;
    FatalErrorCallbackFuncType               FatalErrorCallback;
} BluetoothCallbacks;

/*
 * @brief Represents the standard Bluetooth HAL
 */
struct BluetoothInterface
{
    size_t size;

    /*
     * Opens the interface and provides the callback routines
     * to the implemenation of this interface.
     */
    int (*InitializeBluetooth)(BluetoothCallbacks* callbacks);

    /* Enable Bluetooth. */
    int (*EnableBluetooth)(void);

    /* Disable Bluetooth. */
    int (*DisableBluetooth)(void);

    /* Closes the interface. */
    void (*CleanupBluetooth)(void);

    /* Get all Bluetooth Adapter properties at init */
    int (*GetAdapterProperties)(AdapterProperty *pAdapterProperty);

    /* Get Bluetooth Adapter property of 'type' */
    int (*GetAdapterProperty)(BluetoothProperty *property);

    /* Set Bluetooth Adapter property of 'type' */
    /* Based on the type, val shall be one of Btbdaddr or Btbdname
     *  or bt_scanmode_t etc
     */
    int (*SetAdapterProperty)(const BluetoothProperty *property);

    /* Get all Remote Device properties */
    int (*GetRemoteDeviceProperties)(Btbdaddr *pRemoteAddr);

    /* Get Remote Device property of 'type' */
    int (*GetRemoteDeviceProperty)(Btbdaddr *pRemoteAddr,BluetoothPropertyType type);

    /* Set Remote Device property of 'type' */
    int (*SetRemoteDeviceProperty)(Btbdaddr *pRemoteAddr,const BluetoothProperty *property);

    /* Get Remote Device's service record  for the given UUID */
    int (*GetRemoteServiceRecord)(Btbdaddr *pRemoteAddr,BluetoothUUID *uuid);

    /* Start SDP to get remote services */
    int (*GetRemoteServices)(Btbdaddr *pRemoteAddr);

    /* Start Discovery */
    int (*StartDiscovery)(void);

    /* Cancel Discovery */
    int (*CancelDiscovery)(void);

    /* Create Bluetooth Bonding */
    int (*CreateBond)(const Btbdaddr *pBdAddr, int transport);

    /* Remove Bond */
    int (*RemoveBond)(const Btbdaddr *pBdAddr);

    /* Cancel Bond */
    int (*CancelBond)(const Btbdaddr *pBdAddr);

    /* Get the connection status for a given remote device.*/
    int (*GetConnectionState)(const Btbdaddr *pBdAddr);

    /* BT Legacy PinKey Reply */
    int (*PinReply)(const Btbdaddr *pBdAddr, uint8_t accept,uint8_t pin_len, BluetoothPinCode *pPinCode);

    /* BT SSP (Secure Simple Pairing) Reply */
    int (*SspReply)(const Btbdaddr *pBdAddr, BluetoothSspVariant variant,uint8_t accept, uint32_t passkey);

    /* Get Bluetooth profile interface */
    const void* (*GetProfileInterface) (const char *pProfileId);

    /* Bluetooth Test Mode APIs */

    /* Configure DUT Mode - Use this mode to enter/exit DUT mode */
    int (*DutModeConfigure)(uint8_t enable);

    /* Send any test HCI (vendor-specific) command to the controller. Must be in DUT Mode */
    int (*DutModeSend)(uint16_t opcode, uint8_t *buffer, uint8_t len);

    /* BLE Test Mode APIs */
    /* opcode MUST be one of: LE_Receiver_Test, LE_Transmitter_Test, LE_Test_End */
    int (*LEtestMode)(uint16_t opcode, uint8_t *buffer, uint8_t len);

    /* enable or disable bluetooth HCI snoop log */
    int (*ConfigHciSnoopLog)(uint8_t enable);

    // int (*set_os_callouts)(bt_os_callouts_t *callouts); // NOT NEEDED

    /* Read Energy info details */
    int (*ReadEnergyInfo)();

    /* Modify bluetooth power boost settings */
    void (*EnableBluetoothBoostSetting)(bool enable);

    /* Get bluetooth power boost settings */
    bool (*IsBluetoothBoostEnabled)(void);

    /* Modify bluetooth AFH settings */
    void (*EnableBluetoothAfhSetting)(bool enable);

    /* Get bluetooth AFH settings */
    bool (*IsBluetoothAfhEnabled)(void);

};



/* Callback for connection state change.
 *  state will have one of the values from BluetoothHhConnectionState
 */
typedef void (* BluetoothHhconnectionStateFuncType)(Btbdaddr *pBdAddr, BluetoothHhConnectionState state);

/*
 * Callback for get hid info
 * HidInfo will contain attr_mask, sub_class, app_id, vendor_id, product_id, version, ctry_code, len
 */
typedef void (* BluetoothHhHidInfoFuncType)(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo);

/*
 * Callback for get protocol api.
 * the protocol mode is one of the value from BluetoothHhProtocolMode
 */
typedef void (* BluetoothHhprotocolModeFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus,
                                                 BluetoothHhProtocolMode mode);

/* Callback for get/set_idle_time api. */
typedef void (* BluetoothHhidleTimeFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, int IdleRate);

/* Callback for get report api. */
typedef void (* BluetoothHhgetReportFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, uint8_t* RptData, int RptSize);

/* Callback for vitual unplug api. */
typedef void (* BluetoothHhvirtualUnplugFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus);

/* Callback for set_report/set_protocol api */
typedef void (* BluetoothHhhandshakeFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus);

/* Callback for set_report_status api */
typedef void (* BluetoothHhSetReportStatusFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus status);

/* Callback for get_report_status api */
typedef void (* BluetoothHhGetReportStatusFuncType)(Btbdaddr *pBdAddr, BluetoothHhStatus status, uint16_t length, uint8_t *pData);

/* BT-HH callback structure. */
typedef struct {
    size_t size;   // set to sizeof(BluetoothHhCallbacks)
    BluetoothHhconnectionStateFuncType  ConnectionStateCallback;
    BluetoothHhHidInfoFuncType          HidInfoCallback;
    BluetoothHhprotocolModeFuncType     ProtocolModeCallback;
    BluetoothHhidleTimeFuncType         IdleTimeCallback;
    BluetoothHhgetReportFuncType        GetReportCallback;
    BluetoothHhvirtualUnplugFuncType    VirtualUnplugCallback;
    BluetoothHhhandshakeFuncType        HandshakeCallback;
    BluetoothHhSetReportStatusFuncType  SetReportStatusCallback;
    BluetoothHhGetReportStatusFuncType  GetReportStatusCallback;
} BluetoothHhCallbacks;

/* Represents the standard BT-HH interface. */
struct BluetoothHhInterface {

    size_t          size; // set to sizeof(BluetoothHhHhInterface)

    /* Register the BtHh callbacks */
    BluetoothHhStatus (*InitializeHid)(BluetoothHhCallbacks* callbacks, uint16_t interfaceVersion);

    /* connect to hid device */
    BluetoothHhStatus (*connect)(Btbdaddr *pBdAddr);

    /* Disconnect from hid device */
    BluetoothHhStatus (*disconnect)(Btbdaddr *pBdAddr);

    /* Virtual UnPlug (VUP) the specified HID device */
    BluetoothHhStatus (*VirtualUnplug)(Btbdaddr *pBdAddr);

    /* Set the HID device descriptor for the specified HID device. */
    BluetoothHhStatus (*SetInfo)(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo);

    /* Get the HID proto mode. */
    BluetoothHhStatus (*GetProtocol)(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode);

    /* Set the HID proto mode. */
    BluetoothHhStatus (*SetProtocol)(Btbdaddr *pBdAddr, BluetoothHhProtocolMode protocolMode);

    /* Send a GET_REPORT to HID device. */
    BluetoothHhStatus (*GetReport)(Btbdaddr *pBdAddr, BluetoothHhReportType reportType, uint8_t reportId, int bufferSize);

    /* Send a SET_REPORT to HID device. */
    BluetoothHhStatus (*SetReport)(Btbdaddr *pBdAddr, BluetoothHhReportType reportType, char* report);

    /* Send data to HID device. */
    BluetoothHhStatus (*SendData)(Btbdaddr *pBdAddr, char* data);

    /* Wake Controller */
    BluetoothHhStatus (*WakeController)(Btbdaddr *pBdAddr);

#ifdef __cplusplus
    /* Add paired device settings */
    BluetoothHhStatus (*AddPairedDevice)(nn::settings::system::BluetoothDevicesSettings *pDevice);

    /* Get paired device settings */
    BluetoothHhStatus (*GetPairedDevice)(Btbdaddr *pBdAddr, nn::settings::system::BluetoothDevicesSettings* pDeviceOut);
#endif

    /* Closes the interface. */
    void (*CleanupHid)(void);

    /* Get the latest packet loss rate (for debugging, for WIT) */
    void (*GetLatestPlr)(nn::bluetooth::PlrStatistics *pOut);

    /* Get AFH channel map list */
    void (*GetChannelMap)(ChannelMap* pOutChannelMap);
};

// ///////////////////// HAL ADDITIONAL INTERACES ///////////////////// //
// ///////////////////// HAL ADDITIONAL INTERACES ///////////////////// //
// ///////////////////// HAL ADDITIONAL INTERACES ///////////////////// //
/* Callback for Tsi/Burst */
typedef void (* BluetoothRobsonEvtFuncType)(void *p_data, BluetoothHhStatus hhStatus);
typedef void (* BluetoothGetPendingConnectionsFuncType)(InfoFromExtensionCallbacks *p_data);

/* BT-Ext callback structure. */
typedef struct {
    size_t size; // set to sizeof(BluetoothExtCallbacks)
    BluetoothRobsonEvtFuncType SetTsiCallback;
    BluetoothRobsonEvtFuncType ExitTsiCallback;
    BluetoothRobsonEvtFuncType SetBurstCallback;
    BluetoothRobsonEvtFuncType ExitBurstCallback;
    BluetoothRobsonEvtFuncType SetZeroRetranCallback;
    BluetoothGetPendingConnectionsFuncType GetPendingConnectionsCallback;
} BluetoothExtCallbacks;

/* Represents the BT-Extension . */
struct BluetoothExtension {

    /* set to sizeof(BluetoothExtInterface) */
    size_t          size;

    /* Register the BtExt callbacks */
    BluetoothHhStatus (*InitializeExtension)(BluetoothExtCallbacks* callbacks);

    /* TSI api */
    BluetoothHhStatus (*HalIfSetTSI)(Btbdaddr *pBdAddr, uint8_t tsi_mode);

    /* Exit TSI api */
    BluetoothHhStatus (*HalIfExitTSI)(Btbdaddr *pBdAddr);

    /* Burst mode api */
    BluetoothHhStatus (*HalIfSetBurstMode)(Btbdaddr *pBdAddr);

    /* Exit Burst mode api */
    BluetoothHhStatus (*HalIfExitBurstMode)(Btbdaddr *pBdAddr);

    /* set static AFH */
    BluetoothHhStatus (*HalIfSetSfh)(uint8_t sfh_on, uint8_t *pChannels,uint32_t length, uint32_t sfh_offset);

    /* zero-retrans api */
    BluetoothHhStatus (*HalIfSetZeroRetran)(Btbdaddr *pBdAddr, uint8_t numIds, const uint8_t *pIds);

    /* set mc mode api */
    BluetoothHhStatus (*HalIfSetMcMode)(bool isMcModeEnabled);

    /* start llr mode */
    BluetoothHhStatus (*HalIfStartLlrMode)();

    /* exit llr mode */
    BluetoothHhStatus (*HalIfExitLlrMode)();

    /* set radio on/off */
    BluetoothHhStatus (*HalIfSetRadio)(bool enable);

    /* set radio discoverable/connectable */
    BluetoothHhStatus (*HalIfSetVisibility)(bool discoverable, bool connectable);

    /* set TBFC scan on/off */
    BluetoothHhStatus (*HalIfSetTbfcScan)(bool enable);

    /* get number of pending bluetooth connections */
    BluetoothHhStatus (*HalIfGetPendingConnections)();
};

/*************************************************************************
 * Bluetooth Low Energy Interfaces
 ************************************************************************/
typedef void (* BluetoothLowEnergyFuncType)(InfoFromLowEnergyCallbacks *cbData);
typedef void(*LeConnParamUpdateCbFuncType)(InfoFromLeConnParamUpdateCallback *cbData);
typedef void(*LeConnParamUpdateReqCbFuncType)(InfoFromLeConnParamUpdateReqCallback* cbData);
typedef void(*LeClientStateChangedCbFuncType)(InfoFromLeAppStateChangedCallback *cbData);
typedef void(*LeServerStateChangedCbFuncType)(InfoFromLeAppStateChangedCallback *cbData);
typedef void(*LeClientConnStateChangedCbFuncType)(InfoFromLeConnStateChangedCallback *cbData);
typedef void(*LeServerConnStateChangedCbFuncType)(InfoFromLeConnStateChangedCallback *cbData);
typedef void(*LeScanStateChangedCbFuncType)(InfoFromLeScanStateChangedCallback *cbData);
typedef void(*LeScanFilterStateChangedCbFuncType)(InfoFromLeScanFilterStateChangedCallback *cbData);
typedef void(*LeClientGattOperationCbFuncType)(InfoFromLeGattOperationCallback *cbData);
typedef void(*LeCLientGattServiceDiscoveryCbFuncType)(InfoFromLeClientGattServiceDiscoveryCallback *cbData);
typedef void(*LeCLientMtuConfigCbFuncType)(InfoFromLeClientMtuConfigurationCallback *cbData);
typedef void(*LeServerProfileChangedCbFuncType)(InfoFromLeServerProfileChangedCallback* cbData);
typedef void(*LeServerGattReqCbFuncTye)(InfoFromLeServerGattReqCallback* cbData);

typedef struct {
    size_t size; // set to sizeof(BluetoothLeCallbacks)
    LeConnParamUpdateCbFuncType             LeConnParamUpdateCallback;
    LeConnParamUpdateReqCbFuncType          LeConnParamUpdateReqCallback;
    LeScanStateChangedCbFuncType            LeScanStateChangedCallback;
    LeScanFilterStateChangedCbFuncType      LeScanFilterStateChangedCallback;
    LeClientStateChangedCbFuncType          LeClientStateChangedCallback;
    LeServerStateChangedCbFuncType          LeServerStateChangedCallback;
    LeClientConnStateChangedCbFuncType      LeClientConnStateChangedCallback;
    LeServerConnStateChangedCbFuncType      LeServerConnStateChangedCallback;
    LeClientGattOperationCbFuncType         LeClientGattOperationCallback;
    LeClientGattOperationCbFuncType         LeClientCoreGattOperationCallback;
    LeClientGattOperationCbFuncType         LeClientHidGattOperationCallback;
    LeCLientGattServiceDiscoveryCbFuncType  LeClientGattServiceDiscoveryCallback;
    LeCLientGattServiceDiscoveryCbFuncType  LeClientGattServiceDiscoveryStateChangedCallback;
    LeCLientMtuConfigCbFuncType             LeClientConfigureMtuCallback;
    LeServerProfileChangedCbFuncType        LeServerProfileChangedCallback;
    LeServerGattReqCbFuncTye                LeServerGattReqCallback;
    LeServerGattReqCbFuncTye                LeServerCoreGattReqCallback;
    LeServerGattReqCbFuncTye                LeServerHidGattReqCallback;
} BluetoothLeCallbacks;

/* Represents the standard Bluetooth low energy interface. */
struct BluetoothLeInterface {

    /* set to sizeof(BluetoothLeInterface) */
    size_t          size;

    /*
     * Opens the interface and provides the callback routines
     * to the implemenation of this interface.
     */
    BluetoothLeStatus (*InitializeBle)(BluetoothLeCallbacks* callbacks);

    /* Initialize bluetooth low energy services */
    BluetoothLeStatus (*EnableBle)();

    /* Disables bluetooth low energy */
    BluetoothLeStatus (*DisableBle)();

    /* Cleanup BLE system events and callbacks */
    void (*CleanupBle)(void);

    /* Set BLE discoverable/connectable */
    BluetoothLeStatus (*SetLeVisibility)(bool discoverable, bool connectable);

    /* Set BLE connection parameters */
    BluetoothLeStatus (*SetLeConnectionParameter)(LeConnectionParams connectionParams);

    /* Set BLE default connection parameters */
    BluetoothLeStatus (*SetLeDefaultConnectionParameter)(LeConnectionParams connectionParams);

    /* Set BLE advertisement data */
    BluetoothLeStatus (*SetLeAdvertiseData)(LeAdvertiseData advertiseData);

    /* Set BLE advertisement parameters */
    BluetoothLeStatus (*SetLeAdvertiseParameter)(Btbdaddr *pBdAddr, uint16_t minAdvInt, uint16_t maxAdvInt);

    /* Start BLE scan */
    BluetoothLeStatus (*StartLeScan)();

    /* Stop BLE Scan */
    BluetoothLeStatus (*StopLeScan)();

    /* Set BLE Scan Parameter */
    BluetoothLeStatus(*SetLeScanParameter)(uint16_t scanInterval, uint16_t scanWindow);

    /* Add Scan Filter Condition */
    BluetoothLeStatus(*AddLeScanFilterCondition)(const BleAdvertiseFilter* filter);

    /* Delete Scan Filter Condition */
    BluetoothLeStatus(*DeleteLeScanFilterCondition)(const BleAdvertiseFilter* filter);

    /* Delete a specific Scan Filter */
    BluetoothLeStatus(*DeleteLeScanFilter)(uint8_t filterIndex);

    /* Clear all the Scan Filters */
    BluetoothLeStatus(*ClearLeScanFilters)();

    /* Enable BLE Scan Filter */
    BluetoothLeStatus(*EnableLeScanFilter)(bool enable);

    /* Register BLE GATT client */
    BluetoothLeStatus (*RegisterLeClient)(GattAttributeUuid uuid);

    /* Unregister BLE GATT client */
    BluetoothLeStatus(*UnregisterLeClient)(uint8_t clientIf);

    /* Register BLE GATT client */
    BluetoothLeStatus(*UnregisterLeClientAll)();

    /* Connect to a BLE GATT server */
    BluetoothLeStatus (*LeClientConnect)(uint8_t clientIf, Btbdaddr *pBdAddr, bool isDirect);

    /* Cancel connection request to a BLE GATT server */
    BluetoothLeStatus (*LeClientCancelConnection)(uint8_t clientIf, Btbdaddr* pBdAddr, bool isDirect);

    /* Disconnect from a BLE GATT server */
    BluetoothLeStatus (*LeClientDisconnect)(uint32_t connId);

    /* Get the attributes of a device */
    BluetoothLeStatus (*LeClientGetAttributes)(uint32_t connId, Btbdaddr *pBdAddr);

    /* Discover a BLE GATT service */
    BluetoothLeStatus (*LeClientDiscoverService)(uint32_t connId, GattAttributeUuid filterUuid);

    /* Read characteristic from a BLE GATT service */
    BluetoothLeStatus (*LeClientReadCharacteristic)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType);

    /* Read descriptor from a BLE GATT service */
    BluetoothLeStatus (*LeClientReadDescriptor)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType);

    /* Write a value to a BLE GATT service characteristic */
    BluetoothLeStatus (*LeClientWriteCharacteristic)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse);

    /* Write a value to a BLE GATT service descriptor */
    BluetoothLeStatus (*LeClientWriteDescriptor)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType);

    /* Register to receive notifications for a given characteristic */
    BluetoothLeStatus (*LeClientRegisterNotification)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId);

    /* Deregister a previous request for notifications/indications */
    BluetoothLeStatus (*LeClientDeregisterNotification)(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId);

    /* Configure MTU */
    BluetoothLeStatus (*LeClientConfigureMtu)(uint32_t connId, uint16_t mtu);

    /* Register BLE GATT server */
    BluetoothLeStatus (*RegisterLeServer)(GattAttributeUuid uuid);

    /* Deregister BLE GATT server */
    BluetoothLeStatus (*UnregisterLeServer)(uint8_t serverIf);

    /* Connect to a BLE GATT server */
    BluetoothLeStatus (*LeServerConnect)(uint8_t serverIf, Btbdaddr *pBdAddr, bool isDirect);

    /* Disconnect from a BLE GATT server */
    BluetoothLeStatus (*LeServerDisconnect)(uint8_t serverIf, Btbdaddr *pBdAddr);

    /* Initialize a BLE GATT server service */
    BluetoothLeStatus (*CreateLeService)(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handleNum, bool isPrimaryService);

    /* Start a local BLE GATT server service */
    BluetoothLeStatus (*StartLeService)(uint8_t serverIf, GattAttributeUuid uuid);

    /* Add a BLE GATT characteristic to a service */
    BluetoothLeStatus (*AddLeCharacteristic)(uint8_t serverIf, GattAttributeUuid ServiceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties);

    /* Add a BLE GATT descriptor to a service */
    BluetoothLeStatus (*AddLeDescriptor)(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid descUuid, uint16_t permissions);

    BluetoothLeStatus(*SetLeResponse)(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const uint8_t *pData, uint16_t len);

    /* Send indication/notification to client */
    BluetoothLeStatus (*LeSendIndication)(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t len, const uint8_t* pData, bool confirm);

    /* Get the first characteristic from a service */
    BluetoothLeStatus (*LeGetFirstCharacteristic)(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter);

    /* Get the subsequent characteristic from a service */
    BluetoothLeStatus (*LeGetNextCharacteristic)(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter);

    /* Find the first descriptor of the characteristic on the given server from cache */
    BluetoothLeStatus (*LeGetFirstDescriptor)(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter);

    /* Find the subsequent descriptor of the characteristic on the given server from cache */
    BluetoothLeStatus (*LeGetNextDescriptor)(GattId *descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter);

    /* Register GATT data path to be informed via BLE Core Event */
    BluetoothLeStatus(*RegisterLeCoreDataPath)(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE Core Event */
    BluetoothLeStatus(*UnregisterLeCoreDataPath)(const GattAttributeUuid& uuid);

    /* Register GATT data path to be informed via BLE HID Event */
    BluetoothLeStatus(*RegisterLeHidDataPath)(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE HID Event */
    BluetoothLeStatus(*UnregisterLeHidDataPath)(const GattAttributeUuid& uuid);

    /* Register GATT data path to be informed via BLE Event */
    BluetoothLeStatus(*RegisterLeDataPath)(const GattAttributeUuid& uuid);

    /* Unregister GATT data path to be informed via BLE Event */
    BluetoothLeStatus(*UnregisterLeDataPath)(const GattAttributeUuid& uuid);
};

#ifdef __cplusplus
}
#endif // ifdef __cplusplus
