﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "bluetooth_InternalTypes.h"

#ifdef __cplusplus
#include <nn/bluetooth/bluetooth_Types.h>
#include "bsa_api.h"
#include "xml_param.h"
#include "bt_types.h"

extern "C"
{
#endif // ifdef __cplusplus

#ifndef APP_DISC_NB_DEVICES
enum {APP_DISC_NB_DEVICES=20};
#endif


typedef struct
{
    tBSA_DISC_DEV devs[APP_DISC_NB_DEVICES];
} tAPP_DISCOVERY_CB;

extern tAPP_DISCOVERY_CB app_discovery_cb;

// These Functions are specific to init/close BSA client/server
void                 bcmServerInit();
void                 bcmServerShutdown();
int                  BtHalBsaCoreClientInit();
int                  BtHalSetBluetoothSecurity();
void                 BtHalBsaCloseClient();
int                  BtHalBsaCloseServer();
void                 BtHalInitialize();
void                 BtHalFinalize();

// Bluetooth HAL core API is currently defined in app_robson_main.c
BluetoothHhStatus    BtHalGetRemoteServices(Btbdaddr *pRemoteAddr);
void                 BtHalStartDiscovery(void);
void                 BtHalCancelDiscovery(void);
BluetoothHhStatus    BtHalCreateBond(const Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalRemoveBond(const Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalCancelBond(const Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalPinReply(const Btbdaddr *pBdAddr, uint8_t accept,uint8_t pin_len, BluetoothPinCode *pPinCode);
BluetoothHhStatus    BtHalSspReply(const Btbdaddr *pBdAddr, BluetoothSspVariant variant,uint8_t accept, uint32_t passkey);

// Bluetooth HAL Extensions
BluetoothHhStatus    BtHalSetTsi(Btbdaddr *pBdAddr,uint8_t tsi_mode);
BluetoothHhStatus    BtHalExitTsi(Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalSetBurstMode(Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalExitBurstMode(Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalSetSfh(uint8_t sfh_on, uint8_t *pChannels,uint32_t length, uint32_t sfh_offset);
BluetoothHhStatus    BtHalSetZeroRetran(Btbdaddr *pBdAddr, uint8_t numIds, const uint8_t *pIds);
BluetoothHhStatus    BtHalSetMcMode(bool isMcModeEnabled);
BluetoothHhStatus    BtHalStartLlrMode();
BluetoothHhStatus    BtHalExitLlrMode();
BluetoothHhStatus    BtHalSetRadio(bool enable);
BluetoothHhStatus    BtHalSetVisibility(bool discoverable, bool connectable);
BluetoothHhStatus    BtHalSetTbfcScan(bool enable);
BluetoothHhStatus    BtHalGetNumPendingConnections();

// Misc functions not part of the original HAL spec
tBSA_STATUS          BtHalBsaIntSecAddDev(tAPP_XML_REM_DEVICE *p_xml_dev);
tBSA_STATUS          BtHalBsaIntHhAddDev(tAPP_XML_REM_DEVICE *p_xml_dev);
BluetoothHhStatus    BtHalGetAdapterProperties(AdapterProperty *pAdapterProperty);
BluetoothHhStatus    BtHalGetHostBdAddr(Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalSetHostBdAddr(const Btbdaddr *pBdAddr);
BluetoothHhStatus    BtHalGetHostBdName(Btbdname *pBdName, uint8_t *length);
BluetoothHhStatus    BtHalSetHostBdName(const Btbdname *pBdName, uint8_t length);
BluetoothHhStatus    BtHalGetHostClassOfDevice(ClassOfDevice *pBdClass);
BluetoothHhStatus    BtHalSetHostClassOfDevice(const ClassOfDevice *pBdClass);
BluetoothHhStatus    BtHalGetHostFeatureSet(uint8_t *pfeatureSet);
BluetoothHhStatus    BtHalSetHostFeatureSet(const uint8_t *pfeatureSet);
BluetoothHhStatus    BtHalSetHostSuperBdAddr(const Btbdaddr *pBdAddr);
int                  BtHalBsaGetDscpinfo(uint8_t handle);
tBSA_DISC_DEV       *BtHalBsaDiscFindDevice(BD_ADDR bda);
void                 BtHalBsaDiscParseEir(UINT8 *p_eir);
BluetoothHhStatus    BtHalBsaConvertResult(tBSA_STATUS resultFromBsa);

void bluetoothGpioApWakeBtEnable(bool enable);
#ifdef __cplusplus
}

void bluetoothGpioPowerEnable(bool enable);


namespace nn {
namespace bluetooth {

extern nn::os::EventType g_InternalEvent;
extern nn::os::EventType g_InternalEventForHid;
extern nn::os::EventType g_InternalEventForHidReport;
extern nn::os::EventType g_InternalEventForBle;
extern nn::os::EventType g_InternalEventForBleCore;
extern nn::os::EventType g_InternalEventForBleHid;
extern bool              g_PendingIntEvent;
extern bool              g_PendingIntEventForHid;
extern bool              g_PendingIntEventForHidReport;
extern bool              g_PendingIntEventForBle;
extern bool              g_PendingIntEventForBleCore;
extern bool              g_PendingIntEventForBleHid;

// Global to determine client version
extern uint16_t g_BtInterfaceVersion;

// Global to notify when BSA is done shutting down
extern bool serverShutdownDone;

/**
 * @brief Represents the standard Bluetooth interface.
 */
class BluetoothInterfaceC
{
    NN_DISALLOW_COPY(BluetoothInterfaceC);
    NN_DISALLOW_MOVE(BluetoothInterfaceC);

public:
    BluetoothInterfaceC()                   NN_NOEXCEPT : m_BluetoothInterface() {}
    void InitializeBluetoothInterface(void) NN_NOEXCEPT;

    BluetoothInterface& GetBluetoothInterface() NN_NOEXCEPT
    {
        return m_BluetoothInterface;
    };

private:

    BluetoothInterface   m_BluetoothInterface;

    static int InitializeBluetooth(BluetoothCallbacks* callbacks);
    static int EnableBluetooth(void);
    static int DisableBluetooth(void);
    static void CleanupBluetooth(void);
    static int GetAdapterProperties(AdapterProperty *pAdapterProperty);
    static int GetAdapterProperty(BluetoothProperty *property);
    static int SetAdapterProperty(const BluetoothProperty *property);
    static int GetRemoteDeviceProperties(Btbdaddr *pRemoteAddr);
    static int GetRemoteDeviceProperty(Btbdaddr *pRemoteAddr,BluetoothPropertyType type);
    static int SetRemoteDeviceProperty(Btbdaddr *pRemoteAddr,const BluetoothProperty *property);
    static int GetRemoteServiceRecord(Btbdaddr *pRemoteAddr,BluetoothUUID *uuid);
    static int GetRemoteServices(Btbdaddr *pRemoteAddr);
    static int StartDiscovery(void);
    static int CancelDiscovery(void);
    static int CreateBond(const Btbdaddr *pBdAddr, int transport);
    static int RemoveBond(const Btbdaddr *pBdAddr);
    static int CancelBond(const Btbdaddr *pBdAddr);
    static int PinReply(const Btbdaddr *pBdAddr, uint8_t accept,uint8_t pin_len, BluetoothPinCode *pPinCode);
    static int SspReply(const Btbdaddr *pBdAddr, BluetoothSspVariant variant,uint8_t accept, uint32_t passkey);
    static const void* GetProfileInterface(const char *pProfileId);
    static int DutModeConfigure(uint8_t enable);
    static int DutModeSend(uint16_t opcode, uint8_t *buffer, uint8_t len);
    static int LEtestMode(uint16_t opcode, uint8_t *buffer, uint8_t len);
    static int ConfigHciSnoopLog(uint8_t enable);
    static int ReadEnergyInfo(void);
    static void EnableBluetoothBoostSetting(bool enable);
    static bool IsBluetoothBoostEnabled();
    static void EnableBluetoothAfhSetting(bool enable);
    static bool IsBluetoothAfhEnabled();
};

} // bluetooth
} // nn
#endif // ifdef __cplusplus
