﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/util/util_TypedStorage.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/mem.h>


//-----------------------------------------------------------------------------
//  アロケータの定義
//-----------------------------------------------------------------------------

namespace {

    // malloc の領域として静的な領域を確保
    const size_t                MallocBufferSize = 4 * 1024 * 1024;
    NN_ALIGNAS(4096) uint8_t    g_MallocBuffer[ MallocBufferSize ];

    // 初期化は nninitStartup() から行なうため、
    // nninitStartup() 後にコンストラクタが起動しないようにしておく。
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_SampleAllocator;

}   // anonymous

extern "C" void* malloc(size_t size)
{
    return Get(g_SampleAllocator).Allocate(size);
}

extern "C" void free(void* p)
{
    if (p)
    {
        Get(g_SampleAllocator).Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);

    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    // メモリブロックのサイズを変更する
    // Reallocate() の仕様は realloc() の仕様と同じため、そのまま呼ぶ
    return Get(g_SampleAllocator).Reallocate(p, newSize);
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return Get(g_SampleAllocator).Allocate(size, alignment);
}

extern "C" size_t malloc_usable_size(void* p)
{
    if(!p)
    {
        return 0;
    }

    return Get(g_SampleAllocator).GetSizeOf(p);
}

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------

extern "C" void MallocinitStartup()
{
    NN_LOG("[bluetooth] nninitStartup() function is invoked.\n");

    // nninitStartup() では静的オブジェクトのコンストラクタは
    // まだ呼ばれていないため、placement new で明示的に呼ぶ。
    new( &Get(g_SampleAllocator) ) nn::mem::StandardAllocator;

    // 静的に確保したバッファを malloc 用の領域として初期化
    Get(g_SampleAllocator).Initialize(g_MallocBuffer, MallocBufferSize);

    NN_LOG("[bluetooth] MallocBuffer is 0x%p - 0x%p\n",
                    g_MallocBuffer, g_MallocBuffer + MallocBufferSize);
}

