﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bcat_PowerStateEventManager.h"
#include "bcat_NintendoAccountCacheUpdateThread.h"

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/bgtc.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/bcat/service/bcat_Server.h>
#include <nn/news/service/news_Server.h>
#include <nn/prepo/service/prepo_Server.h>

namespace
{
    void NotifySystemHalfAwaked() NN_NOEXCEPT
    {
        nn::bcat::service::NotifySystemHalfAwaked();
        nn::news::service::NotifySystemHalfAwaked();
        nn::prepo::service::NotifySystemHalfAwaked();

        NintendoAccountCacheUpdateThread::NotifySystemHalfAwaked();
    }

    void NotifySystemFullAwaked() NN_NOEXCEPT
    {
        nn::bcat::service::NotifySystemFullAwaked();
        nn::news::service::NotifySystemFullAwaked();
        nn::prepo::service::NotifySystemFullAwaked();

        NintendoAccountCacheUpdateThread::NotifySystemFullAwaked();
    }

    void NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
    {
        nn::bcat::service::NotifySystemSleepReady(isFromFullAwake);
        nn::news::service::NotifySystemSleepReady(isFromFullAwake);
        nn::prepo::service::NotifySystemSleepReady(isFromFullAwake);

        NintendoAccountCacheUpdateThread::NotifySystemSleepReady(isFromFullAwake);
    }

    void NotifySystemShutdownReady() NN_NOEXCEPT
    {
        nn::bcat::service::NotifySystemShutdownReady();
        nn::news::service::NotifySystemShutdownReady();
        nn::prepo::service::NotifySystemShutdownReady();
    }

    void WaitForSleepReady() NN_NOEXCEPT
    {
        nn::bcat::service::WaitForSleepReady();
        nn::news::service::WaitForSleepReady();
        nn::prepo::service::WaitForSleepReady();
    }

    void WaitForShutdownReady() NN_NOEXCEPT
    {
        nn::bcat::service::WaitForShutdownReady();
        nn::news::service::WaitForShutdownReady();
        nn::prepo::service::WaitForShutdownReady();
    }
}

void PowerStateEventManager::LoopAuto() NN_NOEXCEPT
{
    const nn::psc::PmModuleId dependencies[] = {nn::psc::PmModuleId_Fs, nn::psc::PmModuleId_Bgtc};

    nn::psc::PmModule module;
    NN_ABORT_UNLESS_RESULT_SUCCESS(module.Initialize(nn::psc::PmModuleId_Bcat,
        dependencies, NN_ARRAY_SIZE(dependencies), nn::os::EventClearMode_ManualClear));

    NN_UTIL_SCOPE_EXIT
    {
        module.Finalize();
    };

    nn::bgtc::Task bgtcTask;
    NN_ABORT_UNLESS_RESULT_SUCCESS(bgtcTask.Initialize());

    NN_UTIL_SCOPE_EXIT
    {
        bgtcTask.Finalize();
    };

    bgtcTask.SetClientName("bcat.PM");

    bool isFromFullAwake = true;

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(bgtcTask.GetTriggerEvent().GetBase(), module.GetEventPointer()->GetBase());

        if (bgtcTask.GetTriggerEvent().TryWait())
        {
            bgtcTask.GetTriggerEvent().Clear();

            if (nn::bgtc::IsInHalfAwake())
            {
                NN_DETAIL_BCAT_INFO("[bcat] Resume.(MinimumAwakeForBackground)\n");

                NotifySystemHalfAwaked();
            }
        }
        if (module.GetEventPointer()->TryWait())
        {
            module.GetEventPointer()->Clear();

            nn::psc::PmState state;
            nn::psc::PmFlagSet flags;
            NN_ABORT_UNLESS_RESULT_SUCCESS(module.GetRequest(&state, &flags));

            switch (state)
            {
            case nn::psc::PmState_FullAwake:
                {
                    NN_DETAIL_BCAT_INFO("[bcat] Resume.\n");
                    isFromFullAwake = true;

                    NotifySystemFullAwaked();
                }
                break;
            case nn::psc::PmState_SleepReady:
                {
                    NN_DETAIL_BCAT_INFO("[bcat] Suspend.\n");

                    nn::bgtc::Interval minInterval = INT32_MAX;
                    bool hasTask = false;

                    nn::bgtc::Interval interval;

                    if (nn::bcat::service::GetHalfAwakeScheduleInterval(&interval))
                    {
                        minInterval = std::min(minInterval, interval);
                        hasTask = true;
                    }
                    if (nn::news::service::GetHalfAwakeScheduleInterval(&interval))
                    {
                        minInterval = std::min(minInterval, interval);
                        hasTask = true;
                    }
                    if (nn::prepo::service::GetHalfAwakeScheduleInterval(&interval))
                    {
                        minInterval = std::min(minInterval, interval);
                        hasTask = true;
                    }
                    if (hasTask)
                    {
                        NN_DETAIL_BCAT_INFO("[bcat] nn::bgtc::Task::Schedule(%d) ...\n", minInterval);
                        bgtcTask.Schedule(minInterval);
                    }

                    NotifySystemSleepReady(isFromFullAwake);
                    WaitForSleepReady();

                    isFromFullAwake = false;
                }
                break;
            case nn::psc::PmState_ShutdownReady:
                {
                    NN_DETAIL_BCAT_INFO("[bcat] Shutdown.\n");

                    NotifySystemShutdownReady();
                    WaitForShutdownReady();
                }
                break;
            default:
                break;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(module.Acknowledge(state, nn::ResultSuccess()));
        }
    }
}
