﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bcat_NintendoAccountCacheUpdateThread.h"

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os.h>
#include <nn/account.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/prepo/service/prepo_Server.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
    // ニンテンドーアカウントのキャッシュ有効期間
    const nn::TimeSpan CacheLifetime = nn::TimeSpan::FromDays(1);
    // スケジュール間隔
    const nn::TimeSpan ScheduleInterval = nn::TimeSpan::FromHours(1);
    // スケジュール間隔（スリープ中）
    const nn::TimeSpan ScheduleIntervalInSleep = nn::TimeSpan::FromHours(3);
}

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[16 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_UpdateRequestEvent(nn::os::EventClearMode_ManualClear);

    nn::bgtc::Task g_BgtcTask;
}

namespace
{
    void NotifyUpdateNintendoAccountCacheResult(const nn::account::Uid& uid, nn::Result result) NN_NOEXCEPT
    {
        nn::prepo::service::NotifyUpdateNintendoAccountCacheResult(uid, result);
    }
}

namespace
{
    nn::Result UpdateNintendoAccountCache(const nn::account::Uid& uid) NN_NOEXCEPT
    {
        nn::account::NetworkServiceAccountManager manager;
        NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

        bool matched;
        nn::account::AsyncContext async;

        NN_RESULT_DO(manager.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(&matched, &async, CacheLifetime));

        if (!matched)
        {
            NN_DETAIL_BCAT_INFO("[bcat] The nintendo account cache is still effective. uid = %016llx_%016llx\n",
                uid._data[0], uid._data[1]);
            NN_RESULT_SUCCESS;
        }

        nn::os::SystemEvent completionEvent;
        NN_RESULT_DO(async.GetSystemEvent(&completionEvent));

        while (!completionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(5)))
        {
            if (g_StopEvent.TryWait())
            {
                async.Cancel();
                completionEvent.Wait();
            }
        }

        NN_RESULT_DO(async.GetResult());

        NN_RESULT_SUCCESS;
    }

    void UpdateNintendoAccountCache() NN_NOEXCEPT
    {
        nn::nifm::NetworkConnection connection;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
            nn::nifm::RequirementPreset_InternetForSystemProcessSharable));

        connection.SubmitRequestAndWait();

        if (!connection.IsAvailable())
        {
            return;
        }

        nn::account::Uid users[nn::account::UserCountMax] = {};
        int count;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, nn::account::UserCountMax));

        for (auto i = 0; i < count; i++)
        {
            nn::Result result = UpdateNintendoAccountCache(users[i]);

            NotifyUpdateNintendoAccountCacheResult(users[i], result);

            if (result.IsFailure() && !nn::account::ResultNetworkServiceAccountUnavailable::Includes(result))
            {
                NN_DETAIL_BCAT_INFO("[bcat] Failed to update nintendo account cache. uid = %016llx_%016llx, code = %03d-%04d\n",
                    users[i]._data[0], users[i]._data[1], result.GetModule(), result.GetDescription());
            }
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitAny(g_StopEvent.GetBase(), g_UpdateRequestEvent.GetBase(), g_BgtcTask.GetScheduleEvent().GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }

            if (g_UpdateRequestEvent.TryWait())
            {
                NN_DETAIL_BCAT_INFO("[bcat] (!) NintendoAccountCacheUpdateThread::UpdateRequestEvent signaled.\n");
                g_UpdateRequestEvent.Clear();
            }
            if (g_BgtcTask.GetScheduleEvent().TryWait())
            {
                NN_DETAIL_BCAT_INFO("[bcat] (!) NintendoAccountCacheUpdateThread::BgtcScheduleEvent signaled.\n");
                g_BgtcTask.GetScheduleEvent().Clear();
            }

            if (g_BgtcTask.NotifyStarting().IsSuccess())
            {
                UpdateNintendoAccountCache();

                g_BgtcTask.NotifyFinished();
            }
        }
    }
}

void NintendoAccountCacheUpdateThread::Start() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_BgtcTask.Initialize());

    g_BgtcTask.SetClientName("bcat.NA");
    g_BgtcTask.SchedulePeriodic(0, static_cast<nn::bgtc::Interval>(ScheduleInterval.GetSeconds()));

    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(bcat, NaCacheUpdater)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(bcat, NaCacheUpdater));
    nn::os::StartThread(&g_Thread);
}

void NintendoAccountCacheUpdateThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);

    g_BgtcTask.Finalize();
}

void NintendoAccountCacheUpdateThread::RequestNintendoAccountCacheUpdate() NN_NOEXCEPT
{
    g_UpdateRequestEvent.Signal();
}

void NintendoAccountCacheUpdateThread::NotifySystemHalfAwaked() NN_NOEXCEPT
{
}

void NintendoAccountCacheUpdateThread::NotifySystemFullAwaked() NN_NOEXCEPT
{
    g_BgtcTask.SchedulePeriodic(0, static_cast<nn::bgtc::Interval>(ScheduleInterval.GetSeconds()));
}

void NintendoAccountCacheUpdateThread::NotifySystemSleepReady(bool isFromFullAwake) NN_NOEXCEPT
{
    if (isFromFullAwake)
    {
        nn::bgtc::Interval interval = static_cast<nn::bgtc::Interval>(ScheduleIntervalInSleep.GetSeconds());

        g_BgtcTask.SchedulePeriodic(interval, interval);
    }
}
