﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic> // std::atomic_bool
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/init.h>
#include <nn/psc.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/audio/server/audio_UserServiceHipcServer.h>
#include <nn/audio/server/audio_AppletServiceHipcServer.h>
#include <nn/audio/server/audio_DebuggerHipcServer.h>
#include <nn/audioctrl/server/audioctrl_AudioControllerHipcServer.h>

#include <nn/audio/server/audio_FirmwareDebugSettings.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/codec/server/codec_HardwareOpusDecoderManager.h>
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

#include <audio_AudioInServers.h>
#include <audio_FinalOutputRecorderServers.h>
#include <audio_AudioOutServers.h>
#include <audio_AudioRendererServers.h>
#include <audio_AppletVolumeManager.h>
#include "audio_AudioHardware.h"
#include "detail/audioctrl_AudioControllerService.h" // nn::audioctrl::Sleep/WakeAudioControllerService();

#define NN_AUDIO_PROCESS_LOG(...)
#if !defined(NN_AUDIO_PROCESS_LOG)
#include <nn/audio/detail/audio_Log.h>
#define NN_AUDIO_PROCESS_LOG(...) NN_DETAIL_AUDIO_INFO(__VA_ARGS__)
#endif

namespace  {
NN_OS_ALIGNAS_THREAD_STACK int8_t g_AudioUserServiceThreadStack[12 * 1024];
nn::os::ThreadType g_AudioUserServiceThread;

NN_OS_ALIGNAS_THREAD_STACK int8_t g_AudioAppletServiceThreadStack[12 * 1024];
nn::os::ThreadType g_AudioAppletServiceThread;

NN_OS_ALIGNAS_THREAD_STACK int8_t g_AudioDebuggerServiceThreadStack[12 * 1024];
nn::os::ThreadType g_AudioDebuggerServiceThread;

NN_OS_ALIGNAS_THREAD_STACK int8_t g_AudioControllerThreadStack[12 * 1024];
nn::os::ThreadType g_AudioControllerThread;

NN_OS_ALIGNAS_THREAD_STACK int8_t g_AudioPmModuleHandlerThreadStack[20 * 1024];
nn::os::ThreadType g_AudioPmModuleHandlerThread;
std::atomic_bool g_IsAudioPmModuleHandlerThreadActive;

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
NN_OS_ALIGNAS_THREAD_STACK int8_t g_HardwareOpusDecoderThreadStack[12 * 1024];
nn::os::ThreadType g_HardwareOpusDecoderThread;
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

} // namespace

extern "C" void nninitStartup()
{

}

// If you want to switch log output from LogManager to UART, please enable this block.
#if 0
extern "C" void nndiagStartup()
{

}
#endif

namespace  {
void AudioUserServiceThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::audio::server::LoopUserServiceServer();
}

void AudioAppletServiceThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::audio::server::LoopAppletServiceServer();
}

void AudioDebuggerServiceThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::audio::server::LoopAudioDebugServer();
}

void AudioControllerThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::audioctrl::server::LoopAudioControllerServer();
}

nn::Result WakeAudio() NN_NOEXCEPT;
nn::Result SleepAudio() NN_NOEXCEPT;
nn::Result ShutdownAudio() NN_NOEXCEPT;

void AudioPmModuleHandlerThreadFunc(void* arg) NN_NOEXCEPT
{
    nn::psc::PmModule audioPmModule;

    const nn::psc::PmModuleId audioPmModuleDependencies[] = {
        nn::psc::PmModuleId_PcvClock,    // nne::audio (APE)
        nn::psc::PmModuleId_PcvVoltage,  // nne::audio (APE)
        nn::psc::PmModuleId_I2c,         // nne::audio (CODEC IC)
        nn::psc::PmModuleId_Gpio,        // nn::audioctrl, nne::audio (CODEC IC)
        nn::psc::PmModuleId_Display,     // nne::audio (HDA depends on the SOR1 clock controlled by the display driver)
        nn::psc::PmModuleId_Usb,         // nn::audio (Uac microphones depend on Usb)
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        audioPmModule.Initialize(
            nn::psc::PmModuleId_Audio,
            audioPmModuleDependencies,
            sizeof(audioPmModuleDependencies) / sizeof(audioPmModuleDependencies[0]),
            nn::os::EventClearMode_ManualClear
        )
    );

    while(g_IsAudioPmModuleHandlerThreadActive)
    {
        audioPmModule.GetEventPointer()->Wait();
        audioPmModule.GetEventPointer()->Clear();

        nn::psc::PmFlagSet   flags;
        nn::psc::PmState     state;
        nn::Result           result;

        NN_ABORT_UNLESS_RESULT_SUCCESS(audioPmModule.GetRequest(&state, &flags));

        switch (state)
        {
        case nn::psc::PmState_FullAwake:
            {
                // the audio process executes deferred wake processing
                // after sending a acknowledge to the psc process immediately.
                result = nn::ResultSuccess();
            }
            break;
        case nn::psc::PmState_MinimumAwake:
        case nn::psc::PmState_SleepReady:
            {
                result = SleepAudio();
            }
            break;
        case nn::psc::PmState_ShutdownReady:
            {
                result = ShutdownAudio();
            }
            break;
        default:
            {
                // no-op, but still return ResultSuccess()
                // for unhandled states - new states may be
                // added in the future.
                result = nn::ResultSuccess();
            }
            break;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(audioPmModule.Acknowledge(state, result));

        // deferred wake processing
        if(state == nn::psc::PmState_FullAwake)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(WakeAudio());
        }
    }
    audioPmModule.Finalize();
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
void HardwareOpusDecoderThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    do
    {
        nn::codec::server::HardwareOpusDecoder::Service::DoListening();

        nn::codec::server::HardwareOpusDecoder::Service::Wait();

    } while (!nn::codec::server::HardwareOpusDecoder::Service::IsFinalizing());
}

#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

void WakeAudioServers() NN_NOEXCEPT
{
    nn::audio::g_AudioRendererManager.GetImpl().Wake();
    nn::audio::g_AudioOutManager.GetImpl().Wake();
    nn::audio::g_AudioInManager.GetImpl().Wake();
    nn::audio::g_FinalOutputRecorderManager.GetImpl().Wake();
    nn::audio::AppletVolumeManager::Wake();
    nn::audioctrl::server::detail::WakeAudioControllerService();

    // wake HIPC servers
    nn::audio::server::WakeUserServiceServer();
    nn::audio::server::WakeAppletServiceServer();
    nn::audio::server::WakeAudioDebugServer();
}

void SleepAudioServers() NN_NOEXCEPT
{
    // sleep HIPC servers
    nn::audio::server::SleepAudioDebugServer();
    nn::audio::server::SleepAppletServiceServer();
    nn::audio::server::SleepUserServiceServer();

    nn::audioctrl::server::detail::SleepAudioControllerService();
    nn::audio::AppletVolumeManager::Sleep();
    nn::audio::g_FinalOutputRecorderManager.GetImpl().Sleep();
    nn::audio::g_AudioInManager.GetImpl().Sleep();
    nn::audio::g_AudioOutManager.GetImpl().Sleep();
    nn::audio::g_AudioRendererManager.GetImpl().Sleep();
}

nn::Result WakeAudio() NN_NOEXCEPT
{
    NN_AUDIO_PROCESS_LOG("[WakeAudio] Start (Hardware)\n");
    nn::audio::WakeAudioHardware();

    NN_AUDIO_PROCESS_LOG("[WakeAudio] Start (IPC Servers)\n");
    WakeAudioServers();

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_AUDIO_PROCESS_LOG("[WakeAudio] Start (HwOpus)\n");
    nn::codec::server::HardwareOpusDecoder::Service::Wake();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

    NN_AUDIO_PROCESS_LOG("[WakeAudio] End\n");

    NN_RESULT_SUCCESS;
}

nn::Result SleepAudio() NN_NOEXCEPT
{
    NN_AUDIO_PROCESS_LOG("[SleepAudio] Start (IPC Servers)\n");
    SleepAudioServers();

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_AUDIO_PROCESS_LOG("[SleepAudio] Start (HwOpus)\n");
    nn::codec::server::HardwareOpusDecoder::Service::Sleep();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

    NN_AUDIO_PROCESS_LOG("[SleepAudio] Start (Hardware)\n");
    nn::audio::SleepAudioHardware();

    NN_AUDIO_PROCESS_LOG("[SleepAudio] End\n");
    NN_RESULT_SUCCESS;
}

nn::Result ShutdownAudio() NN_NOEXCEPT
{
    NN_AUDIO_PROCESS_LOG("[ShutdownAudio] Start\n");

    // Fade out
    nn::audio::AppletVolumeManager::Sleep();

    // Shutdown minimum modules for reducing a noise.
    nn::audio::ShutdownAudioHardware();

    NN_AUDIO_PROCESS_LOG("[ShutdownAudio] End\n");

    NN_RESULT_SUCCESS;
}
} // namespace

extern "C" void nnMain()
{
    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start\n");

    // Load firmware debug settings
    nn::audio::server::LoadFirmwareDebugSettings();

    // Get platform configuration information
    const auto hasCodecIc = nn::audio::server::HasCodecIc();
    const auto hasHda = nn::audio::server::HasHda();

    // Get firmware debug settings
    const auto isUacEnabled = nn::audio::server::IsUacEnabled();

    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start (Hardware)\n");
    nn::audio::InitializeAudioHardware(hasCodecIc, hasHda);

    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start (AVM)\n");
    nn::audio::AppletVolumeManager::Initialize();

    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start (IPC Servers)\n");
    nn::audio::server::InitializeUserServiceServer();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_AudioUserServiceThread, AudioUserServiceThreadFunc, nullptr, g_AudioUserServiceThreadStack, sizeof(g_AudioUserServiceThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, UserServiceIpcServer)));
    nn::os::SetThreadNamePointer(&g_AudioUserServiceThread, NN_SYSTEM_THREAD_NAME(audio, UserServiceIpcServer));
    nn::os::StartThread(&g_AudioUserServiceThread);
    nn::audio::server::InitializeAppletServiceServer();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_AudioAppletServiceThread, AudioAppletServiceThreadFunc, nullptr, g_AudioAppletServiceThreadStack, sizeof(g_AudioAppletServiceThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, AppletServiceIpcServer)));
    nn::os::SetThreadNamePointer(&g_AudioAppletServiceThread, NN_SYSTEM_THREAD_NAME(audio, AppletServiceIpcServer));
    nn::os::StartThread(&g_AudioAppletServiceThread);

    nn::audio::server::InitializeAudioDebugServer();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_AudioDebuggerServiceThread, AudioDebuggerServiceThreadFunc, nullptr, g_AudioDebuggerServiceThreadStack, sizeof(g_AudioDebuggerServiceThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, DebugServiceIpcServer)));
    nn::os::SetThreadName(&g_AudioDebuggerServiceThread, NN_SYSTEM_THREAD_NAME(audio, DebugServiceIpcServer));
    nn::os::StartThread(&g_AudioDebuggerServiceThread);

    nn::audioctrl::server::InitializeAudioControllerServer(hasCodecIc, hasHda, isUacEnabled);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_AudioControllerThread, AudioControllerThreadFunc, nullptr, g_AudioControllerThreadStack, sizeof(g_AudioControllerThreadStack), NN_SYSTEM_THREAD_PRIORITY(audioctrl, AudioControllerIpcServer)));
    nn::os::SetThreadNamePointer(&g_AudioControllerThread, NN_SYSTEM_THREAD_NAME(audioctrl, AudioControllerIpcServer));
    nn::os::StartThread(&g_AudioControllerThread);

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start (HwOpus)\n");
    nn::codec::server::HardwareOpusDecoder::Service::Initialize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_HardwareOpusDecoderThread, HardwareOpusDecoderThreadFunc, nullptr, g_HardwareOpusDecoderThreadStack, sizeof(g_HardwareOpusDecoderThreadStack), NN_SYSTEM_THREAD_PRIORITY(codec, HardwareOpusDecoderIpcServer)));
    nn::os::SetThreadNamePointer(&g_HardwareOpusDecoderThread, NN_SYSTEM_THREAD_NAME(codec, HardwareOpusDecoderIpcServer));
    nn::os::StartThread(&g_HardwareOpusDecoderThread);
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

    // Setup psc (Power State Coordinator) handler thread
    // This thread must start in last to avoid a race condition.
    NN_AUDIO_PROCESS_LOG("[InitializeAudio] Start (PscHandler)\n");
    g_IsAudioPmModuleHandlerThreadActive = true;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_AudioPmModuleHandlerThread, AudioPmModuleHandlerThreadFunc, nullptr, g_AudioPmModuleHandlerThreadStack, sizeof(g_AudioPmModuleHandlerThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, PmModuleHandler)));
    nn::os::SetThreadNamePointer(&g_AudioPmModuleHandlerThread, NN_SYSTEM_THREAD_NAME(audio, PmModuleHandler));
    nn::os::StartThread(&g_AudioPmModuleHandlerThread);

    NN_AUDIO_PROCESS_LOG("[InitializeAudio] End\n");

    //----------------------------------------
    // 終了処理ステージ
    //----------------------------------------
    nn::os::WaitThread(&g_AudioUserServiceThread);
    nn::os::DestroyThread(&g_AudioUserServiceThread);
    nn::audio::AppletVolumeManager::Finalize();
    nn::audio::server::FinalizeUserServiceServer();

    nn::os::WaitThread(&g_AudioAppletServiceThread);
    nn::os::DestroyThread(&g_AudioAppletServiceThread);
    nn::audio::server::FinalizeAppletServiceServer();

    nn::os::WaitThread(&g_AudioDebuggerServiceThread);
    nn::os::DestroyThread(&g_AudioDebuggerServiceThread);
    nn::audio::server::FinalizeAudioDebugServer();

    nn::os::WaitThread(&g_AudioControllerThread);
    nn::os::DestroyThread(&g_AudioControllerThread);
    nn::audioctrl::server::FinalizeAudioControllerServer();

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::codec::server::HardwareOpusDecoder::Service::Exit();
    nn::os::WaitThread(&g_HardwareOpusDecoderThread);
    nn::os::DestroyThread(&g_HardwareOpusDecoderThread);
    nn::codec::server::HardwareOpusDecoder::Service::Finalize();
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX)

    g_IsAudioPmModuleHandlerThreadActive = false;
    nn::os::WaitThread(&g_AudioPmModuleHandlerThread);
    nn::os::DestroyThread(&g_AudioPmModuleHandlerThread);

    nn::audio::FinalizeAudioHardware();
}
