﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/psc.h>

#include <nn/os/os_Thread.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/tcap/detail/tcap_Log.h>
#include <nn/tcap/tcap_ServiceName.h>
#include <nn/tcap/server/tcap_Api.h>
#include <nn/tcap/server/tcap_IManager.sfdl.h>
#include <nn/tcap/server/tcap_Lib.h>
#include <nn/tcap/server/tcap_ManagerImpl.h>
#include <nn/tcap/tcap_Lib.h>
#include <nn/tcap/tcap_ShimInternal.h>

#include "am_TcapServer.h"

namespace nn { namespace am {

namespace {

const int TcapMaxSession = 3; // DevMenu, application + 1(margin) (spsm と omm が使用するが同一プロセスであるため考慮不要)
const int TcapSessionCountMax = TcapMaxSession; // Manager 以外のセッションは存在しない為同値で良い
const int TcapPortCountMax = 1;
class TcapServerManager : public nn::sf::HipcSimpleAllInOneServerManager<TcapSessionCountMax, TcapPortCountMax> { };

TcapServerManager g_TcapServerManager;

sf::UnmanagedServiceObject<nn::tcap::server::IManager, nn::tcap::server::ManagerImpl> g_TcapManager;

nn::os::ThreadType              g_TcapServerThread;
NN_OS_ALIGNAS_THREAD_STACK char g_TcapServerThreadStack[0x2000];

nn::psc::PmModule g_PmModule;

void HandleTcapServer(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    NN_DETAIL_TCAP_INFO("Starting tcap server\n");

    g_TcapServerManager.Start();

    const nn::psc::PmModuleId PmDependencies[] =
    {
        // TC はいつでも値を返せるので依存していない扱いとする
        nn::psc::PmModuleId_Hid,
    };
    auto result = g_PmModule.Initialize(
        nn::psc::PmModuleId_Tcap,
        PmDependencies,
        NN_ARRAY_SIZE(PmDependencies),
        nn::os::EventClearMode_ManualClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_DETAIL_TCAP_INFO("Started tcap server\n");

    const uintptr_t UpdateTcapServerTag = 0x80000001;
    const uintptr_t PmModuleEventStatusTag = 0x80000002;

    nn::os::MultiWaitHolderType pmEventHolder;

    nn::os::InitializeMultiWaitHolder(&pmEventHolder, g_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&pmEventHolder, PmModuleEventStatusTag);

    g_TcapServerManager.AddUserWaitHolder(&pmEventHolder);

    nn::os::TimerEventType timerEvent;
    nn::os::MultiWaitHolderType timerEventHolder;

    nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    nn::os::SetMultiWaitHolderUserData(&timerEventHolder, UpdateTcapServerTag);

    // TCAP の更新間隔も TC の更新間隔と同じ 1 秒にする
    const auto UpdateTcapServerInterval = nn::TimeSpan::FromMilliSeconds(1000);
    nn::os::StartPeriodicTimerEvent(&timerEvent, UpdateTcapServerInterval, UpdateTcapServerInterval);

    g_TcapServerManager.AddUserWaitHolder(&timerEventHolder);

    while ( auto p = g_TcapServerManager.Wait() )
    {
        uintptr_t userData =  nn::os::GetMultiWaitHolderUserData(p);
        switch ( userData )
        {
        case TcapServerManager::InvokeTag:
        case TcapServerManager::AcceptTag:
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_TcapServerManager.ProcessAuto(p));
            break;
        case UpdateTcapServerTag:
            nn::os::ClearTimerEvent(&timerEvent);
            g_TcapServerManager.AddUserWaitHolder(p);
            nn::tcap::server::Update(UpdateTcapServerInterval);
            break;
        case PmModuleEventStatusTag:
            {
                bool tryWaitResult = g_PmModule.GetEventPointer()->TryWait();
                NN_SDK_ASSERT(tryWaitResult);
                NN_UNUSED(tryWaitResult);
            }
            g_PmModule.GetEventPointer()->Clear();
            g_TcapServerManager.AddUserWaitHolder(&pmEventHolder);

            nn::psc::PmState state;
            nn::psc::PmFlagSet flags;
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_PmModule.GetRequest(&state, &flags));

            switch ( state )
            {
            case nn::psc::PmState_MinimumAwake:
                nn::os::StartPeriodicTimerEvent(&timerEvent, UpdateTcapServerInterval, UpdateTcapServerInterval);
                break;
            case nn::psc::PmState_SleepReady:
            case nn::psc::PmState_ShutdownReady:
                nn::os::StopTimerEvent(&timerEvent);
                nn::os::ClearTimerEvent(&timerEvent);
                nn::tcap::server::ClearElapsedTime();
                break;
            default:
                break;
            }

            g_PmModule.Acknowledge(state, ResultSuccess());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::os::StopTimerEvent(&timerEvent);
    nn::os::FinalizeMultiWaitHolder(&timerEventHolder);
    nn::os::FinalizeTimerEvent(&timerEvent);

    nn::os::FinalizeMultiWaitHolder(&pmEventHolder);
    g_PmModule.Finalize();

    nn::tcap::Finalize();
    nn::tcap::server::Finalize();
}

} // namespace

void StartTcapServer() NN_NOEXCEPT
{
    g_TcapServerManager.RegisterObjectForPort(g_TcapManager.GetShared(), TcapMaxSession, nn::tcap::ServiceName);

    // SPSM 初期化前に TCAP の Shim の初期化を保証するためにここで初期化する
    // ブロックされるリスクを減らすため本来は RegisterObjectForPort とスレッド生成処理以外はメインスレッドで行うべきではない
    nn::tcap::server::Initialize();
    nn::tcap::InitializeWith(g_TcapManager.GetShared());

    auto result = nn::os::CreateThread(&g_TcapServerThread, HandleTcapServer,nullptr,
        g_TcapServerThreadStack, sizeof(g_TcapServerThreadStack), NN_SYSTEM_THREAD_PRIORITY(tcap, IpcServer));

    nn::os::SetThreadNamePointer(&g_TcapServerThread, NN_SYSTEM_THREAD_NAME(tcap, IpcServer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread(&g_TcapServerThread);
}

}}  // namespace nn::am
