﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_AccountDaemonTaskScheduler.h"

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>

namespace nn {
NN_DEFINE_STATIC_CONSTANT(const int32_t AccountDaemonTaskScheduler::InitialScheduleDelaySeconds);
NN_DEFINE_STATIC_CONSTANT(const int32_t AccountDaemonTaskScheduler::DefaultSchedulePeriodicitySeconds);
NN_DEFINE_STATIC_CONSTANT(const int32_t AccountDaemonTaskScheduler::DefaultProfileSynchronizationIntervalSeconds);
NN_DEFINE_STATIC_CONSTANT(const int32_t AccountDaemonTaskScheduler::DefaultNintendoAccountInfoRefreshIntervalSeconds);

namespace {
NN_FORCEINLINE void WaitAsyncContextHasDone(account::AsyncContext& ctx, CancelPoint& cp) NN_NOEXCEPT
{
    CancelPoint::Attachment attachment(cp, ctx);
    os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();

#if !defined(NN_SDK_BUILD_RELEASE)
    bool done;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.HasDone(&done));
    NN_ABORT_UNLESS(done);
#endif
}

#define EXEC_OPERATION(operation, uid, cp) \
    do \
    { \
        NN_RESULT_THROW_UNLESS(!cp.Test(), account::ResultCancelled()); \
        auto _r = operation(uid, cp); \
        NN_RESULT_DO(HandleError(_r, #operation, uid)); \
    } while (NN_STATIC_CONDITION(false))

NN_FORCEINLINE Result HandleError(const Result& r, const char* opearation, const account::Uid& uid) NN_NOEXCEPT
{
    if (!r.IsSuccess())
    {
        if (!account::ResultCancelled::Includes(r))
        {
            NN_ACCOUNT_ERROR_WITH_TIMESTAMP(
                "%s() failed with %03d-%04d\n",
                opearation, r.GetModule(), r.GetDescription());
        }
        // NSA を利用できないエラーは無視する
        NN_RESULT_THROW_UNLESS(account::ResultNetworkServiceAccountUnavailable::Includes(r), r);
    }
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::<anonymous>

void AccountDaemonTaskScheduler::ScheduleImpl() NN_NOEXCEPT
{
    auto SetTimer = [this](TimeSpan v) -> void {
        if (m_Event.TryWait())
        {
            auto t = os::GetSystemTick().ToTimeSpan() - m_NextExecution;
            NN_UNUSED(t);
            // 既にシグナル済みの場合何もしない
            NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                "Already signaled %02dh%02dm%02ds ago (estimate)\n",
                t.GetHours(), t.GetMinutes() % 60, t.GetSeconds() % 60);
            return;
        }

        m_Event.StartOneShot(v);
        m_NextExecution = os::GetSystemTick().ToTimeSpan() + v;
        NN_ACCOUNT_INFO_WITH_TIMESTAMP(
            "Next execution is scheduled after %02dh%02dm%02ds\n",
            v.GetHours(), v.GetMinutes() % 60, v.GetSeconds() % 60);
    };

    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    if (!m_pLastExecution)
    {
        SetTimer(std::min(TimeSpan::FromSeconds(InitialScheduleDelaySeconds), m_Interval));
        return;
    }
    auto passage = os::GetSystemTick().ToTimeSpan() - *m_pLastExecution;
    if (!(passage < m_Interval))
    {
        RescheduleImmediatelyImpl();
        return;
    }
    SetTimer(m_Interval - passage);
}
void AccountDaemonTaskScheduler::RescheduleImmediatelyImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Lock.IsLockedByCurrentThread());
    if (m_Event.TryWait())
    {
        auto t = os::GetSystemTick().ToTimeSpan() - m_NextExecution;
        NN_UNUSED(t);
        // 既にシグナル済みの場合何もしない
        NN_ACCOUNT_INFO_WITH_TIMESTAMP(
            "Already signaled %02dh%02dm%02ds ago (estimate)\n",
            t.GetHours(), t.GetMinutes() % 60, t.GetSeconds() % 60);
        return;
    }

    m_Event.Signal();
    m_NextExecution = os::GetSystemTick().ToTimeSpan();
    NN_ACCOUNT_INFO_WITH_TIMESTAMP("Next execution is scheduled immediately\n");
}
void AccountDaemonTaskScheduler::UpdateLastExecutionTime() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    auto current = os::GetSystemTick().ToTimeSpan();
    m_pLastExecution.emplace(current);
    ScheduleImpl();
}
void AccountDaemonTaskScheduler::UpdateInterval(TimeSpan interval) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    if (m_Interval == interval)
    {
        return;
    }
    NN_ACCOUNT_INFO_WITH_TIMESTAMP(
        "Interval updated: %02dh%02dm%02ds -> %02dh%02dm%02ds\n",
        m_Interval.GetHours(), m_Interval.GetMinutes() % 60, m_Interval.GetSeconds() % 60,
        interval.GetHours(), interval.GetMinutes() % 60, interval.GetSeconds() % 60);
    m_Interval = interval;
    ScheduleImpl();
}
void AccountDaemonTaskScheduler::RescheduleImmediately() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    RescheduleImmediatelyImpl();
}
TimeSpan AccountDaemonTaskScheduler::GetNextExecutionInUptime() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    return m_NextExecution;
}

Result AccountDaemonTaskScheduler::UpdateUsers(int* pOutCount) NN_NOEXCEPT
{
    account::Uid users[account::UserCountMax];
    int actualUserCount;
    NN_RESULT_DO(account::ListAllUsers(&actualUserCount, users, std::extent<decltype(users)>::value));

    bool hasChanged = false;
    for (auto i = 0; i < std::extent<decltype(users)>::value; ++ i)
    {
        if (i < actualUserCount
            ? !(m_UserStates[i].uid == users[i])
            : !(!m_UserStates[i].uid))
        {
            // ユーザーリストに変化があった
            hasChanged = true;
        }
    }
    if (hasChanged)
    {
        const auto StateArraySize = std::extent<decltype(m_UserStates)>::value;
        UserState states[StateArraySize];
        for (int i = 0; i < actualUserCount; ++ i)
        {
            bool found = false;
            for (auto j = 0; j < StateArraySize && !found && m_UserStates[j].uid; ++ j)
            {
                if (m_UserStates[j].uid == users[i])
                {
                    states[i] = m_UserStates[j];
                    found = true;
                }
            }
            if (!found)
            {
                states[i].Reset(users[i]);
            }
        }
        for (auto i = actualUserCount; i < StateArraySize; ++ i)
        {
            states[i].Reset(account::InvalidUid);
        }

        std::memcpy(m_UserStates, states, sizeof(m_UserStates));
    }

    *pOutCount = actualUserCount;
    NN_RESULT_SUCCESS;
}

Result AccountDaemonTaskScheduler::ExecuteScheduledTasksImpl(CancelPoint& cp) NN_NOEXCEPT
{
    auto currentUptime = os::GetSystemTick().ToTimeSpan().GetSeconds();
    NN_UNUSED(currentUptime);

    nifm::NetworkConnection con;
    NN_RESULT_DO(ConnectToInternet(con, cp));

    // Per-user
    int userCount;
    NN_RESULT_DO(UpdateUsers(&userCount));
    for (auto i = 0; i < userCount; ++ i)
    {
        if (!con.IsAvailable())
        {
            NN_RESULT_DO(ConnectToInternet(con, cp));
        }

        // -----------------------------------------------------
        // Nintendo Account の状態の同期
        auto& state = m_UserStates[i];
        EXEC_OPERATION(RefreshNintendoAccountStatus, state.uid, cp);

        // -----------------------------------------------------
        // プロフィール同期
        account::ProfileEditor editor;
        NN_RESULT_DO(account::GetProfileEditor(&editor, state.uid));
        bool immediate = false;
        if (editor.IsLocalOrigin())
        {
            auto timeStamp = editor.GetTimeStamp();
            if (!(true
                && state.lastExecutionState.profileUpdate.first
                && timeStamp == state.lastExecutionState.profileUpdate.second))
            {
                // その本体で作成されたプロフィールで、過去に同期したプロフィールと異なっていれば即時同期
                EXEC_OPERATION(SynchronizeProfileImmediately, state.uid, cp);
                state.lastExecutionState.profileUpdate = std::pair<bool, uint64_t>(true, timeStamp);
                immediate = true;
            }
        }

        if (!immediate)
        {
            EXEC_OPERATION(SynchronizeProfile, state.uid, cp);
        }
    }
    NN_RESULT_SUCCESS;
}

Result AccountDaemonTaskScheduler::RefreshNintendoAccountStatus(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT
{
    account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(account::GetNetworkServiceAccountAdministrator(&admin, uid));

    // NSA 利用可能かつ、 NA 連携済みの場合のみ同期する
    bool isNsaAvailable;
    NN_RESULT_DO(admin.IsNetworkServiceAccountAvailable(&isNsaAvailable));
    if (!isNsaAvailable)
    {
        NN_RESULT_SUCCESS;
    }
    bool isLinked;
    NN_RESULT_DO(admin.IsLinkedWithNintendoAccount(&isLinked));
    if (!isLinked)
    {
        NN_RESULT_SUCCESS;
    }

    // 更新処理
    bool matched;
    account::AsyncContext ctx;

    // NA のユーザー情報のキャッシュ
    NN_RESULT_DO(admin.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(
        &matched, &ctx, TimeSpan::FromSeconds(m_Settings.naInfoRefreshInterval)));
    if (matched)
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        auto uidRaw = reinterpret_cast<const uint32_t*>(&uid);
        NN_ACCOUNT_INFO_WITH_TIMESTAMP(
            "Periodical NA-info-refresh for %08x_%08x_%08x_%08x (%02dh%02dm%02ds)\n",
            uidRaw[1], uidRaw[0], uidRaw[3], uidRaw[2],
            m_Settings.naInfoRefreshInterval / (60 * 60),
            (m_Settings.naInfoRefreshInterval / 60) % 60,
            m_Settings.naInfoRefreshInterval % 60);
#endif

        WaitAsyncContextHasDone(ctx, cp);
        NN_RESULT_DO(ctx.GetResult());
    }

    // OP2 加入状態
    NN_RESULT_DO(admin.RefreshCachedNetworkServiceLicenseInfoAsyncIfTimeElapsed(
        &matched, &ctx, TimeSpan::FromSeconds(m_Settings.naInfoRefreshInterval)));
    if (matched)
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        auto uidRaw = reinterpret_cast<const uint32_t*>(&uid);
        NN_ACCOUNT_INFO_WITH_TIMESTAMP(
            "Periodical License-status-refresh for %08x_%08x_%08x_%08x (%02dh%02dm%02ds)\n",
            uidRaw[1], uidRaw[0], uidRaw[3], uidRaw[2],
            m_Settings.naInfoRefreshInterval / (60 * 60),
            (m_Settings.naInfoRefreshInterval / 60) % 60,
            m_Settings.naInfoRefreshInterval % 60);
#endif

        WaitAsyncContextHasDone(ctx, cp);
        NN_RESULT_DO(ctx.GetResult());
    }

    NN_RESULT_SUCCESS;
}

Result AccountDaemonTaskScheduler::SynchronizeProfileImmediately(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT
{
    account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool isNsaAvailable;
    NN_RESULT_DO(admin.IsNetworkServiceAccountAvailable(&isNsaAvailable));
    if (isNsaAvailable)
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        auto uidRaw = reinterpret_cast<const uint32_t*>(&uid);
        NN_ACCOUNT_INFO_WITH_TIMESTAMP(
            "Immediate profile-sync for %08x_%08x_%08x_%08x\n",
            uidRaw[1], uidRaw[0], uidRaw[3], uidRaw[2]);
#endif
        // NSA 利用可能の場合のみ同期する
        account::AsyncContext ctx;
        NN_RESULT_DO(admin.SynchronizeProfileAsync(&ctx));
        WaitAsyncContextHasDone(ctx, cp);
        return ctx.GetResult();
    }
    NN_RESULT_SUCCESS;
}

Result AccountDaemonTaskScheduler::SynchronizeProfile(const account::Uid& uid, CancelPoint& cp) NN_NOEXCEPT
{
    account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool isNsaAvailable;
    NN_RESULT_DO(admin.IsNetworkServiceAccountAvailable(&isNsaAvailable));
    if (isNsaAvailable)
    {
        // NSA 利用可能の場合のみ同期する
        bool matched;
        account::AsyncContext ctx;
        NN_RESULT_DO(admin.SynchronizeProfileAsyncIfTimeElapsed(
            &matched, &ctx, TimeSpan::FromSeconds(m_Settings.profileSyncInterval)));
        if (matched)
        {
#if !defined(NN_SDK_BUILD_RELEASE)
            auto uidRaw = reinterpret_cast<const uint32_t*>(&uid);
            NN_ACCOUNT_INFO_WITH_TIMESTAMP(
                "Periodical profile-sync for %08x_%08x_%08x_%08x (%02dh%02dm%02ds)\n",
                uidRaw[1], uidRaw[0], uidRaw[3], uidRaw[2],
                m_Settings.profileSyncInterval / (60 * 60),
                (m_Settings.profileSyncInterval / 60) % 60,
                m_Settings.profileSyncInterval % 60);
#endif
            WaitAsyncContextHasDone(ctx, cp);
            return ctx.GetResult();
        }
    }
    NN_RESULT_SUCCESS;
}

} // ~namespace nn
