﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/pinmux/pinmux_ShimInternal.h>
#include <nn/pinmux/driver/pinmux_Lib.h>
#include <nn/pinmux/server/pinmux_Server.h>

#include "../detail/Bus_Log.h"
#include "Bus_PinmuxServer.h"

namespace nn { namespace bus { namespace pinmux {

namespace
{
    const size_t ThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    //! サーバのループスレッド
    void ServerThread(void *)
    {
        // 追加のモジュール初期化処理（他モジュールへの依存のためにメインスレッド内では呼ばないほうが良いもの）
        nn::pinmux::driver::Initialize();
        nn::pinmux::server::StartHipcServer();

        NN_DETAIL_BUS_INFO("[Bus] Entering Pinmux Server Loop\n");
        nn::pinmux::server::LoopHipcServer();

        // モジュール終了処理
        nn::pinmux::driver::Finalize();
    };
}

//! プロセスメインスレッド内で早い段階で同期的に済ませておく必要がある処理（プロセス内他スレッドが起動直後に API を叩いて来たら競合しうるもの）
void InitializeServer() NN_NOEXCEPT
{
    // sm への HIPC サーバの登録
    nn::pinmux::server::RegisterHipcServer();

    // プロセス内シムへのサービスオブジェクト登録
    nn::pinmux::InitializeWith(nn::pinmux::server::GetServiceObject());
}

void StartServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_Thread, ServerThread, nullptr, g_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(pinmux, IpcServer))
    );
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(pinmux, IpcServer));
    nn::os::StartThread(&g_Thread);
}

void StopServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    nn::pinmux::server::StopHipcServer();
}

void WaitAndFinalizeServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}
