﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/i2c/driver/i2c_Lib.h>
#include <nn/i2c/server/i2c_Server.h>

#include "../detail/Bus_Log.h"
#include "Bus_I2cServer.h"

namespace nn { namespace bus { namespace i2c {

namespace
{
    const size_t ThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    const size_t ThreadStackSizePowerBus = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStackPowerBus[ThreadStackSize];
    nn::os::ThreadType g_ThreadPowerBus;

    //! PCV 向けサーバのループスレッド
    void ServerThreadPowerBus(void *)
    {
        NN_DETAIL_BUS_INFO("[Bus] Entering I2c PowerBus Server Loop\n");
        nn::i2c::server::LoopHipcServerPowerBus();
    };

    //! 通常用途向けサーバのループスレッド
    void ServerThread(void *)
    {
        // 追加のモジュール初期化処理（他モジュールへの依存のためにメインスレッド内では呼ばないほうが良いもの）
        nn::i2c::driver::Initialize();

        nn::i2c::server::StartHipcServer();

        // ドライバの初期化が完了したところでもう一つのサーバスレッドを開始
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
                &g_ThreadPowerBus, ServerThreadPowerBus, nullptr, g_ThreadStackPowerBus, ThreadStackSizePowerBus,
                NN_SYSTEM_THREAD_PRIORITY(i2c, ForPcvIpcServer))
        );
        nn::os::SetThreadNamePointer(&g_ThreadPowerBus, NN_SYSTEM_THREAD_NAME(i2c, ForPcvIpcServer));
        nn::os::StartThread(&g_ThreadPowerBus);

        NN_DETAIL_BUS_INFO("[Bus] Entering I2c Server Loop\n");
        nn::i2c::server::LoopHipcServer();

        // もう一つのサーバスレッドの終了を待機してからでないとドライバの終了処理は行わない
        nn::os::WaitThread(&g_ThreadPowerBus);
        nn::os::DestroyThread(&g_ThreadPowerBus);

        // モジュール終了処理
        nn::i2c::driver::Finalize();
    };
}

//! プロセスメインスレッド内で早い段階で同期的に済ませておく必要がある処理（プロセス内他スレッドが起動直後に API を叩いて来たら競合しうるもの）
void InitializeServer() NN_NOEXCEPT
{
    // sm への HIPC サーバの登録
    nn::i2c::server::RegisterHipcServer();

    // プロセス内シムへのサービスオブジェクト登録
    // nn::i2c::InitializeWith(nn::i2c::server::GetServiceObject());
}

void StartServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_Thread, ServerThread, nullptr, g_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(i2c, IpcServer))
    );
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(i2c, IpcServer));
    nn::os::StartThread(&g_Thread);
}

void StopServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    nn::i2c::server::StopHipcServer();
}

void WaitAndFinalizeServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}
