﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/gpio/gpio_ShimInternal.h>
#include <nn/gpio/driver/gpio_Lib.h>
#include <nn/gpio/server/gpio_Server.h>

#include "../detail/Bus_Log.h"
#include "Bus_GpioServer.h"

namespace nn { namespace bus { namespace gpio {

namespace
{
    const size_t ThreadStackSize = 4 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;

    //! サーバのループスレッド
    void ServerThread(void *)
    {
        // 追加のモジュール初期化処理（他モジュールへの依存のためにメインスレッド内では呼ばないほうが良いもの）
        nn::gpio::driver::Initialize();
        nn::gpio::server::StartHipcServer();

        NN_DETAIL_BUS_INFO("[Bus] Entering Gpio Server Loop\n");
        nn::gpio::server::LoopHipcServer();

        // モジュール終了処理
        nn::gpio::driver::Finalize();
    };
}

//! プロセスメインスレッド内で早い段階で同期的に済ませておく必要がある処理（プロセス内他スレッドが起動直後に API を叩いて来たら競合しうるもの）
void InitializeServer() NN_NOEXCEPT
{
    // sm への HIPC サーバの登録
    nn::gpio::server::RegisterHipcServer();

    // プロセス内シムへのサービスオブジェクト登録
    nn::gpio::InitializeWith(nn::gpio::server::GetServiceObject());
}

void StartServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_Thread, ServerThread, nullptr, g_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(gpio, IpcServer))
    );
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(gpio, IpcServer));
    nn::os::StartThread(&g_Thread);
}

void StopServer() NN_NOEXCEPT
{
    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    nn::gpio::server::StopHipcServer();
}

void WaitAndFinalizeServer() NN_NOEXCEPT
{
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}
