﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/hid/detail/hid_RxPacketHistoryTypes.h>
#include <nn/os/os_Tick.h>
#include <nn/xcd/xcd_Device.h>

#include "xcd_FirmwareVersionTypes.h"

namespace nn { namespace xcd {

//!< 通信状況をモニタリングするクラス
class TransactionMonitor
{

private:
    //!< 有効化されているかどうか
    bool m_IsActivated;

    //!< デバイスの種類
    DeviceInfo m_Info;

    //!< ファームウェアのバージョン
    FirmwareVersionImpl m_FirmwareVersion;

    //!< 通信間隔
    ::nn::TimeSpan m_Interval;

    //!< 最後に処理したサンプル番号 ver3.43 以降で使用
    uint8_t m_LastSampleNumberFromDevice;

    //!< 最後にセンサーがサンプリングされたタイミング ver3.42 以前で使用
    ::nn::os::Tick m_SensorLastSampleTime;

    //!< デバイスから受信するサンプル番号の最大値
    static const uint16_t SampleNumberFromDeviceMax = 0x100; // 0xff + 1

    //!< パケット受信履歴の累計カウント
    uint64_t m_RxPacketHistoryCount;

    //!< パケット履歴を保存する最大数
    static const int PacketHistoryCountMax = nn::hid::detail::PacketHistoryCount + 1;

    //!< 上位に渡すためのリザルト
    nn::hid::detail::PacketInfo m_RxPacketHistory[PacketHistoryCountMax];

public:
    TransactionMonitor() NN_NOEXCEPT;
    ~TransactionMonitor() NN_NOEXCEPT;

    //!< デバイスが接続されたときに呼ばれる関数
    void Activate(DeviceInfo& info, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT;

    //!< Ble デバイスが接続されたときに呼ばれる関数
    void ActivateBleDevice() NN_NOEXCEPT;

    //!< 有効化されているかどうか
    bool IsActivated() NN_NOEXCEPT;

    //!< デバイスが切断された時に呼ばれる関数
    void Deactivate() NN_NOEXCEPT;

    //!< Packet の受信状況を更新する
    void UpdateRxMonitoringResult(int* pOutGap, uint8_t sampleNumber, ::nn::TimeSpan interval) NN_NOEXCEPT;

    //!< Packet の受信状況を取得する
    nn::hid::detail::RxPacketHistory GetRxPacketHistory() NN_NOEXCEPT;
};

}} // namespace nn::xcd
