﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_TransactionMonitor.h"

namespace nn { namespace xcd {

namespace
{
    static auto SampleUnitInterval = ::nn::TimeSpan::FromMilliSeconds(5);
}

TransactionMonitor::TransactionMonitor() NN_NOEXCEPT :
    m_LastSampleNumberFromDevice(0),
    m_SensorLastSampleTime(0)
{
    for (auto& history : m_RxPacketHistory)
    {
        history.status = nn::hid::detail::PacketStatus_Dropped;
        history.interval = 0;
    }
}

TransactionMonitor::~TransactionMonitor() NN_NOEXCEPT
{
    // 何もしない
}

void TransactionMonitor::Activate(DeviceInfo& info, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    if (m_IsActivated == false)
    {
        m_Info = info;
        m_FirmwareVersion = firmwareVersion;

        // デバイスからのサンプルナンバーの初期化
        m_LastSampleNumberFromDevice = 0;

        m_RxPacketHistoryCount = std::numeric_limits<uint64_t>().min();

        m_IsActivated = true;
    }
}

void TransactionMonitor::ActivateBleDevice() NN_NOEXCEPT
{
    if (m_IsActivated == false)
    {
        m_Info.deviceType = DeviceType_Unknown;
        m_FirmwareVersion.bluetooth = FirmwareVersionBt_0343;

        // デバイスからのサンプルナンバーの初期化
        m_LastSampleNumberFromDevice = 0;

        m_RxPacketHistoryCount = std::numeric_limits<uint64_t>().min();

        m_IsActivated = true;
    }
}

bool TransactionMonitor::IsActivated() NN_NOEXCEPT
{
    return m_IsActivated;
}

void TransactionMonitor::Deactivate() NN_NOEXCEPT
{
    if (m_IsActivated == true)
    {
        m_IsActivated = false;
    }
}

void TransactionMonitor::UpdateRxMonitoringResult(int* pOutGap, uint8_t sampleNumber, ::nn::TimeSpan interval) NN_NOEXCEPT
{
    *pOutGap = 0;

    if (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_0343 ||
        m_Info.deviceType == DeviceType_MiyabiLeft ||
        m_Info.deviceType == DeviceType_MiyabiRight )
    {
        // サンプルナンバーの差異でサンプル数を計算
        uint32_t const SampleNumberFromDeviceModulo = SampleNumberFromDeviceMax;
        *pOutGap = ((uint32_t(sampleNumber) + SampleNumberFromDeviceModulo) - m_LastSampleNumberFromDevice) % SampleNumberFromDeviceModulo;
        m_LastSampleNumberFromDevice = sampleNumber;
    }
    else
    {
        // 前回からのサンプル受信時間の差異でサンプル数を予測
        auto currentTick = ::nn::os::GetSystemTick();
        *pOutGap = static_cast<int>(((currentTick - m_SensorLastSampleTime).ToTimeSpan().GetMilliSeconds() + 1) / 5);
        m_SensorLastSampleTime = currentTick;
    }

    if (interval == 0)
     {
         return;
     }

    auto slotsSinceLast = *pOutGap / (interval.GetMilliSeconds() / SampleUnitInterval.GetMilliSeconds());

    for (int i = 0; i < slotsSinceLast; i++)
    {
        auto index = m_RxPacketHistoryCount % PacketHistoryCountMax;
        m_RxPacketHistory[index].status = (i == slotsSinceLast - 1) ? nn::hid::detail::PacketStatus_Received : nn::hid::detail::PacketStatus_Dropped;
        m_RxPacketHistory[index].interval = static_cast<uint8_t>(interval.GetMilliSeconds());
        m_RxPacketHistoryCount++;
    }
}

nn::hid::detail::RxPacketHistory TransactionMonitor::GetRxPacketHistory() NN_NOEXCEPT
{
    nn::hid::detail::RxPacketHistory returnVal = {};
    returnVal.samplingCount = m_RxPacketHistoryCount;
    auto currentIndexOnBuffer = m_RxPacketHistoryCount % PacketHistoryCountMax;
    auto count = std::min(static_cast<uint64_t>(nn::hid::detail::PacketHistoryCount), returnVal.samplingCount);
    for (int i = 0; i < count; ++i)
    {
        auto index = (PacketHistoryCountMax + currentIndexOnBuffer - 1 - i) % PacketHistoryCountMax;
        returnVal.history[i] = m_RxPacketHistory[index];
    }
    return returnVal;
}

}} // namespace nn::xcd
