﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_TickTypes.h>
#include "xcd_IEventTask.h"
#include "xcd_MultiWaitEvent.h"

namespace nn { namespace xcd {

// EventTask を保持するリストの最大長
//  - Bluetooth x1, Uart x4 (Recv x2, SendTimer x2) が最大のはず
const int EventTaskListCountMax = 8;

// 定常処理のタスクを保持するリストの最大長
const int PeriodicTaskCountMax = 100;

//! Event駆動でタスクを処理するスレッドを扱うクラスです
class TaskManager
{
    NN_DISALLOW_MOVE(TaskManager);
    NN_DISALLOW_COPY(TaskManager);

private:
    //!< 多重待ち
    nn::os::MultiWaitType*  m_pMultiWait;

    IMultiWaitBase* m_pEventTaskList[EventTaskListCountMax];
    IEventTask*     m_pPeriodicTaskList[PeriodicTaskCountMax];
    int             m_EventTaskCount;
    int             m_PeriodicTaskCount;

    //!< 直近で定常処理を実行した Tick
    nn::os::Tick    m_LastPeriodicEventTick;

    bool            m_Activated;

public:
    TaskManager() NN_NOEXCEPT;
    ~TaskManager() NN_NOEXCEPT;

    void Activate(nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    bool Proceed(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;

    nn::os::MultiWaitHolderType* WaitEvent() NN_NOEXCEPT;

    void ProceedPeriodic() NN_NOEXCEPT;

    void RegisterEvent(IMultiWaitBase* pEvent, IEventTask* pEventTask) NN_NOEXCEPT;

    void RegisterPeriodicTask(IEventTask* pEventTask) NN_NOEXCEPT;

    void UnregisterEvent(IMultiWaitBase* pEvent) NN_NOEXCEPT;

    void UnregisterPeriodicTask(IEventTask* pEventTask) NN_NOEXCEPT;
};

//!< タスクマネージャーを取得する
TaskManager& GetTaskManager() NN_NOEXCEPT;

}} // namespace nn::xcd
