﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_TaskManager.h"
#include "xcd_IEventTask.h"

#if 0
#define NN_XCD_TASKMANAGER_DUMP(...)    NN_DETAIL_XCD_WARN(__VA_ARGS__)
#else
#define NN_XCD_TASKMANAGER_DUMP(...)    static_cast<void>(0)
#endif

namespace nn { namespace xcd {

namespace
{

// 定常処理イベント発生間隔
const auto PeriodicEventInterval = nn::TimeSpan::FromMilliSeconds(30);

}  // anonymous

TaskManager::TaskManager() NN_NOEXCEPT
    : m_pMultiWait(nullptr)
    , m_pEventTaskList()
    , m_pPeriodicTaskList()
    , m_EventTaskCount(0)
    , m_PeriodicTaskCount(0)
    , m_LastPeriodicEventTick()
    , m_Activated(false)
{
    // 何もしない
}

TaskManager::~TaskManager() NN_NOEXCEPT
{
    // 何もしない
}

void TaskManager::Activate(nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Activated, false);

    m_pMultiWait        = pMultiWait;
    m_EventTaskCount    = 0;
    m_PeriodicTaskCount = 0;

    m_Activated = true;
}

void TaskManager::Deactivate() NN_NOEXCEPT
{
    m_Activated = false;
}

void TaskManager::RegisterEvent(IMultiWaitBase* pEvent, IEventTask* pEventTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    for (auto* pItem : m_pEventTaskList)
    {
        if (pItem == pEvent)
        {
            // 登録済み
            return;
        }
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_pEventTaskList); i++)
    {
        if (m_pEventTaskList[i] == nullptr)
        {
            m_pEventTaskList[i] = pEvent;
            pEvent->Link(m_pMultiWait);
            pEvent->SetUserData(reinterpret_cast<uintptr_t>(pEventTask));
            m_EventTaskCount++;
            NN_XCD_TASKMANAGER_DUMP("%s: %d -> %d\n", NN_CURRENT_FUNCTION_NAME, m_EventTaskCount - 1, m_EventTaskCount);
            return;
        }
    }

    NN_DETAIL_XCD_ERROR("Exceed maximum number of events\n");
}

void TaskManager::RegisterPeriodicTask(IEventTask* pEventTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventTask);

    for (auto* pItem : m_pPeriodicTaskList)
    {
        if (pItem == pEventTask)
        {
            // 登録済み
            return;
        }
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_pPeriodicTaskList); i++)
    {
        if (m_pPeriodicTaskList[i] == nullptr)
        {
            m_pPeriodicTaskList[i] = pEventTask;
            m_PeriodicTaskCount++;
            NN_XCD_TASKMANAGER_DUMP("%s: %d -> %d\n", NN_CURRENT_FUNCTION_NAME, m_PeriodicTaskCount - 1, m_PeriodicTaskCount);
            return;
        }
    }

    NN_DETAIL_XCD_ERROR("Exceed maximum number of periodic tasks\n");
}

void TaskManager::UnregisterEvent(IMultiWaitBase* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    for (auto& pItem : m_pEventTaskList)
    {
        if (pItem == pEvent)
        {
            pItem->Unlink();
            pItem = nullptr;
            m_EventTaskCount--;
            NN_XCD_TASKMANAGER_DUMP("%s: %d -> %d\n", NN_CURRENT_FUNCTION_NAME, m_EventTaskCount + 1, m_EventTaskCount);
        }
    }
}

void TaskManager::UnregisterPeriodicTask(IEventTask* pEventTask) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEventTask);

    for (int i = 0; i < NN_ARRAY_SIZE(m_pPeriodicTaskList); i++)
    {
        if (m_pPeriodicTaskList[i] == pEventTask)
        {
            m_pPeriodicTaskList[i] = nullptr;
            m_PeriodicTaskCount--;
            NN_XCD_TASKMANAGER_DUMP("%s: %d -> %d\n", NN_CURRENT_FUNCTION_NAME, m_PeriodicTaskCount + 1, m_PeriodicTaskCount);
        }
    }
}

bool TaskManager::Proceed(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    for (auto* pItem : m_pEventTaskList)
    {
        if (pItem != nullptr && pItem->GetWaitId() == pMultiWaitHolder)
        {
            pItem->Clear();

            auto pHandler = reinterpret_cast<IEventTask*>(nn::os::GetMultiWaitHolderUserData(pItem->GetWaitId()));
            pHandler->EventFunction(pMultiWaitHolder);
            return true;
        }
    }

    return false;
}

nn::os::MultiWaitHolderType* TaskManager::WaitEvent() NN_NOEXCEPT
{
    if (m_EventTaskCount == 0)
    {
        // 何も登録されていないときは単純に Sleep
        nn::os::SleepThread(PeriodicEventInterval);
        return nullptr;
    }

    auto diffTime = (nn::os::GetSystemTick() - m_LastPeriodicEventTick).ToTimeSpan();
    if (diffTime < PeriodicEventInterval)
    {
        auto waitTime = PeriodicEventInterval - diffTime;
        auto* pHolder = nn::os::TimedWaitAny(m_pMultiWait, waitTime);
        if (pHolder != nullptr)
        {
            return pHolder;
        }
    }

    m_LastPeriodicEventTick = nn::os::GetSystemTick();

    return nullptr;
}

void TaskManager::ProceedPeriodic() NN_NOEXCEPT
{
    int taskCount = 0;

    // 登録されている関数を呼び出す
    for (auto* pTask : m_pPeriodicTaskList)
    {
        if (pTask != nullptr)
        {
            pTask->PeriodicEventFunction();
            taskCount++;
            if (taskCount >= m_PeriodicTaskCount)
            {
                // 登録数に達したらそれ以上見ない
                break;
            }
        }
    }
}

TaskManager& GetTaskManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(TaskManager, s_TaskManager);

    return s_TaskManager;
}

}} // namespace nn::xcd
