﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_PalmaCommandHandler.h"
#include "detail/xcd_BleGattUuids.h"

namespace nn { namespace xcd {

namespace
{
    enum CommandId : uint8_t
    {
        CommandId_PlayActivity             = 0x03,
        CommandId_ReadFactorySetting       = 0x04,
        CommandId_ReadSystemSaveData       = 0x06,
        CommandId_WriteSystemSaveData      = 0x07,
        CommandId_WriteDatabaseEntry       = 0x08,
        CommandId_ReadApplicationSaveData  = 0x09,
        CommandId_WriteApplicationSaveData = 0x0A,
        CommandId_SuspendFeatureSet        = 0x0C,
        CommandId_ReadPlayLog              = 0x0D,
        CommandId_ResetPlayLog             = 0x0E,
    };

    enum ResponseId : uint8_t
    {
        ResponseId_Ack                = 0x01,
        ResponseId_RespFactorySetting = 0x02,
        ResponseId_RespSystemSaveData = 0x03,
        ResponseId_RespRetransmission = 0x04,
        ResponseId_RespReadApplicationSaveData = 0x05,
        ResponseId_RespSuspendFeatureSet = 0x07,
        ResponseId_RespReadPlayLog = 0x08,
    };

    enum AckResultType : uint8_t
    {
        AckResultType_Success               = 0x00,
        AckResultType_UnknownCommandId      = 0x01,
        AckResultType_InvalidPayloadLength  = 0x02,
        AckResultType_InvalidCommandPayload = 0x03,
        AckResultType_Busy                  = 0x04,
        AckResultType_InvalidWriteRequest   = 0x05,
    };

    enum FactorySettingEntryId : uint8_t
    {
        FactorySettingEntryId_ProductType = 0x01,
        FactorySettingEntryId_ModelInformation = 0x02,
        FactorySettingEntryId_SensorCalibration = 0x03,
        FactorySettingEntryId_AnalogStickCalibration = 0x04,
        FactorySettingEntryId_ContentUniqueCode = 0x05,
    };

    enum SystemSaveDataEntryId : uint8_t
    {
        SystemSaveDataEntryId_StepCount    = 0x00,
        SystemSaveDataEntryId_Buddy        = 0x04,
        SystemSaveDataEntryId_ContentState = 0x06,
        SystemSaveDataEntryId_StepState    = 0x07,
        SystemSaveDataEntryId_DatabaseIdentificationVersion = 0x08,
    };

    enum ContentStateType : uint32_t
    {
        ContentStateType_Invalid,
        ContentStateType_Valid,
    };

    enum StepStateType : uint32_t
    {
        StepStateType_Disable,
        StepStateType_Enable,
    };

    const size_t ResponseAckSize = 2;
    const size_t ResponseReadProductTypeSize = 3;
    const size_t ResponseReadModelInformationSize = 31;
    const size_t ResponseReadSensorCalibrationValueSize = 25;
    const size_t ResponseReadAnalogStickCalibrationValueSize = 13;
    const size_t ResponseReadContentUniqueCodeSize = 21;
    const size_t ResponseReadStepSize = 5;
    const size_t ResponseReadContentStateSize = 2;
    const size_t ResponseReadDatabaseIdentificationVersionSize = 5;
    const size_t ResponseRetransmissionSize = 1;

    const size_t PlayActivityCommandSize = 2;
    const size_t ReadProductTypeCommandSize = 1;
    const size_t ReadModelInformationCommandSize = 1;
    const size_t ReadSensorCalibrationValueCommandSize = 1;
    const size_t ReadAnalogStickCalibrationValueCommandSize = 1;
    const size_t ReadContentUniqueCodeCommandSize = 1;
    const size_t ReadStepCommandSize = 1;
    const size_t ReadContentStateCommandSize = 1;
    const size_t ReadDatabaseIdentificationVersionCommandSize = 1;
    const size_t WriteStepCommandSize = 5;
    const size_t WriteFrModeTypeCommandSize = 5;
    const size_t WriteContentStateCommandSize = 5;
    const size_t WriteStepStateCommandSize = 5;
    const size_t WriteDatabaseIdentificationVersionCommandSize = 5;
    const size_t ReadApplicationSaveDataCommandSize = 8;
    const size_t SuspendFeatureCommandSize = 4;
    const size_t ReadPlayLogCommandSize = 2;
    const size_t ResetPlayLogCommandSize = 2;

    // Activity Entry と RgbLed Entry に付与されるヘッダ
    const uint8_t ActivityEntryHeaderRaw[] = { 0x41, 0x43, 0x54, 0x49, 0x00, 0x00, 0x00, 0x00, 0x00 };
    const size_t ActivityEntryHeaderLength = sizeof(ActivityEntryHeaderRaw);
    const uint8_t RgbLedEntryHeaderRaw[] = { 0x4c, 0x45, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    const size_t RgbLedEntryHeaderLength = sizeof(RgbLedEntryHeaderRaw);

    // OutputCommand のヘッダーサイズ
    const uint8_t OutputCommandHeaderLength = 3;

    // WriteDatabaseEntry のヘッダーサイズ
    const uint8_t OutputCommandWriteDatabaseEntryStartFrameHeaderLength = 5;
    const uint8_t OutputCommandWriteDatabaseEntryHeaderLength = 2;

    // CommandResponse のヘッダーサイズ
    const uint8_t CommandResponseHeaderLength = 3;

    // OutputCommand のオフセット情報
    enum OutputCommandFormatOffset
    {
        OutputCommandFormatOffset_Id = 0,
        OutputCommandFormatOffset_Length = 1,
        OutputCommandFormatOffset_Payload = 3
    };

    // OutputCommand の DatabaseEntry のオフセット情報
    enum DatabaseEntryByteOffset
    {
        DatabaseEntryByteOffset_FrameType = 0,
        DatabaseEntryByteOffset_SequenceNumber = 1,
        DatabaseEntryByteOffset_Payload = 2
    };
    enum DatabaseEntryStartFrameByteOffset
    {
        DatabaseEntryStartFrameByteOffset_FrameType = 0,
        DatabaseEntryStartFrameByteOffset_SequenceNumber = 1,
        DatabaseEntryStartFrameByteOffset_DatabaseType = 2,
        DatabaseEntryStartFrameByteOffset_Index = 3,
        DatabaseEntryStartFrameByteOffset_Payload = 5
    };

    // DatabaseEntry の FrameType
    enum DatabaseEntryFrameType : uint8_t
    {
        DatabaseEntryFrameType_Fragment = 0,   // 後続がある中間データ
        DatabaseEntryFrameType_StartFrame = 1, // 分割されたパケットの開始データ
        DatabaseEntryFrameType_EndFrame = 2,   // 分割されたパケットの終端データ
        DatabaseEntryFrameType_OneShot = 3,    // 分割されていない単発のデータ
    };

    // CommandResponse のオフセット情報
    enum CommandResponseFormatOffset
    {
        CommandResponseFormatOffset_Id = 0,
        CommandResponseFormatOffset_Length = 1,
        CommandResponseFormatOffset_Payload = 3
    };

    // ModelInformation のオフセット情報
    enum ModelInformationOffset
    {
        ModelInformationOffset_SensorHorizontalOffset = 0,
        ModelInformationOffset_IsScalingFlag = 6,
        ModelInformationOffset_Noise = 7,
        ModelInformationOffset_TypicalStroke = 8,
        ModelInformationOffset_OriginPlay = 10,
        ModelInformationOffset_CircuitValidRatio = 12,
        ModelInformationOffset_MinimumStrokePositive = 14,
        ModelInformationOffset_MinimumStrokeNegative = 18,
        ModelInformationOffset_OriginRangeMax = 22,
        ModelInformationOffset_OriginRangeMin = 26
    };

    template<typename T>
    inline T min(const T& a, const T& b)
    {
        return a < b ? a : b;
    }

    inline int16_t LittleEndianToInt16(const uint8_t* buffer)
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        return (buffer[0] | ((buffer[1] << 8) & 0xff00));
    }

    inline int32_t LittleEndianToInt32(const uint8_t* buffer)
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        return (buffer[0] | (buffer[1] << 8) | (buffer[2] << 16) | (buffer[3] << 24));
    }


    BleDeviceOperationResultType ConvertToOperationResultType(uint8_t ack)
    {
        switch (static_cast<AckResultType>(ack))
        {
            case AckResultType_Success:
            {
                return BleDeviceOperationResultType_Success;
            }
            case AckResultType_Busy:
            {
                return BleDeviceOperationResultType_Busy;
            }
            case AckResultType_InvalidPayloadLength:
            case AckResultType_InvalidCommandPayload:
            case AckResultType_InvalidWriteRequest:
            case AckResultType_UnknownCommandId:
            {
                NN_DETAIL_XCD_ERROR("PalmaCommandHandler: Unexpected Ack 0x%02X\n", ack);
                return BleDeviceOperationResultType_OperationNotSupported;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    // 現在のフレームが最後かどうか
    bool IsEndFrame(const BleOutputCommandDescriptor& descriptor)
    {
        return descriptor.maxDataSize == descriptor._sentByte;
    }

    // 現在のフレームが OneShot フレームかどうか
    bool IsOneShotFrame(const BleOutputCommandDescriptor& descriptor)
    {
        return descriptor.maxDataSize <= BleOutputCommand_PayloadMaxLength - OutputCommandWriteDatabaseEntryStartFrameHeaderLength;
    }
}

PalmaCommandHandler::PalmaCommandHandler() NN_NOEXCEPT
    : m_pClient(nullptr)
    , m_pListener(nullptr)
    , m_IsWritingDatabaseEntry(false)
    , m_IsBusy(false)
{
}

PalmaCommandHandler::~PalmaCommandHandler() NN_NOEXCEPT
{
    // 何もしない
}

void PalmaCommandHandler::CharacteristicWriteComplete(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    // OutputCommandCharacteristic 以外はなにもしない
    if (uuid != detail::OutputCommandCharacteristic.Uuid)
    {
        return;
    }
}

bool PalmaCommandHandler::IsSupportedVersion(BleOutputCommandFormatVersion commandVer, BleCommandResponseFormatVersion responseVer) NN_NOEXCEPT
{
    if (commandVer == BleOutputCommandFormatVersion_Palma
        && responseVer == BleCommandResponseFormatVersion_Palma)
    {
        return true;
    }
    return false;
}

void PalmaCommandHandler::SetGattClient(detail::IBleNhogClient* pClient) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pClient);

    m_pClient = pClient;
}

void PalmaCommandHandler::SetListener(IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    m_pListener = pListener;
}

void PalmaCommandHandler::Close() NN_NOEXCEPT
{
    m_IsBusy = false;
    m_IsWritingDatabaseEntry = false;
    m_pClient = nullptr;
    m_pListener = nullptr;
}

Result PalmaCommandHandler::HandleResponseReceivedEvent(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, CommandResponseHeaderLength);

    // コマンド発行中でない場合はレスポンスを破棄する
    if (!m_IsBusy && !m_IsWritingDatabaseEntry)
    {
        NN_DETAIL_XCD_ERROR("Disposed a response. response id=%d\n", buffer[CommandResponseFormatOffset_Id]);
        NN_RESULT_SUCCESS;
    }

    auto responseId = buffer[CommandResponseFormatOffset_Id];
    auto* pLength = reinterpret_cast<const uint16_t*>(&buffer[CommandResponseFormatOffset_Length]);
    auto* payload = &buffer[CommandResponseFormatOffset_Payload];

    // 受信したパケットサイズと Length フィールドの値をチェックする
    size_t payloadSize = size - CommandResponseHeaderLength;
    NN_SDK_REQUIRES_EQUAL(*pLength, payloadSize);
    NN_UNUSED(pLength);

    switch (responseId)
    {
        case ResponseId_Ack:
        {
            NN_SDK_REQUIRES_EQUAL(payloadSize, ResponseAckSize);
            return ParseAck(payload, ResponseAckSize);
        }
        case ResponseId_RespFactorySetting:
        {
            return ParseRespFactorySetting(payload, payloadSize);
        }
        case ResponseId_RespRetransmission:
        {
            NN_SDK_REQUIRES_EQUAL(payloadSize, ResponseRetransmissionSize);
            return ParseRespRetransmission(payload, ResponseRetransmissionSize);
        }
        case ResponseId_RespSystemSaveData:
        {
            return ParseRespSystemSaveData(payload, payloadSize);
        }
        case ResponseId_RespReadApplicationSaveData:
        {
            return ParseRespReadApplicationSaveData(payload, payloadSize);
        }
        case ResponseId_RespSuspendFeatureSet:
        {
            return ParseRespSuspendFeatureSet(payload, payloadSize);
        }
        case ResponseId_RespReadPlayLog:
        {
            return ParseRespPlayLog(payload, payloadSize);
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result PalmaCommandHandler::PlayActivity(uint16_t slotNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_PlayDeviceActivity;

    SendCommand(CommandId_PlayActivity, &slotNumber, PlayActivityCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetProductType() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    uint8_t packedData[ReadProductTypeCommandSize];
    packedData[0] = FactorySettingEntryId_ProductType;
    SendCommand(CommandId_ReadFactorySetting, packedData, ReadProductTypeCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetModelInformation() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    uint8_t packedData[ReadModelInformationCommandSize];
    packedData[0] = FactorySettingEntryId_ModelInformation;
    SendCommand(CommandId_ReadFactorySetting, packedData, ReadModelInformationCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetSensorCalibrationValue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    uint8_t packedData[ReadSensorCalibrationValueCommandSize];
    packedData[0] = FactorySettingEntryId_SensorCalibration;
    SendCommand(CommandId_ReadFactorySetting, packedData, ReadSensorCalibrationValueCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetAnalogStickCalibrationValue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    uint8_t packedData[ReadAnalogStickCalibrationValueCommandSize];
    packedData[0] = FactorySettingEntryId_AnalogStickCalibration;
    SendCommand(CommandId_ReadFactorySetting, packedData, ReadAnalogStickCalibrationValueCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetContentUniqueCode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ReadContentUniqueCode;

    uint8_t packedData[ReadContentUniqueCodeCommandSize];
    packedData[0] = FactorySettingEntryId_ContentUniqueCode;
    SendCommand(CommandId_ReadFactorySetting, packedData, ReadContentUniqueCodeCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::SetDatabaseIdentificationVersion(int32_t version) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_SetDatabaseIdentificationVersion;

    uint8_t packedData[WriteDatabaseIdentificationVersionCommandSize];
    packedData[0] = SystemSaveDataEntryId_DatabaseIdentificationVersion;
    memcpy(&packedData[1], &version, sizeof(int32_t));

    SendCommand(CommandId_WriteSystemSaveData, packedData, WriteDatabaseIdentificationVersionCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::SetFrModeType(uint32_t frModeType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_SetFrModeType;

    uint8_t packedData[WriteFrModeTypeCommandSize];
    packedData[0] = SystemSaveDataEntryId_Buddy;
    memcpy(&packedData[1], &frModeType, sizeof(uint32_t));

    SendCommand(CommandId_WriteSystemSaveData, packedData, WriteFrModeTypeCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ResetStepCount() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ResetStepCount;

    uint32_t defaultStepCount = 0;
    uint8_t packedData[WriteStepCommandSize];
    packedData[0] = SystemSaveDataEntryId_StepCount;
    memcpy(&packedData[1], &defaultStepCount, sizeof(uint32_t));

    SendCommand(CommandId_WriteSystemSaveData, packedData, WriteStepCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::EnableStepCounter(bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_EnableStep;

    StepStateType stateType = isEnabled ? StepStateType_Enable : StepStateType_Disable;
    uint8_t packedData[WriteStepStateCommandSize];
    packedData[0] = SystemSaveDataEntryId_StepState;
    memcpy(&packedData[1], &stateType, sizeof(StepStateType));

    SendCommand(CommandId_WriteSystemSaveData, packedData, WriteStepStateCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetContentState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ReadContentUniqueCode;

    uint8_t packedData[ReadContentStateCommandSize];
    packedData[0] = SystemSaveDataEntryId_ContentState;
    SendCommand(CommandId_ReadSystemSaveData, packedData, ReadContentStateCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::SetContentState(bool isEnabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_SetContentUniqueCodeInvalid;

    ContentStateType stateType = isEnabled ? ContentStateType_Valid : ContentStateType_Invalid;
    uint8_t packedData[WriteContentStateCommandSize];
    packedData[0] = SystemSaveDataEntryId_ContentState;
    memcpy(&packedData[1], &stateType, sizeof(ContentStateType));

    SendCommand(CommandId_WriteSystemSaveData, packedData, WriteContentStateCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetDatabaseIdentificationVersion() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_GetDatabaseIdentificationVersion;

    uint8_t packedData[ReadDatabaseIdentificationVersionCommandSize];
    packedData[0] = SystemSaveDataEntryId_DatabaseIdentificationVersion;
    SendCommand(CommandId_ReadSystemSaveData, packedData, ReadDatabaseIdentificationVersionCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::GetStepCount() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_GetStepCount;

    uint8_t packedData[ReadStepCommandSize];
    packedData[0] = SystemSaveDataEntryId_StepCount;
    SendCommand(CommandId_ReadSystemSaveData, packedData, ReadStepCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsWritingDatabaseEntry == false, nn::xcd::ResultBluetoothLeBusy());

    m_IsWritingDatabaseEntry = true;

    m_OutputCommandDescriptor.isCancelRequested = false;
    m_OutputCommandDescriptor.pDataPtr = buffer;
    m_OutputCommandDescriptor.maxDataSize = size;
    m_OutputCommandDescriptor._initialSequenceNumber = 0;
    m_OutputCommandDescriptor._sequenceNumber = 0;
    m_OutputCommandDescriptor._sentByte = 0;

    m_DatabaseEntryConfig = config;

    detail::GattOperationPayload rawData;
    rawData.length = OutputCommandHeaderLength + GetDatabaseEntryFirstFragment(m_BufferForFragmentation, m_OutputCommandDescriptor, config);
    rawData.dataptr = m_BufferForFragmentation;

    m_pClient->WriteOutputCommandCharacteristicAsync(rawData, this);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ReadApplicationSection(int32_t address, int32_t readSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ReadApplicationSection;

    uint8_t packedData[ReadApplicationSaveDataCommandSize];
    memcpy(&packedData[0], &address, sizeof(int32_t));
    memcpy(&packedData[4], &readSize, sizeof(int32_t));

    SendCommand(CommandId_ReadApplicationSaveData, packedData, ReadApplicationSaveDataCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::WriteApplicationSection(int32_t address, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_WriteApplicationSection;
    m_IsBusy = true;

    m_Buffer[OutputCommandFormatOffset_Id] = CommandId_WriteApplicationSaveData;

    auto* pLength = reinterpret_cast<uint16_t*>(&m_Buffer[OutputCommandFormatOffset_Length]);
    uint16_t payloadLength = sizeof(int32_t) + sizeof(int32_t) + static_cast<uint16_t>(size);
    *pLength = static_cast<uint16_t>(payloadLength);

    memcpy(&m_Buffer[OutputCommandFormatOffset_Payload], &address, sizeof(int32_t));
    memcpy(&m_Buffer[OutputCommandFormatOffset_Payload + 4], &size, sizeof(int32_t));
    memcpy(&m_Buffer[OutputCommandFormatOffset_Payload + 8], buffer, size);

    detail::GattOperationPayload rawData;
    rawData.length = OutputCommandHeaderLength + payloadLength;
    rawData.dataptr = m_Buffer;

    m_pClient->WriteOutputCommandCharacteristicAsync(rawData, this);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::SuspendFeature(uint32_t featureFlagSet) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_SuspendFeature;

    SendCommand(CommandId_SuspendFeatureSet, &featureFlagSet, SuspendFeatureCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ReadPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ReadPlayLog;

    SendCommand(CommandId_ReadPlayLog, &index, ReadPlayLogCommandSize);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ResetPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    NN_RESULT_THROW_UNLESS(m_pClient != nullptr, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsBusy == false, nn::xcd::ResultBluetoothLeBusy());

    m_OperationTypeForAck = BleDeviceOperationType_ResetPlayLog;

    SendCommand(CommandId_ResetPlayLog, &index, ResetPlayLogCommandSize);

    NN_RESULT_SUCCESS;
}

void PalmaCommandHandler::HandleResponseTimeout() NN_NOEXCEPT
{
    if (m_IsBusy)
    {
        m_IsBusy = false;
        m_pListener->NotifyCommandCompleted(m_OperationTypeForAck, BleDeviceOperationResultType_ResponseTimeout);
    }
    if (m_IsWritingDatabaseEntry)
    {
        CancelWriteDatabaseEntry();
        m_IsWritingDatabaseEntry = false;
        m_pListener->NotifyCommandCompleted(BleDeviceOperationType_WriteDatabaseEntry, BleDeviceOperationResultType_ResponseTimeout);
    }
}

void PalmaCommandHandler::CancelWriteDatabaseEntry() NN_NOEXCEPT
{
    if (m_IsWritingDatabaseEntry)
    {
        m_OutputCommandDescriptor.isCancelRequested = true;
        m_OutputCommandDescriptor.pDataPtr = nullptr;
    }
}

Result PalmaCommandHandler::ParseAck(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    auto commandId = static_cast<CommandId>(buffer[0]);
    auto ackResult = static_cast<AckResultType>(buffer[1]);

    // WriteDatabaseEntry コマンドを実行中かつ、Ack が Success の場合は後続のパケットを送信する
    if (commandId == CommandId_WriteDatabaseEntry
        && !IsEndFrame(m_OutputCommandDescriptor)
        && ackResult == AckResultType_Success)
    {
        detail::GattOperationPayload rawData;
        rawData.length = OutputCommandHeaderLength + GetDatabaseEntryFragment(m_BufferForFragmentation, m_OutputCommandDescriptor);
        rawData.dataptr = m_BufferForFragmentation;
        m_pClient->WriteOutputCommandCharacteristicAsync(rawData, this);
        return ResultBluetoothOtherTriggerEventOnProgress();
    }
    else
    {
        HandleCommandCompleted(commandId, ackResult);
        NN_RESULT_SUCCESS;
    }
}

Result PalmaCommandHandler::ParseRespFactorySetting(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, sizeof(FactorySettingEntryId));
    m_IsBusy = false;
    auto entryId = static_cast<FactorySettingEntryId>(buffer[0]);
    auto* entryDataBuffer = &buffer[1];

    switch (entryId)
    {
        case FactorySettingEntryId_ProductType:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadProductTypeSize);
            BleDeviceProductType productType;
            ParseProductType(&productType, entryDataBuffer, ResponseReadProductTypeSize - 1);
            m_pListener->NotifyReadProductType(productType);
            break;
        }
        case FactorySettingEntryId_ModelInformation:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadModelInformationSize);
            BleDeviceModelInformation info;
            ParseModelInformation(&info, entryDataBuffer, ResponseReadModelInformationSize - 1);
            m_pListener->NotifyReadModelInformation(info);
            break;
        }
        case FactorySettingEntryId_SensorCalibration:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadSensorCalibrationValueSize);
            SensorCalibrationValue calValue;
            ParseSensorCalibration(&calValue, entryDataBuffer, ResponseReadSensorCalibrationValueSize - 1);
            m_pListener->NotifyReadSensorCalibration(calValue);
            break;
        }
        case FactorySettingEntryId_AnalogStickCalibration:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadAnalogStickCalibrationValueSize);
            AnalogStickValidRange calValue;
            ParseAnalogStickCalibration(&calValue, entryDataBuffer, ResponseReadAnalogStickCalibrationValueSize - 1);
            m_pListener->NotifyReadAnalogStickCalibration(calValue);
            break;
        }
        case FactorySettingEntryId_ContentUniqueCode:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadContentUniqueCodeSize);
            m_pListener->NotifyReadContentUniqueCode(BleDeviceOperationResultType_Success, entryDataBuffer, ResponseReadContentUniqueCodeSize - 1);
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ParseRespSystemSaveData(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    m_IsBusy = false;
    auto entryId = static_cast<SystemSaveDataEntryId>(buffer[0]);
    auto* entryDataBuffer = &buffer[1];

    switch (entryId)
    {
        case SystemSaveDataEntryId_StepCount:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadStepSize);
            NN_UNUSED(ResponseReadStepSize);
            m_pListener->NotifyReadStep(BleDeviceOperationResultType_Success, static_cast<uint32_t>(LittleEndianToInt32(entryDataBuffer)));
            break;
        }
        case SystemSaveDataEntryId_ContentState:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadContentStateSize);
            NN_UNUSED(ResponseReadContentStateSize);
            bool isAvailable = (entryDataBuffer[0] == ContentStateType_Valid);
            m_pListener->NotifyReadContentState(BleDeviceOperationResultType_Success, isAvailable);
            break;
        }
        case SystemSaveDataEntryId_DatabaseIdentificationVersion:
        {
            NN_SDK_REQUIRES_EQUAL(size, ResponseReadDatabaseIdentificationVersionSize);
            NN_UNUSED(ResponseReadDatabaseIdentificationVersionSize);
            m_pListener->NotifyReadDatabaseVersion(BleDeviceOperationResultType_Success, LittleEndianToInt32(entryDataBuffer));
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ParseRespRetransmission(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_SDK_REQUIRES_EQUAL(size, sizeof(uint8_t));
    m_IsBusy = false;
    uint8_t sequenceNumber = buffer[0];
    HandleRetransmission(sequenceNumber);

    // 再送中はコマンド進行中のエラーを返す
    return ResultBluetoothOtherTriggerEventOnProgress();
}

Result PalmaCommandHandler::ParseRespReadApplicationSaveData(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    m_IsBusy = false;
    int32_t readAddress = LittleEndianToInt32(&buffer[0]);
    int32_t readSize = LittleEndianToInt32(&buffer[4]);

    NN_SDK_REQUIRES_EQUAL(size, static_cast<size_t>(sizeof(int32_t) * 2 + readSize));
    NN_SDK_REQUIRES_LESS_EQUAL(static_cast<size_t>(readSize), BleDeviceApplicationSectionAccessUnitSizeMax);

    m_pListener->NotifyReadApplicationSection(BleDeviceOperationResultType_Success, &buffer[8], readSize, readAddress);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ParseRespSuspendFeatureSet(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    m_IsBusy = false;
    uint32_t suspendFeatureFlagSet = static_cast<uint32_t>(LittleEndianToInt32(&buffer[0]));
    m_pListener->NotifyRespSuspendFeatureSet(BleDeviceOperationResultType_Success, suspendFeatureFlagSet);

    NN_RESULT_SUCCESS;
}

Result PalmaCommandHandler::ParseRespPlayLog(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    m_IsBusy = false;
    uint16_t index = static_cast<uint16_t>(LittleEndianToInt16(&buffer[0]));
    uint8_t valueLength = buffer[2];

    NN_SDK_REQUIRES_EQUAL(size, sizeof(uint16_t) + sizeof(uint8_t) + valueLength);

    m_pListener->NotifyReadPlayLog(BleDeviceOperationResultType_Success, index, &buffer[3], static_cast<int32_t>(valueLength));

    NN_RESULT_SUCCESS;
}

void PalmaCommandHandler::ParseProductType(BleDeviceProductType* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    pOutValue->main = buffer[0];
    pOutValue->sub = buffer[1];
}

void PalmaCommandHandler::ParseModelInformation(BleDeviceModelInformation* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    pOutValue->sensorHorizontalOffset.x = LittleEndianToInt16(&buffer[ModelInformationOffset_SensorHorizontalOffset]);
    pOutValue->sensorHorizontalOffset.y = LittleEndianToInt16(&buffer[ModelInformationOffset_SensorHorizontalOffset + 2]);
    pOutValue->sensorHorizontalOffset.z = LittleEndianToInt16(&buffer[ModelInformationOffset_SensorHorizontalOffset + 4]);

    // PALMA では常に不要
    pOutValue->isScalingRequired = false;

    pOutValue->analogStickNoise = buffer[ModelInformationOffset_Noise];
    pOutValue->analogStickTypicalStroke = LittleEndianToInt16(&buffer[ModelInformationOffset_TypicalStroke]);
    pOutValue->analogStickOriginPlay = LittleEndianToInt16(&buffer[ModelInformationOffset_OriginPlay]);
    pOutValue->analogStickCircuitValidRatio = LittleEndianToInt16(&buffer[ModelInformationOffset_CircuitValidRatio]);
    pOutValue->analogStickMinimumStrokePositive.x = LittleEndianToInt16(&buffer[ModelInformationOffset_MinimumStrokePositive]);
    pOutValue->analogStickMinimumStrokePositive.y = LittleEndianToInt16(&buffer[ModelInformationOffset_MinimumStrokePositive + 2]);
    pOutValue->analogStickMinimumStrokeNegative.x = LittleEndianToInt16(&buffer[ModelInformationOffset_MinimumStrokeNegative]);
    pOutValue->analogStickMinimumStrokeNegative.y = LittleEndianToInt16(&buffer[ModelInformationOffset_MinimumStrokeNegative + 2]);
    pOutValue->analogStickOriginRangeMin.x = LittleEndianToInt16(&buffer[ModelInformationOffset_OriginRangeMin]);
    pOutValue->analogStickOriginRangeMin.y = LittleEndianToInt16(&buffer[ModelInformationOffset_OriginRangeMin + 2]);
    pOutValue->analogStickOriginRangeMax.x = LittleEndianToInt16(&buffer[ModelInformationOffset_OriginRangeMax]);
    pOutValue->analogStickOriginRangeMax.y = LittleEndianToInt16(&buffer[ModelInformationOffset_OriginRangeMax + 2]);
}

void PalmaCommandHandler::ParseSensorCalibration(SensorCalibrationValue* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    pOutValue->accelerometerOrigin.x = LittleEndianToInt16(&buffer[0]);
    pOutValue->accelerometerOrigin.y = LittleEndianToInt16(&buffer[2]);
    pOutValue->accelerometerOrigin.z = LittleEndianToInt16(&buffer[4]);
    pOutValue->accelerometerSensitivity.x = LittleEndianToInt16(&buffer[6]);
    pOutValue->accelerometerSensitivity.y = LittleEndianToInt16(&buffer[8]);
    pOutValue->accelerometerSensitivity.z = LittleEndianToInt16(&buffer[10]);
    pOutValue->gyroscopeOrigin.x = LittleEndianToInt16(&buffer[12]);
    pOutValue->gyroscopeOrigin.y = LittleEndianToInt16(&buffer[14]);
    pOutValue->gyroscopeOrigin.z = LittleEndianToInt16(&buffer[16]);
    pOutValue->gyroscopeSensitivity.x = LittleEndianToInt16(&buffer[18]);
    pOutValue->gyroscopeSensitivity.y = LittleEndianToInt16(&buffer[20]);
    pOutValue->gyroscopeSensitivity.z = LittleEndianToInt16(&buffer[22]);
}

void PalmaCommandHandler::ParseAnalogStickCalibration(AnalogStickValidRange* pOutValue, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    pOutValue->circuitMax.x = LittleEndianToInt16(&buffer[0]);
    pOutValue->circuitMax.y = LittleEndianToInt16(&buffer[2]);
    pOutValue->origin.x = LittleEndianToInt16(&buffer[4]);
    pOutValue->origin.y = LittleEndianToInt16(&buffer[6]);
    pOutValue->circuitMin.x = LittleEndianToInt16(&buffer[8]);
    pOutValue->circuitMin.y = LittleEndianToInt16(&buffer[10]);
    pOutValue->originPlay = 0;
}

void PalmaCommandHandler::HandleRetransmission(uint8_t sequenceNumber) NN_NOEXCEPT
{
    // シーケンス番号と送信済みデータ量を巻き戻し、フラグメントを取得しなおす
    if (IsOneShotFrame(m_OutputCommandDescriptor))
    {
        m_OutputCommandDescriptor._sequenceNumber = m_OutputCommandDescriptor._initialSequenceNumber;
        m_OutputCommandDescriptor._sentByte = 0;
    }
    else
    {
        uint32_t sentCount = (256 + sequenceNumber - m_OutputCommandDescriptor._initialSequenceNumber) % 256;
        size_t sentByte = 0;
        for (uint32_t i = 0; i < sentCount; ++i)
        {
            if (i == 0)
            {
                sentByte += BleOutputCommand_PayloadMaxLength - OutputCommandWriteDatabaseEntryStartFrameHeaderLength;
            }
            else
            {
                sentByte += BleOutputCommand_PayloadMaxLength - OutputCommandWriteDatabaseEntryHeaderLength;
            }
        }
        NN_SDK_REQUIRES_LESS_EQUAL(sentByte, m_OutputCommandDescriptor.maxDataSize);

        m_OutputCommandDescriptor._sequenceNumber = sequenceNumber;
        m_OutputCommandDescriptor._sentByte = sentByte;
    }

    detail::GattOperationPayload rawData;
    rawData.length = OutputCommandHeaderLength + GetDatabaseEntryFragment(m_BufferForFragmentation, m_OutputCommandDescriptor);
    rawData.dataptr = m_BufferForFragmentation;
    m_pClient->WriteOutputCommandCharacteristicAsync(rawData, this);
}

size_t PalmaCommandHandler::GetDatabaseEntryFirstFragment(uint8_t* pOutFragment, BleOutputCommandDescriptor& descriptor, const BleDeviceDatabaseEntryConfig& config) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFragment);

    // 送信するコマンドのヘッダ分をずらして、分割フレームのデータを書き込む
    auto payloadBuffer = &pOutFragment[OutputCommandFormatOffset_Payload];
    size_t byteOffset = 0;
    size_t copyByte;

    payloadBuffer[DatabaseEntryStartFrameByteOffset_SequenceNumber] = descriptor._sequenceNumber;
    payloadBuffer[DatabaseEntryStartFrameByteOffset_DatabaseType] = static_cast<uint8_t>(config.type);
    memcpy(&payloadBuffer[DatabaseEntryStartFrameByteOffset_Index], &config.index, sizeof(config.index));
    byteOffset += OutputCommandWriteDatabaseEntryStartFrameHeaderLength;

    // Activity Database への書き込みだったら、ヘッダをつける
    if (config.type == BleDeviceDatabaseType_Activity)
    {
        memcpy(&payloadBuffer[DatabaseEntryStartFrameByteOffset_Payload], ActivityEntryHeaderRaw, ActivityEntryHeaderLength);
        byteOffset += ActivityEntryHeaderLength;
    }
    // RgbLed Database への書き込みだったら、ヘッダをつける
    else if (config.type == BleDeviceDatabaseType_Led)
    {
        memcpy(&payloadBuffer[DatabaseEntryStartFrameByteOffset_Payload], RgbLedEntryHeaderRaw, RgbLedEntryHeaderLength);
        byteOffset += RgbLedEntryHeaderLength;
    }

    // 送信データがコマンドペイロードに収まるサイズだったら、分割しない
    if (descriptor.maxDataSize <= BleOutputCommand_PayloadMaxLength - byteOffset)
    {
        payloadBuffer[DatabaseEntryStartFrameByteOffset_FrameType] = DatabaseEntryFrameType_OneShot;
        copyByte = descriptor.maxDataSize;
    }
    else
    {
        payloadBuffer[DatabaseEntryStartFrameByteOffset_FrameType] = DatabaseEntryFrameType_StartFrame;
        copyByte = BleOutputCommand_PayloadMaxLength - byteOffset;
    }

    if (descriptor.isCancelRequested)
    {
        payloadBuffer[DatabaseEntryStartFrameByteOffset_FrameType] = DatabaseEntryFrameType_OneShot;
        // すべてのデータを送ったことにして終了フレームを送る
        descriptor._sentByte = descriptor.maxDataSize;
        copyByte = 0;
    }
    else
    {
        memcpy(&payloadBuffer[byteOffset], &descriptor.pDataPtr[descriptor._sentByte], copyByte);
        byteOffset += copyByte;
        descriptor._sentByte += copyByte;
    }

    descriptor._sequenceNumber++;

    // 送信するコマンドのヘッダを付与する
    auto* headerBuffer = pOutFragment;
    headerBuffer[OutputCommandFormatOffset_Id] = CommandId_WriteDatabaseEntry;
    auto* pCommandLength = reinterpret_cast<uint16_t*>(&headerBuffer[OutputCommandFormatOffset_Length]);
    *pCommandLength = static_cast<uint16_t>(byteOffset);

    return byteOffset;
}

size_t PalmaCommandHandler::GetDatabaseEntryFragment(uint8_t* pOutFragment, BleOutputCommandDescriptor& descriptor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutFragment);

    // 送信するコマンドのヘッダ分をずらして、分割フレームのデータを書き込む
    auto payloadBuffer = &pOutFragment[OutputCommandFormatOffset_Payload];
    size_t byteOffset = 0;
    size_t copyByte;

    // 残りの送信データがコマンドペイロードに収まるサイズだったら、終端データを返す
    if (descriptor.maxDataSize - descriptor._sentByte <= BleOutputCommand_PayloadMaxLength - OutputCommandWriteDatabaseEntryHeaderLength)
    {
        payloadBuffer[DatabaseEntryByteOffset_FrameType] = DatabaseEntryFrameType_EndFrame;
        payloadBuffer[DatabaseEntryByteOffset_SequenceNumber] = descriptor._sequenceNumber;

        byteOffset += OutputCommandWriteDatabaseEntryHeaderLength;
        copyByte = descriptor.maxDataSize - descriptor._sentByte;
    }
    else
    {
        payloadBuffer[DatabaseEntryByteOffset_FrameType] = DatabaseEntryFrameType_Fragment;
        payloadBuffer[DatabaseEntryByteOffset_SequenceNumber] = descriptor._sequenceNumber;

        byteOffset += OutputCommandWriteDatabaseEntryHeaderLength;
        copyByte = BleOutputCommand_PayloadMaxLength - OutputCommandWriteDatabaseEntryHeaderLength;
    }

    if (descriptor.isCancelRequested)
    {
        payloadBuffer[DatabaseEntryByteOffset_FrameType] = DatabaseEntryFrameType_EndFrame;
        // すべてのデータを送ったことにして終了フレームを送る
        descriptor._sentByte = descriptor.maxDataSize;
        copyByte = 0;
    }
    else
    {
        memcpy(&payloadBuffer[byteOffset], &descriptor.pDataPtr[descriptor._sentByte], copyByte);
        byteOffset += copyByte;
        descriptor._sentByte += copyByte;
    }

    descriptor._sequenceNumber++;

    // 送信するコマンドのヘッダを付与する
    auto* headerBuffer = pOutFragment;
    headerBuffer[OutputCommandFormatOffset_Id] = CommandId_WriteDatabaseEntry;
    auto* pCommandLength = reinterpret_cast<uint16_t*>(&headerBuffer[OutputCommandFormatOffset_Length]);
    *pCommandLength = static_cast<uint16_t>(byteOffset);

    return byteOffset;
}

void PalmaCommandHandler::HandleCommandCompleted(uint8_t commandId, uint8_t ackResult) NN_NOEXCEPT
{
    BleDeviceOperationType operationType = m_OperationTypeForAck;
    if (commandId == CommandId_WriteDatabaseEntry)
    {
        m_IsWritingDatabaseEntry = false;
        operationType = BleDeviceOperationType_WriteDatabaseEntry;
    }
    else
    {
        m_IsBusy = false;
    }
    BleDeviceOperationResultType resultType = ConvertToOperationResultType(ackResult);
    m_pListener->NotifyCommandCompleted(operationType, resultType);
}

Result PalmaCommandHandler::SendCommand(uint8_t id, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    m_IsBusy = true;

    m_Buffer[OutputCommandFormatOffset_Id] = id;

    auto* pLength = reinterpret_cast<uint16_t*>(&m_Buffer[OutputCommandFormatOffset_Length]);
    uint16_t payloadLength = static_cast<uint16_t>(size);
    *pLength = static_cast<uint16_t>(payloadLength);

    memcpy(&m_Buffer[OutputCommandFormatOffset_Payload], buffer, payloadLength);

    detail::GattOperationPayload rawData;
    rawData.length = OutputCommandHeaderLength + payloadLength;
    rawData.dataptr = m_Buffer;

    m_pClient->WriteOutputCommandCharacteristicAsync(rawData, this);

    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
