﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_PadInput.h"
#include "xcd_ParserUtil.h"
#include "xcd_SerialFlashMap.h"
#include "xcd_ReportTypes.h"

#if 0
#define PAD_DUMP_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define PAD_DUMP_LOG(...)
#endif

namespace nn { namespace xcd {

namespace
{
    // デフォルト定義一覧
    const AccelerometerFsr DefaultAccelerometerFsr = AccelerometerFsr_8G;
    const GyroscopeFsr DefaultGyroscopeFsr = GyroscopeFsr_2000dps;
    const AnalogStickState DefaultAnalogStick = {0,0};
    const SensorCalibrationValue DefaultSensorCalibration = { {0,0,0},
                                                              {16384,16384,16384},
                                                              {0,0,0},
                                                              {13371,13371,13371} };

    // モデル値カウントを FSR = 8G から FSR = 2G の単位系に変換
    const SensorState JoyLeftSensorHorizontalOffsetTypical = { 350 * 4, 0, 4081 * 4 };
    const SensorState JoyRightSensorHorizontalOffsetTypical = { 350 * 4, 0, -4081 * 4 };
    const SensorState FullKeySensorHorizontalOffsetTypical = { -688 * 4, 0, 4038 * 4 };

    const int16_t StickMax = 4096;         //!< アナログスティックの最大値

    struct Range
    {
        int16_t min;
        int16_t max;
    };

    const AnalogStickStaticValues JoyAnalogStickValuesOld = {
        false,           // isScalingRequired
        174,             // originPlay
        3604,            // circuitValidRatio
        {                    // deviceParameter
            { 700, 700 },    // minimumStrokePositive
            { 700, 700 },    // minimumStrokeNegative
            { 1228, 1228 },  // zeroRangeMin
            { 2868, 2868 },  // zeroRangeMax
        }
    };
    const AnalogStickStaticValues FullKeyAnalogStickValuesOld = {
        false,           // isScalingRequired
        274,             // originPlay
        3891,            // circuitValidRatio
        {                    // deviceParameter
            { 1450, 1450 },  // minimumStrokePositive
            { 1450, 1450 },  // minimumStrokeNegative
            { 1605, 1605 },  // zeroRangeMin
            { 2531, 2531 },  // zeroRangeMax
        }
    };

    const AnalogStickStaticValues JoyAnalogStickValues = {
        false,           // isScalingRequired
        174,             // originPlay
        3604,            // circuitValidRatio
        {                    // deviceParameter
            { 750, 750 },    // minimumStrokePositive
            { 750, 750 },    // minimumStrokeNegative
            { 1174, 1174 },  // zeroRangeMin
            { 2740, 2740 },  // zeroRangeMax
        }
    };
    const AnalogStickStaticValues FullKeyAnalogStickValues = {
        false,           // isScalingRequired
        150,             // originPlay
        3891,            // circuitValidRatio
        {                    // deviceParameter
            { 1236, 1236 },  // minimumStrokePositive
            { 1236, 1236 },  // minimumStrokeNegative
            { 1587, 1587 },  // zeroRangeMin
            { 2503, 2503 },  // zeroRangeMax
        }
    };

    const Range AccelerometerOriginRange         = {-3276,  3276};
    const Range AccelerometerSensitivityRange    = {14745, 18022};
    const Range GyroscopeOriginRange             = { -819,   819};
    const Range GyroscopeSensitivityRange        = {12034, 16868};

    inline bool ValidateRange(int16_t value, int16_t min, int16_t max)
    {
        return (value > min && value < max);
    };

    inline bool ValidateRange(int32_t value, int16_t min, int16_t max)
    {
        return (value > min && value < max);
    };

    inline bool ValidateRange(int32_t value, Range range)
    {
        return ValidateRange(value, range.min, range.max);
    };

    bool ValidateCalibrationValue(const SensorCalibrationValue& value)
    {
        bool validate =
            (ValidateRange(value.accelerometerOrigin.x, AccelerometerOriginRange) &&
             ValidateRange(value.accelerometerOrigin.y, AccelerometerOriginRange) &&
             ValidateRange(value.accelerometerOrigin.z, AccelerometerOriginRange) &&
             ValidateRange(value.accelerometerSensitivity.x, AccelerometerSensitivityRange) &&
             ValidateRange(value.accelerometerSensitivity.y, AccelerometerSensitivityRange) &&
             ValidateRange(value.accelerometerSensitivity.z, AccelerometerSensitivityRange) &&
             ValidateRange(value.gyroscopeOrigin.x, GyroscopeOriginRange) &&
             ValidateRange(value.gyroscopeOrigin.y, GyroscopeOriginRange) &&
             ValidateRange(value.gyroscopeOrigin.z, GyroscopeOriginRange) &&
             ValidateRange(value.gyroscopeSensitivity.x, GyroscopeSensitivityRange) &&
             ValidateRange(value.gyroscopeSensitivity.y, GyroscopeSensitivityRange) &&
             ValidateRange(value.gyroscopeSensitivity.z, GyroscopeSensitivityRange)
             );

        return validate;
    };

    //!< 外周値に不感帯を付加します
    void AddPlayOnStickCircuit(AnalogStickValidRange* pValue, const AnalogStickStaticValues staticValues)
    {
        int xScale = (staticValues.isScalingRequired) ? 90 : 100;
        pValue->circuitMin.x = static_cast<int16_t>(
                                   (static_cast<int64_t>(pValue->circuitMin.x) * staticValues.circuitValidRatio * xScale) / StickMax / 100
                               );
        pValue->circuitMin.y = static_cast<int16_t>(
                                   (static_cast<int64_t>(pValue->circuitMin.y) * staticValues.circuitValidRatio) / StickMax
                               );
        pValue->circuitMax.x = static_cast<int16_t>(
                                   (static_cast<int64_t>(pValue->circuitMax.x) * staticValues.circuitValidRatio * xScale) / StickMax / 100
                               );
        pValue->circuitMax.y = static_cast<int16_t>(
                                   (static_cast<int64_t>(pValue->circuitMax.y) * staticValues.circuitValidRatio) / StickMax
                               );
    }

    void DumpStickValue(const AnalogStickState& state)
    {
        NN_UNUSED(state);
        PAD_DUMP_LOG("x:%5d y:%5d", state.x, state.y);
    }

    void DumpSensorValue(const SensorState& state)
    {
        NN_UNUSED(state);
        PAD_DUMP_LOG("x:%5d y:%5d z:%5d", state.x, state.y, state.z);
    }
}

PadInput::PadInput() NN_NOEXCEPT :
    m_SensorMutex(),
    m_SensorBufferIndex(0),
    m_SensorBufferCount(0),
    m_LastSampleNumberForBuffer(0)
{
    m_Config.accelerometerFsr = DefaultAccelerometerFsr;
    m_Config.gyroscopeFsr = DefaultGyroscopeFsr;
    m_Config.analogStickLValidRange.origin = DefaultAnalogStick;
    m_Config.analogStickRValidRange.origin = DefaultAnalogStick;
    m_Config.sensorCalibrationValue = DefaultSensorCalibration;
    m_Config.sensorSleep = false;

    for (auto& state : m_SensorStates)
    {
        state = SixAxisSensorState();
    }
}

PadInput::~PadInput() NN_NOEXCEPT
{
    // 何もしない
}

//!< デバイスが接続されたときに呼ばれる関数
void PadInput::Activate(DeviceType type, FirmwareVersionImpl firmwareVersion) NN_NOEXCEPT
{
    m_Type = type;
    m_FirmwareVersion = firmwareVersion;

    m_SensorBufferCount = 0;

    m_Config.isSensorCalReadDone = false;
    m_Config.isStickCalReadDone = false;
    m_Config.isLeftStickCalibrated = false;
    m_Config.isRightStickCalibrated = false;

    switch (m_Type)
    {
    case DeviceType_Unknown:
        m_ValidAnalogStick.isAnalogStickMainValid = false;
        m_ValidAnalogStick.isAnalogStickSubValid  = false;
        break;
    case DeviceType_Right:
    case DeviceType_MiyabiRight:
        m_ValidAnalogStick.isAnalogStickMainValid = false;
        m_ValidAnalogStick.isAnalogStickSubValid  = true;
        break;
    case DeviceType_Left:
    case DeviceType_MiyabiLeft:
        m_ValidAnalogStick.isAnalogStickMainValid = true;
        m_ValidAnalogStick.isAnalogStickSubValid  = false;
        break;
    case DeviceType_FullKey:
    case DeviceType_Tarragon:
        m_ValidAnalogStick.isAnalogStickMainValid = true;
        m_ValidAnalogStick.isAnalogStickSubValid  = true;
        break;
    default:NN_UNEXPECTED_DEFAULT;
    }

    m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;
    m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;
    m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_None;

    // 接続時のボタン押下状態を取得
    m_pCommand->GetButtonTriggerElapsedTime(this);

    // CAL 値の読み出し処理を開始する。読み出しは以下の順番で行われる
    // Model1 -> Model2 -> UserCal1 -> Cal2
    //                              -> UserCal2 or Cal1
    m_pCommand->ReadModel1(this);
}

//!< 接続直後の初期化処理が完了したかどうかを確認する関数。初期化が完了したら必ずtrueを返してください
bool PadInput::IsActivated() NN_NOEXCEPT
{
    // CAL 値の読み出しが完了していれば、終了
    return (
    m_Config.isSensorCalReadDone &&
    m_Config.isStickCalReadDone &&
    m_Config.isLeftStickCalibrated &&
    m_Config.isRightStickCalibrated
    );
};

void PadInput::Deactivate() NN_NOEXCEPT
{
    m_Activated = false;
}


void PadInput::ParseInputReport(const uint8_t* pBuffer, size_t size, int samplesSinceLast, SensorSleepValueType sensor, int validSensorCount) NN_NOEXCEPT
{

    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES((sensor == SensorSleepValueType_Sleep && size >= InputReportSize_PadBasic) ||
                    (sensor != SensorSleepValueType_Sleep && size >= InputReportSize_PadFull),
                    "Invalid size(%d) set", size);
    NN_UNUSED(size);

    m_State.sampleNumber = (m_State.sampleNumber + samplesSinceLast) % (std::numeric_limits<uint8_t>().max() + 1);

    auto bit = nn::util::BitPack8();

    bit.SetMaskedBits(0xFF, pBuffer[InputReportByte_Button]);
    m_State.buttons.Set<PadButton::Y>       (bit.GetBit(0));
    m_State.buttons.Set<PadButton::X>       (bit.GetBit(1));
    m_State.buttons.Set<PadButton::B>       (bit.GetBit(2));
    m_State.buttons.Set<PadButton::A>       (bit.GetBit(3));
    m_State.buttons.Set<PadButton::SR>      (bit.GetBit(4));
    m_State.buttons.Set<PadButton::SL>      (bit.GetBit(5));
    m_State.buttons.Set<PadButton::R>       (bit.GetBit(6));
    m_State.buttons.Set<PadButton::ZR>      (bit.GetBit(7));

    bit.SetMaskedBits(0xFF, pBuffer[InputReportByte_Button + 1]);
    m_State.buttons.Set<PadButton::Select>  (bit.GetBit(0));
    m_State.buttons.Set<PadButton::Start>   (bit.GetBit(1));
    m_State.buttons.Set<PadButton::StickR>  (bit.GetBit(2));
    m_State.buttons.Set<PadButton::StickL>  (bit.GetBit(3));
    m_State.buttons.Set<PadButton::Home>    (bit.GetBit(4));
    m_State.buttons.Set<PadButton::Shot>    (bit.GetBit(5));
//    m_State.buttons.Set<PadButton::Mute>    (bit.GetBit(6));
//    m_State.buttons.Set<PadButton::Reserv>  (bit.GetBit(7));

    bit.SetMaskedBits(0xFF, pBuffer[InputReportByte_Button + 2]);
    m_State.buttons.Set<PadButton::Down>    (bit.GetBit(0));
    m_State.buttons.Set<PadButton::Up>      (bit.GetBit(1));
    m_State.buttons.Set<PadButton::Right>   (bit.GetBit(2));
    m_State.buttons.Set<PadButton::Left>    (bit.GetBit(3));
    // SL,SR はorをByte1のbit4, 5とのORをとる
    m_State.buttons.Set<PadButton::SR>      (bit.GetBit(4) || m_State.buttons.Test<PadButton::SR>());
    m_State.buttons.Set<PadButton::SL>      (bit.GetBit(5) || m_State.buttons.Test<PadButton::SL>());
    m_State.buttons.Set<PadButton::L>       (bit.GetBit(6));
    m_State.buttons.Set<PadButton::ZL>      (bit.GetBit(7));

    const uint8_t InvalidData[3] = { 0x00, 0x00, 0x00 };

    if (m_ValidAnalogStick.isAnalogStickMainValid)
    {
        ParseAnalogStick(&(m_State.analogStickL), &pBuffer[InputReportByte_AnalogStickMain]);
    }
    else
    {
        ParseAnalogStick(&(m_State.analogStickL), InvalidData);
    }

    if (m_ValidAnalogStick.isAnalogStickSubValid)
    {
        ParseAnalogStick(&(m_State.analogStickR), &pBuffer[InputReportByte_AnalogStickSub]);
    }
    else
    {
        ParseAnalogStick(&(m_State.analogStickR), InvalidData);
    }

    // CAL 値読み出しが完了し、それでもスティックが CAL されていない場合は、取得位置を原点とする
    if (m_Config.isStickCalReadDone == true)
    {
        if (m_Config.isLeftStickCalibrated == false)
        {
            m_Config.analogStickLValidRange.origin = m_State.analogStickL;
            m_Config.isLeftStickCalibrated = true;
        }
        if (m_Config.isRightStickCalibrated == false)
        {
            m_Config.analogStickRValidRange.origin = m_State.analogStickR;
            m_Config.isRightStickCalibrated = true;
        }
    }

    if (sensor != SensorSleepValueType_Sleep)
    {
        ::std::lock_guard<decltype(m_SensorMutex)
                          > locker(m_SensorMutex);

            if (sensor == SensorSleepValueType_Dscale1 ||
            sensor == SensorSleepValueType_Dscale1Payload6 ||
            sensor == SensorSleepValueType_Dscale1Payload12 ||
            sensor == SensorSleepValueType_Dscale2)
        {
            /* In LEGACY code, sampleSinceLast represent the number of new "different" samples inside one BT packet.
               (See explanation below)
               In DScale mode, we use the whole packet each time, because we encoded data on all of it.
             */
            int const samplesPerState = 3;
            NN_ABORT_UNLESS(validSensorCount == samplesPerState);

            int const newStates = validSensorCount / samplesPerState;
            for (int s = 0; s < newStates; ++s)
            {
                // Read the buffer
                SixAxisSensorRawParsing sixAxisStateForDScale[samplesPerState];
                for (int i = 0; i < samplesPerState; ++i)
                {
                    auto& sensorState = sixAxisStateForDScale[i];
                    auto const startByte = (s * samplesPerState) + i * InputReportSize_SixAxisSensor;
                    auto const startByteAccl = InputReportByte_Accelerometer + startByte;
                    auto const startByteGyro = InputReportByte_Gyroscope + startByte;
                    sensorState.sampleNumber = m_LastSampleNumberForBuffer + (s * samplesPerState) + i;

                    ParseSensorData(&sensorState.data0, 4, &pBuffer[startByteAccl]);
                    ParseSensorData(&sensorState.data1, 1, &pBuffer[startByteGyro]);
                }

                // Send it through DScale
                SixAxisSensorState outputStateForHid[samplesPerState];
                uint32_t outputStateForHidCount = 0;
                m_DScaleProcessor.InsertOneState(
                    sixAxisStateForDScale,
                    outputStateForHid,
                    outputStateForHidCount,
                    sensor); // Inside, Static assertion verifies that we pass the correct Mode

                // Write result
                for (uint32_t i = 0; i < outputStateForHidCount; ++i)
                {
                    m_SensorBufferIndex++;
                    if (m_SensorBufferIndex == SixAxisSensorSampleCountMax)
                    {
                        m_SensorBufferIndex = 0;
                    }
                    NN_ABORT_UNLESS_LESS(m_SensorBufferIndex, SixAxisSensorSampleCountMax);

                    m_SensorStates[m_SensorBufferIndex] = outputStateForHid[i];
                    m_SensorBufferCount++;
                }
            }
            m_LastSampleNumberForBuffer += samplesPerState; //Actually never used for DScale mode
        }
        else
        {
            NN_ABORT_UNLESS(sensor == SensorSleepValueType_Active);
            /* In LEGACY code, sampleSinceLast represent the number of new "different" samples inside one BT packet.
            In 15ms sniff mode, the BT packets contains roughly 3 new samples:
            - A B C
            - D E F
            - G H I
            In 5ms sniff mode, the BT packets contains roughly 1 new sample:
            - A B C
            - B C D (D is the newest)
            - C D E (E is the newest)
            And of course, we cannot extract more than what is inside the BT packet, so validSensorCount.
            */
            auto newSamples = (samplesSinceLast > validSensorCount) ? validSensorCount : samplesSinceLast;
            m_LastSampleNumberForBuffer += (samplesSinceLast - newSamples);
            for (int i = newSamples - 1; i >= 0; i--)
            {
                m_SensorBufferIndex++;
                if (m_SensorBufferIndex == SixAxisSensorSampleCountMax)
                {
                    m_SensorBufferIndex = 0;
                }
                NN_ABORT_UNLESS_LESS(m_SensorBufferIndex, SixAxisSensorSampleCountMax);

                auto& sensorState = m_SensorStates[m_SensorBufferIndex];
                auto startByte = InputReportByte_Accelerometer + i * InputReportSize_SixAxisSensor;
                sensorState.sampleNumber = ++m_LastSampleNumberForBuffer;
                ParseSensorData(&sensorState.accelerometer, 4, &pBuffer[startByte]);
                ParseSensorData(&sensorState.gyroscope, 1, &pBuffer[startByte + 6]);
                sensorState.deltaTimeInSec = 0.005f; // TODO replace with a constant ?

                if (m_SensorBufferCount < SixAxisSensorSampleCountMax)
                {
                    m_SensorBufferCount++;
                }

                // 旧実装に対する互換性維持。削除予定
                auto oldStartByte = InputReportByte_Accelerometer + i*InputReportSize_SixAxisSensor;
                ParseSensorData(&m_State.accelerometer[i], 4, &pBuffer[oldStartByte]);
                ParseSensorData(&m_State.gyroscope[i], 1, &pBuffer[oldStartByte + 6]);
            }

            for (int i = newSamples; i < validSensorCount; i++)
            {
                // 旧実装に対する互換性維持。削除予定
                m_State.accelerometer[i].x = 0;
                m_State.accelerometer[i].y = 0;
                m_State.accelerometer[i].z = 0;
                m_State.gyroscope[i].x = 0;
                m_State.gyroscope[i].y = 0;
                m_State.gyroscope[i].z = 0;
            }
        }
    }
    else
    {
        m_State.sensorSampleCount = 0;
    }
} //NOLINT(impl/function_size)

size_t PadInput::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_UNUSED(pOutValue);
    NN_UNUSED(size);

    // Padについては現状Outputはなし
    return 0;
}

Result PadInput::GetPadState(PadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    memcpy(pOutValue, &m_State, sizeof(PadState));
    NN_RESULT_SUCCESS;
}

template<typename BufferType>
Result GetSensorStatesImpl(
    int* pOutCount,
    BufferType* pOutValue,
    int length,
    BufferType const aBuffer[SixAxisSensorSampleCountMax],
    int iSensorBufferIndex,
    int iSensorBufferCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER(length, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(length, SixAxisSensorSampleCountMax);

    int64_t lastSamplingNumber = 0;
    *pOutCount = 0;

    if (iSensorBufferCount == 0)
    {
        NN_RESULT_SUCCESS;
    }

    for (int i = 0; i < length; i++)
    {
        auto sensorIndex = (iSensorBufferIndex + SixAxisSensorSampleCountMax - i) % SixAxisSensorSampleCountMax;
        auto sensorState = aBuffer[sensorIndex];

        if (lastSamplingNumber < sensorState.sampleNumber && lastSamplingNumber != 0)
        {
            break;
        }

        pOutValue[i] = sensorState;
        (*pOutCount)++;

        if (length == *pOutCount)
        {
            break;
        }

        lastSamplingNumber = sensorState.sampleNumber;
    }

    NN_ABORT_UNLESS_GREATER(*pOutCount, 0);
    NN_ABORT_UNLESS_LESS_EQUAL(*pOutCount, length);

    NN_RESULT_SUCCESS;
}

Result PadInput::GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_SensorMutex)> locker(m_SensorMutex);
    return GetSensorStatesImpl(pOutCount, pOutValue, length, m_SensorStates, m_SensorBufferIndex, m_SensorBufferCount);
}

Result PadInput::GetButtonTriggerElapsedTime(nn::os::Tick* pOutValue, int buttonIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(buttonIndex == PadButton::L::Index  ||
                    buttonIndex == PadButton::R::Index  ||
                    buttonIndex == PadButton::ZL::Index ||
                    buttonIndex == PadButton::ZR::Index ||
                    buttonIndex == PadButton::SL::Index ||
                    buttonIndex == PadButton::SR::Index, "Only L, R, ZL, ZR, SL, SR buttons are allowed");

    int index = 0;

    switch (buttonIndex)
    {
    case PadButton::L::Index:
        index = 0;
        break;

    case PadButton::R::Index:
        index = 1;
        break;

    case PadButton::ZL::Index:
        index = 2;
        break;

    case PadButton::ZR::Index:
        index = 3;
        break;

    case PadButton::SL::Index:
        index = 4;
        break;

    case PadButton::SR::Index:
        index = 5;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (m_ButtonTriggers[index].triggered == true)
    {
        *pOutValue = m_ButtonTriggers[index].time;
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(ResultPadButtonNotTriggered());
}

void PadInput::SleepSensor(bool sleep) NN_NOEXCEPT
{
    m_TemporalConfig.sensorSleep = sleep;

    m_pCommand->SensorSleep(sleep, this);
}

void PadInput::IsSensorSleep(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_Config.sensorSleep;
}

void PadInput::SetSensorConfig(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyroscopeFsr) NN_NOEXCEPT
{
    m_TemporalConfig.accelerometerFsr = accelerometerFsr;
    m_TemporalConfig.gyroscopeFsr = gyroscopeFsr;

    m_pCommand->SensorConfig(accelerometerFsr, gyroscopeFsr, this);
}

void PadInput::GetSensorConfig(AccelerometerFsr* pOutAccelerometerFsr, GyroscopeFsr* pOutGyroscopeFsr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAccelerometerFsr);
    NN_SDK_REQUIRES_NOT_NULL(pOutGyroscopeFsr);

    *pOutAccelerometerFsr = m_Config.accelerometerFsr;
    *pOutGyroscopeFsr = m_Config.gyroscopeFsr;
}

void PadInput::GetSensorCalibrationValue(SensorCalibrationValue* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_Config.sensorCalibrationValue;
}


Result PadInput::UpdateSensorCalibrationValue(const SensorCalibrationValue& value) NN_NOEXCEPT
{
    m_TemporalConfig.sensorCalibrationValue = value;
    NN_RESULT_DO(m_pCommand->UpdateSixAxisSensorUserCal(value, this));
    m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_OnUpdate;

    NN_RESULT_SUCCESS;
}

Result PadInput::ResetSensorCalibrationValue() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pCommand->DeleteSixAxisSensorUserCal(this));
    m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_OnReset;

    NN_RESULT_SUCCESS;
}

CalibrationUpdateStatus PadInput::GetSensorCalibrationValueUpdateStatus() NN_NOEXCEPT
{
    return m_SensorCalibrationUpdateStatus;
}

SensorState PadInput::GetSensorHorizontalOffset() NN_NOEXCEPT
{
    return m_SensorHorizontalOffset;
}

void PadInput::GetLeftAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    *pOutValue = m_Config.analogStickLValidRange;
}

Result PadInput::UpdateLeftAnalogStickValidRange(const AnalogStickValidRange& value) NN_NOEXCEPT
{
    m_TemporalConfig.analogStickLValidRange = value;
    m_TemporalConfig.analogStickLValidRange.originPlay = m_Config.analogStickLValidRange.originPlay;
    NN_RESULT_DO(m_pCommand->UpdateLeftAnalogStickUserCal(value, this));
    m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_OnUpdate;

    NN_RESULT_SUCCESS;
}

Result PadInput::ResetLeftAnalogStickValidRange() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pCommand->DeleteLeftAnalogStickUserCal(this));
    m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_OnReset;

    NN_RESULT_SUCCESS;
}

CalibrationUpdateStatus PadInput::GetLeftAnalogStickValidRangeUpdateStatus() NN_NOEXCEPT
{
    return m_LeftAnalogStickValidRangeUpdateStatus;
}

AnalogStickDeviceParameter PadInput::GetLeftAnalogStickDeviceParameter() NN_NOEXCEPT
{
    return m_StickLeftStaticValues.deviceParameter;
}

void PadInput::GetRightAnalogStickValidRange(AnalogStickValidRange* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_Config.analogStickRValidRange;
}

Result PadInput::UpdateRightAnalogStickValidRange(const AnalogStickValidRange& value) NN_NOEXCEPT
{
    m_TemporalConfig.analogStickRValidRange = value;
    m_TemporalConfig.analogStickRValidRange.originPlay = m_Config.analogStickRValidRange.originPlay;
    NN_RESULT_DO(m_pCommand->UpdateRightAnalogStickUserCal(value, this));
    m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_OnUpdate;

    NN_RESULT_SUCCESS;
}

Result PadInput::ResetRightAnalogStickValidRange() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pCommand->DeleteRightAnalogStickUserCal(this));
    m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_OnReset;

    NN_RESULT_SUCCESS;
}

CalibrationUpdateStatus PadInput::GetRightAnalogStickValidRangeUpdateStatus() NN_NOEXCEPT
{
    return m_RightAnalogStickValidRangeUpdateStatus;
}

AnalogStickDeviceParameter PadInput::GetRightAnalogStickDeviceParameter() NN_NOEXCEPT
{
    return m_StickRightStaticValues.deviceParameter;
}

void PadInput::NotifyAck(Result result, uint8_t id) NN_NOEXCEPT
{
    // SensorConfig, SensorSleepはResultのハンドリングはなし
    NN_UNUSED(result);

    // SensorConfigに対するAck
    if(id == Command_SensorConfig.Id)
    {
        m_Config.accelerometerFsr = m_TemporalConfig.accelerometerFsr;
        m_Config.gyroscopeFsr = m_TemporalConfig.gyroscopeFsr;
    }
    // SensorSleepに対するAck
    else if(id == Command_SensorSleep.Id)
    {
        m_Config.sensorSleep = m_TemporalConfig.sensorSleep;
    }
}

void PadInput::NotifyButtonTriggerElapsedTime(const ButtonTriggerElapsedTime& value) NN_NOEXCEPT
{
    for (uint16_t i = 0; i < 6; i++)
    {
        if (value.time[i] == 0)
        {
            m_ButtonTriggers[i].triggered = false;
        }
        else
        {
            m_ButtonTriggers[i].triggered = true;
            m_ButtonTriggers[i].time = nn::os::GetSystemTick() - nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(value.time[i] * 10));
        }
    }

    m_Activated = true;
}

void PadInput::NotifyCal1(const SensorCalibrationValue& value) NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] Read CAL1\n");

    if(ValidateCalibrationValue(value) == true)
    {
        m_Config.sensorCalibrationValue = value;
    }
    else
    {
        PAD_DUMP_LOG("    -> Invalid SixAxisSensor CAL\n");
        m_Config.sensorCalibrationValue = DefaultSensorCalibration;
    }

    m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_None;

    // センサーのCAL値の状態に限らず CAL 値読み出しは完了
    m_Config.isSensorCalReadDone = true;
}

void PadInput::NotifyCal2(const AnalogStickValidRange& leftValue,
                          const AnalogStickValidRange& rightValue,
                          ::nn::util::Color4u8Type& mainColor,
                          ::nn::util::Color4u8Type& subColor) NN_NOEXCEPT
{
    NN_UNUSED(mainColor);
    NN_UNUSED(subColor);

    PAD_DUMP_LOG("[xcd] Read Done CAL2\n");

    struct AnalogStickRangeImpl
    {
        AnalogStickValidRange received;    // 読み込んだスティックの値
        AnalogStickValidRange* pOutput;    // 読み込んだ値をもとに値を設定するバッファ
        AnalogStickStaticValues statics;   // CAL 値計算のための静的なパラメータ
        bool* isCalibrated;                // キャリブレーション済みかどうかを保持するための、bool へのポインタ
        CalibrationUpdateStatus* userCalibratoinStatus; // ユーザーキャリブレーションの状況
    };
    AnalogStickRangeImpl stickImpls[] = {
        {leftValue, &m_Config.analogStickLValidRange, m_StickLeftStaticValues, &m_Config.isLeftStickCalibrated, &m_LeftAnalogStickValidRangeUpdateStatus},
        {rightValue, &m_Config.analogStickRValidRange, m_StickRightStaticValues, &m_Config.isRightStickCalibrated, &m_RightAnalogStickValidRangeUpdateStatus}
    };

    for (auto& stickImpl : stickImpls)
    {
        // キャリブレーションのリセット処理中はリセットが行われたものだけ値を更新する
        if ((m_LeftAnalogStickValidRangeUpdateStatus == CalibrationUpdateStatus_OnReset || m_RightAnalogStickValidRangeUpdateStatus == CalibrationUpdateStatus_OnReset) &&
            *stickImpl.userCalibratoinStatus != CalibrationUpdateStatus_OnReset)
        {
            continue;
        }

        // ユーザーCAL 済みの場合はスキップ
        if (*(stickImpl.isCalibrated) == true)
        {
            continue;
        }
        if (ValidateAnalogStickValidRange(stickImpl.received, stickImpl.statics) == true && m_SerialFlashFormatVersion >= SerialFlashVersion_AnalogStickValid)
        {
            *stickImpl.pOutput = stickImpl.received;
            stickImpl.pOutput->originPlay = stickImpl.statics.originPlay;
            AddPlayOnStickCircuit(stickImpl.pOutput,
                                  stickImpl.statics);
            *stickImpl.isCalibrated = true;
        }
        else
        {
            stickImpl.pOutput->origin.x = 0;
            stickImpl.pOutput->origin.y = 0;

            stickImpl.pOutput->circuitMin = stickImpl.statics.deviceParameter.minimumStrokePositive;
            stickImpl.pOutput->circuitMax = stickImpl.statics.deviceParameter.minimumStrokeNegative;
            // 中央不感帯 の2倍確保
            stickImpl.pOutput->originPlay = stickImpl.statics.originPlay * 2;

            *stickImpl.isCalibrated = false;
            PAD_DUMP_LOG("    -> CAL invalid\n");
        }
    }

    m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;
    m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;

    m_Config.isStickCalReadDone = true;
}

void PadInput::NotifyModel1(const SensorState& sensorHorizontalOffset,
                            const AnalogStickDeviceParameter& leftStickParam,
                            const uint16_t leftOriginPlay,
                            const uint16_t leftCircuitValidRatio,
                            const bool     leftIsXScalingRequired) NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] Read Model1\n");

    if (m_SerialFlashFormatVersion >= SerialFlashVersion_AnalogStickValid)
    {
        m_SensorHorizontalOffset = sensorHorizontalOffset;
        // Scaling の要/不要は ジョイコンだけ
        m_StickLeftStaticValues.isScalingRequired = (m_Type == DeviceType_Left || m_Type == DeviceType_Right) ? leftIsXScalingRequired : false;
        m_StickLeftStaticValues.originPlay = leftOriginPlay;
        m_StickLeftStaticValues.circuitValidRatio = leftCircuitValidRatio;
        m_StickLeftStaticValues.deviceParameter = leftStickParam;
    }
    else
    {
        PAD_DUMP_LOG("    -> Model Info Invalid\n");

        switch (m_Type)
        {
        case DeviceType_Left:
        case DeviceType_MiyabiLeft:
            m_SensorHorizontalOffset = JoyLeftSensorHorizontalOffsetTypical;
            m_StickLeftStaticValues = (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_025b) ?
                JoyAnalogStickValues :
                JoyAnalogStickValuesOld;
            break;
        case DeviceType_Right:
        case DeviceType_MiyabiRight:
            m_SensorHorizontalOffset = JoyRightSensorHorizontalOffsetTypical;
            m_StickLeftStaticValues = (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_025b) ?
                JoyAnalogStickValues :
                JoyAnalogStickValuesOld;
            break;
        case DeviceType_FullKey:
        case DeviceType_Tarragon:
            m_SensorHorizontalOffset = FullKeySensorHorizontalOffsetTypical;
            m_StickLeftStaticValues = (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_025b) ?
                FullKeyAnalogStickValues :
                FullKeyAnalogStickValuesOld;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // モデル情報 2 の読み出しへ
    m_pCommand->ReadModel2(this);
}

void PadInput::NotifyModel2(const AnalogStickDeviceParameter& rightStickParam,
                            const uint16_t rightOriginPlay,
                            const uint16_t rightCircuitValidRatio,
                            const bool     rightIsXScalingRequired) NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] Read MODEL2\n");

    if (m_SerialFlashFormatVersion >= SerialFlashVersion_AnalogStickValid)
    {
        // Scaling の要/不要は ジョイコンだけ
        m_StickRightStaticValues.isScalingRequired = (m_Type == DeviceType_Left || m_Type == DeviceType_Right) ? rightIsXScalingRequired : false;
        m_StickRightStaticValues.originPlay = rightOriginPlay;
        m_StickRightStaticValues.circuitValidRatio = rightCircuitValidRatio;
        m_StickRightStaticValues.deviceParameter = rightStickParam;
    }
    else
    {
        PAD_DUMP_LOG("    -> Model Info Invalid\n");
        switch (m_Type)
        {
        case DeviceType_Left:
        case DeviceType_Right:
        case DeviceType_MiyabiLeft:
        case DeviceType_MiyabiRight:
            m_StickRightStaticValues = (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_025b) ?
                JoyAnalogStickValues :
                JoyAnalogStickValuesOld;
            break;
        case DeviceType_FullKey:
        case DeviceType_Tarragon:
            m_StickRightStaticValues = (m_FirmwareVersion.bluetooth >= FirmwareVersionBt_025b) ?
                FullKeyAnalogStickValues :
                FullKeyAnalogStickValuesOld;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // UserCal1 の読み出しへ
    m_pCommand->ReadUserCal1(this);
}

void PadInput::NotifyUserCal1(const bool isLeftStickAvailable,
                              const AnalogStickValidRange& leftValue,
                              const bool isRightStickAvailable,
                              const AnalogStickValidRange& rightValue,
                              const bool isSensorAvailable) NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] Read USERCAL1\n");

    struct AnalogStickRangeImpl
    {
        AnalogStickValidRange received;    // 読み込んだスティックの値
        AnalogStickValidRange* pOutput;    // 読み込んだ値をもとに値を設定するバッファ
        AnalogStickStaticValues statics;   // CAL 値計算のための静的なパラメータ
        bool isUserCalAvailable;           // ユーザー CAL 情報が有効かどうか
        bool* isCalibrated;                // キャリブレーション済みかどうかを保持するための、bool へのポインタ
    };
    AnalogStickRangeImpl stickImpls[] = {
        {leftValue, &m_Config.analogStickLValidRange, m_StickLeftStaticValues, isLeftStickAvailable, &m_Config.isLeftStickCalibrated },
        {rightValue, &m_Config.analogStickRValidRange, m_StickRightStaticValues, isRightStickAvailable, &m_Config.isRightStickCalibrated}
    };

    for (auto& stickImpl : stickImpls)
    {
        if (ValidateAnalogStickValidRange(stickImpl.received, stickImpl.statics) == true && m_SerialFlashFormatVersion >= SerialFlashVersion_AnalogStickValid && stickImpl.isUserCalAvailable)
        {
            *stickImpl.pOutput = stickImpl.received;
            stickImpl.pOutput->originPlay = stickImpl.statics.originPlay;
            AddPlayOnStickCircuit(stickImpl.pOutput,
                                  stickImpl.statics);
            *stickImpl.isCalibrated = true;
        }
        else
        {
            *stickImpl.isCalibrated = false;
        }
    }

    if (m_Config.isLeftStickCalibrated && m_Config.isRightStickCalibrated)
    {
        // 両方ともキャリブレーションされていれば、スティックの CAL 値読み出しは完了
        m_Config.isStickCalReadDone = true;
    }

    // 色情報があるため、アナログスティックの CAL の有無関係なく CAL2 領域は Read する
    m_pCommand->ReadCal2(this);

    // センサーの CAL 値読み出しは、ユーザー CAL の有無で分岐
    if (isSensorAvailable)
    {
        PAD_DUMP_LOG("    -> Sensor USERCAL exists\n");
        // 6軸センサーのユーザー CAL 値が存在する
        m_pCommand->ReadUserCal2(this);
    }
    else
    {
        PAD_DUMP_LOG("    -> Sensor USERCAL do not exists\n");
        // 6軸センサーのユーザー CAL 値が存在しないため、工程 CAL 値を読み出す
        m_pCommand->ReadCal1(this);
    }
}

void PadInput::NotifyUserCal2(const SensorCalibrationValue& value) NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] Read USERCAL2\n");

    if(ValidateCalibrationValue(value) == true)
    {
        m_Config.sensorCalibrationValue = value;
        m_Config.isSensorCalReadDone = true;
    }
    else
    {
        PAD_DUMP_LOG("    -> Invalid. Read CAL1\n");
        // データが壊れているため、工程 CAL 値を読み出す
        m_pCommand->ReadCal1(this);
    }
}

void PadInput::NotifyUpdateSixAxisSensorUserCal(Result result) NN_NOEXCEPT
{
    if (m_SensorCalibrationUpdateStatus == CalibrationUpdateStatus_OnReset && result.IsSuccess())
    {
        // 工程キャリブレーション値を読む
        m_pCommand->ReadCal1(this);
        return;
    }

    if (result.IsSuccess())
    {
        m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_None;
        m_Config.sensorCalibrationValue = m_TemporalConfig.sensorCalibrationValue;
    }
    else if (ResultSerialFlashVerifyError().Includes(result))
    {
        m_SensorCalibrationUpdateStatus = CalibrationUpdateStatus_VerifyError;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
};

void PadInput::NotifyUpdateLeftAnalogStickUserCal(Result result) NN_NOEXCEPT
{
    if (m_LeftAnalogStickValidRangeUpdateStatus == CalibrationUpdateStatus_OnReset && result.IsSuccess())
    {
        // 工程キャリブレーション値を読む
        m_Config.isStickCalReadDone = false;
        m_Config.isLeftStickCalibrated = false;
        m_pCommand->ReadCal2(this);
        return;
    }

    if (result.IsSuccess())
    {
        m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;
        m_Config.analogStickLValidRange = m_TemporalConfig.analogStickLValidRange;
        AddPlayOnStickCircuit(&(m_Config.analogStickLValidRange),
            m_StickLeftStaticValues);
    }
    else if (ResultSerialFlashVerifyError().Includes(result))
    {
        m_LeftAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_VerifyError;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
};

void PadInput::NotifyUpdateRightAnalogStickUserCal(Result result) NN_NOEXCEPT
{
    if (m_RightAnalogStickValidRangeUpdateStatus == CalibrationUpdateStatus_OnReset && result.IsSuccess())
    {
        // 工程キャリブレーション値を読む
        m_Config.isRightStickCalibrated = false;
        m_Config.isStickCalReadDone = false;
        m_pCommand->ReadCal2(this);
        return;
    }

    if (result.IsSuccess())
    {
        m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_None;
        m_Config.analogStickRValidRange = m_TemporalConfig.analogStickRValidRange;
        AddPlayOnStickCircuit(&(m_Config.analogStickRValidRange),
            m_StickRightStaticValues);
    }
    else if (ResultSerialFlashVerifyError().Includes(result))
    {
        m_RightAnalogStickValidRangeUpdateStatus = CalibrationUpdateStatus_VerifyError;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
};

void PadInput::SetSerialFlashFormatVersion(int version) NN_NOEXCEPT
{
    m_SerialFlashFormatVersion = version;
}

bool PadInput::ValidateAnalogStickValidRange(const AnalogStickValidRange& value, const AnalogStickStaticValues& staticValues) NN_NOEXCEPT
{
    return (ValidateRange(value.origin.x,     staticValues.deviceParameter.originRangeMin.x, staticValues.deviceParameter.originRangeMax.x) &&
            ValidateRange(value.origin.y,     staticValues.deviceParameter.originRangeMin.y, staticValues.deviceParameter.originRangeMax.y) &&
            ValidateRange(value.circuitMin.x, staticValues.deviceParameter.minimumStrokeNegative.x, StickMax) &&
            ValidateRange(value.circuitMin.y, staticValues.deviceParameter.minimumStrokeNegative.y, StickMax) &&
            ValidateRange(value.circuitMax.x, staticValues.deviceParameter.minimumStrokePositive.x, StickMax) &&
            ValidateRange(value.circuitMax.y, staticValues.deviceParameter.minimumStrokePositive.y, StickMax));
};

void PadInput::PrintDeviceInformation() NN_NOEXCEPT
{
    PAD_DUMP_LOG("[xcd] SixAxisSensor");
    PAD_DUMP_LOG("\n    -> acc origin :");
    DumpSensorValue(m_Config.sensorCalibrationValue.accelerometerOrigin);
    PAD_DUMP_LOG("\n    -> acc scale  :");
    DumpSensorValue(m_Config.sensorCalibrationValue.accelerometerSensitivity);
    PAD_DUMP_LOG("\n    -> gyro origin:");
    DumpSensorValue(m_Config.sensorCalibrationValue.gyroscopeOrigin);
    PAD_DUMP_LOG("\n    -> gyro scale :");
    DumpSensorValue(m_Config.sensorCalibrationValue.gyroscopeSensitivity);
    PAD_DUMP_LOG("\n");

    PAD_DUMP_LOG("[xcd] LeftAnalogStick");
    PAD_DUMP_LOG("\n    -> originPlay: %d", m_Config.analogStickLValidRange.originPlay);
    PAD_DUMP_LOG("\n    -> origin: ");
    DumpStickValue(m_Config.analogStickLValidRange.origin);
    PAD_DUMP_LOG("\n    -> circuitMin: ");
    DumpStickValue(m_Config.analogStickLValidRange.circuitMin);
    PAD_DUMP_LOG("\n    -> circuitMax: ");
    DumpStickValue(m_Config.analogStickLValidRange.circuitMax);
    PAD_DUMP_LOG("\n");

    PAD_DUMP_LOG("[xcd] RightAnalogStick");
    PAD_DUMP_LOG("\n    -> originPlay: %d", m_Config.analogStickRValidRange.originPlay);
    PAD_DUMP_LOG("\n    -> origin: ");
    DumpStickValue(m_Config.analogStickRValidRange.origin);
    PAD_DUMP_LOG("\n    -> circuitMin: ");
    DumpStickValue(m_Config.analogStickRValidRange.circuitMin);
    PAD_DUMP_LOG("\n    -> circuitMax: ");
    DumpStickValue(m_Config.analogStickRValidRange.circuitMax);
    PAD_DUMP_LOG("\n");

    PAD_DUMP_LOG("[xcd] SensorModelInfo");
    PAD_DUMP_LOG("\n    -> horizontal offset :");
    DumpSensorValue(m_SensorHorizontalOffset);
    PAD_DUMP_LOG("\n");

    PAD_DUMP_LOG("[xcd] LeftAnalogStickModelInfo");
    PAD_DUMP_LOG("\n    -> originPlay:        %d", m_StickLeftStaticValues.originPlay);
    PAD_DUMP_LOG("\n    -> circuitValidRatio: %d", m_StickLeftStaticValues.circuitValidRatio);
    PAD_DUMP_LOG("\n    -> originRangeMin: ");
    DumpStickValue(m_StickLeftStaticValues.deviceParameter.originRangeMin);
    PAD_DUMP_LOG("\n    -> originRangeMax: ");
    DumpStickValue(m_StickLeftStaticValues.deviceParameter.originRangeMax);
    PAD_DUMP_LOG("\n    -> minimumStrokeNegative: ");
    DumpStickValue(m_StickLeftStaticValues.deviceParameter.minimumStrokeNegative);
    PAD_DUMP_LOG("\n    -> minimumStrokePositive: ");
    DumpStickValue(m_StickLeftStaticValues.deviceParameter.minimumStrokePositive);
    PAD_DUMP_LOG("\n");

    PAD_DUMP_LOG("[xcd] RightAnalogStickModelInfo");
    PAD_DUMP_LOG("\n    -> originPlay:        %d", m_StickRightStaticValues.originPlay);
    PAD_DUMP_LOG("\n    -> circuitValidRatio: %d", m_StickRightStaticValues.circuitValidRatio);
    PAD_DUMP_LOG("\n    -> originRangeMin: ");
    DumpStickValue(m_StickRightStaticValues.deviceParameter.originRangeMin);
    PAD_DUMP_LOG("\n    -> originRangeMax: ");
    DumpStickValue(m_StickRightStaticValues.deviceParameter.originRangeMax);
    PAD_DUMP_LOG("\n    -> minimumStrokeNegative: ");
    DumpStickValue(m_StickRightStaticValues.deviceParameter.minimumStrokeNegative);
    PAD_DUMP_LOG("\n    -> minimumStrokePositive: ");
    DumpStickValue(m_StickRightStaticValues.deviceParameter.minimumStrokePositive);
    PAD_DUMP_LOG("\n");
}

}} // namespace nn::xcd
