﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include "xcd_IrsensorBase.h"
#include "xcd_IrsensorClusteringProcessor.h"

namespace nn { namespace xcd {

const float IrsensorClusteringProcessor::ClusteringAverageIntensityDivider = 256.0f;
const float IrsensorClusteringProcessor::ClusteringCentroidDivider         = 64.0f;

IrsensorClusteringProcessor::~IrsensorClusteringProcessor() NN_NOEXCEPT
{
    // 何もしない
}

Result IrsensorClusteringProcessor::ParseClusteringData(const uint8_t* pBuffer, size_t size, uint8_t sampleNumber) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);
    NN_UNUSED(sampleNumber);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ImageHeader の情報を取得する
    ParseIrsensorHeader(pBuffer, size, sampleNumber);

    // とりあえず常に先頭バッファを上書きする
    // TODO:通信レートなどに合わせて取得するパケット数を設定する
    m_ValidStatesCnt = 0;

    // Timestampの更新
    m_pClusteringProcessorStates[m_ValidStatesCnt]->samplingNumber = sampleNumber;
    // diffTimestampの更新
    m_pClusteringProcessorStates[m_ValidStatesCnt]->diffTimeStampCount = m_DiffTimeStamp;

    // コピーを行う。内部でデータチェックする。
    // (IAAA-2362) ヘッダは正しいが、中身が不正なパケットが送られてくる問題対策
    Result result = CopyClusteringPacketData(pBuffer, size, m_ValidStatesCnt);
    if (result.IsSuccess())
    {
        m_ValidStatesCnt++;
    }
    else
    {
        // 途中までコピーされていた可能性があるため、クリアしておく。
        std::memset(m_pClusteringProcessorStates[m_ValidStatesCnt], 0, sizeof(IrClusteringProcessorState));
    }
    NN_SDK_ASSERT(m_ValidStatesCnt <= ClusteringSamplingCountMax);
    NN_RESULT_SUCCESS;
}

Result IrsensorClusteringProcessor::SetupProcessor(IrCommonData* pIrCommonWorkBuffer, IrClusteringProcessorState* pClusteringProcessorWorkBuffer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pClusteringProcessorWorkBuffer);

    // ワークバッファを保持する
    for (int i = 0; i < IrClusteringProcessorStateCountMax; i++)
    {
        m_pClusteringProcessorStates[i] = pClusteringProcessorWorkBuffer + i;
    }

    m_DiffTimeStamp = 0;
    m_TimeStamp = 0;
    m_IsProcessorReady = true;
    NN_RESULT_SUCCESS;
}

Result IrsensorClusteringProcessor::StopProcessor() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // ワークバッファを解放する
    for (int i = 0; i < ClusteringSamplingCountMax; i++)
    {
        m_pClusteringProcessorStates[i] = nullptr;
    }

    m_IsProcessorReady = false;
    NN_RESULT_SUCCESS;
}

Result IrsensorClusteringProcessor::GetClusteringStates(
    IrCommonData* pOutIrCommonData,
    IrClusteringProcessorState* pOutClusteringProcessorStates,
    int* pOutAckCount,
    int* pOutCount,
    int countMax) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    int copyCountMax = std::min(countMax, m_ValidStatesCnt);

    // TODO : 最新のものから順に返すようにする
    for(int i = 0; i < copyCountMax; i++)
    {
        *(pOutIrCommonData + i) = m_IrCommonData[i];
        std::memcpy((pOutClusteringProcessorStates + i), m_pClusteringProcessorStates[i], sizeof(IrClusteringProcessorState));
    }
    *pOutCount = copyCountMax;
    *pOutAckCount = m_LatestAckCount;

    // コピーが終わったらMutexを解いて、Eventを返す
    NN_RESULT_SUCCESS;
}

Result IrsensorClusteringProcessor::CopyClusteringPacketData(const uint8_t* pBuffer, size_t size, int index) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_UNUSED(size);

    if (!IsProcessorReady())
    {
        return nn::xcd::ResultIrProcessorNotReady();
    }

    // 検出できたオブジェクト数を取得
    int8_t objectCount = pBuffer[ClusteringProcessorInputReportOffset];
    if (objectCount < 0 || objectCount > nn::xcd::IrClusteringProcessorObjectCountMax)
    {
        NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: objectCount:%d\n", objectCount);
        return nn::xcd::ResultIrInvalidPacketData();
    }
    m_pClusteringProcessorStates[index]->objectCount = objectCount;

    // 各オブジェクトのデータ
    for (int i = 0; i < m_pClusteringProcessorStates[index]->objectCount; i++)
    {
        ClusterRawBlock rawBlock = {0};
        std::memcpy(&rawBlock, &pBuffer[ClusteringProcessorDataOffset + sizeof(ClusterRawBlock) * i], sizeof(ClusterRawBlock));
        float averageIntensity = static_cast<float>(rawBlock.averageIntensity / ClusteringAverageIntensityDivider);
        if (averageIntensity < 0.0f || averageIntensity > 255.0f)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: averageIntensity:%f\n", averageIntensity);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].averageIntensity = averageIntensity;

        // pixelCountについては全ての値を取りうるので、値チェックできない
        m_pClusteringProcessorStates[index]->objects[i].pixelCount = rawBlock.pixelCount;

        float centroidX = static_cast<float>(rawBlock.centroidX) / ClusteringCentroidDivider;
        if (centroidX < 0 || centroidX >= nn::xcd::IrImageWidthMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: centroidX:%f\n", centroidX);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].centroid.x = centroidX;

        float centroidY = static_cast<float>(rawBlock.centroidY) / ClusteringCentroidDivider;
        if (centroidY < 0 || centroidY >= nn::xcd::IrImageHeightMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: centroidY:%f\n", centroidY);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].centroid.y = centroidY;

        int16_t boundX = rawBlock.left;
        if (boundX < 0 || boundX >= nn::xcd::IrImageWidthMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: boundX:%d\n", boundX);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].bound.x = boundX;

        int16_t boundWidth = rawBlock.right - rawBlock.left + 1;
        if (boundWidth < 1 || boundWidth > nn::xcd::IrImageWidthMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: boundWidth:%d\n", boundWidth);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].bound.width = boundWidth;

        int16_t boundY = rawBlock.up;
        if (boundY < 0 || boundY >= nn::xcd::IrImageHeightMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: boundY:%d\n", boundY);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].bound.y = boundY;

        int16_t boundHeight = rawBlock.down - rawBlock.up + 1;
        if (boundHeight < 1 || boundHeight > nn::xcd::IrImageHeightMax)
        {
            NN_XCD_IRSENSOR_LOG("ClusteringInvalidData: boundHeight:%d\n", boundHeight);
            return nn::xcd::ResultIrInvalidPacketData();
        }
        m_pClusteringProcessorStates[index]->objects[i].bound.height = boundHeight;
    }

    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
