﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_AttachmentDevice.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/xcd/xcd_Pairing.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/xcd_ResultForPrivate.h>
#include <nn/xcd/detail/xcd_Log.h>

#include "xcd_CommandHandler.h"
#include "xcd_CommandTypes.h"
#include "xcd_ParserUtil.h"
#include "xcd_ReportTypes.h"
#include "xcd_SerialFlashMap.h"

#if 0
#define CMD_DUMP_LOG(...) NN_SDK_LOG(__VA_ARGS__)
#else
#define CMD_DUMP_LOG(...)
#endif

namespace nn { namespace xcd {

namespace {

// ユーザーキャリブレーションの有効/無効判定を行うための、MagicNumber
const uint16_t UserCalMagicNumber = 0xA1B2;

// シリアルフラッシュの読み出し回数を減らすための、読み出しアドレスとサイズの定義
const uint32_t SerialFlashBlockAddress_IdentificationCode          = SerialFlashAddress_IdentificationCode;    //!< シリアルナンバー
const uint32_t SerialFlashBlockSize_IdentificationCode             = 0x10;                                     //!< シリアルナンバー
const uint32_t SerialFlashBlockAddress_Cal1                        = SerialFlashAddress_SensorCal;             //!< CAL1 領域(センサー)
const uint32_t SerialFlashBlockSize_Cal1                           = 0x18;                                     //!< CAL1 領域(センサー)
const uint32_t SerialFlashBlockAddress_Cal2                        = SerialFlashAddress_StickLeftCal;          //!< CAL2 領域(スティックの CAL 値, 色情報)
const uint32_t SerialFlashBlockSize_Cal2                           = 0x19;                                     //!< CAL2 領域(スティックの CAL 値, 色情報)
const uint32_t SerialFlashBlockAddress_Design                      = SerialFlashAddress_ColorMain;             //!< デザイン情報領域
const uint32_t SerialFlashBlockSize_Design                         = 0x0d;                                     //!< デザイン情報領域
const uint32_t SerialFlashBlockAddress_Model1                      = SerialFlashAddress_SensorHorizontal;      //!< MODEL1 領域(センサー, 左スティックのモデル値)
const uint32_t SerialFlashBlockSize_Model1                         = 0x18;                                     //!< MODEL1 領域(センサー, 左スティックのモデル値)
const uint32_t SerialFlashBlockAddress_Model2                      = SerialFlashAddress_ModelStickRight;       //!< MODEL2 領域(右スティックのモデル値)
const uint32_t SerialFlashBlockSize_Model2                         = 0x12;                                     //!< MODEL2 領域(右スティックのモデル値)
const uint32_t SerialFlashBlockAddress_UserCal1                    = SerialFlashAddress_UserStickLeftMagicNum; //!< USERCAL1 領域(スティックのCAL値 + センサーのマジックナンバー)
const uint32_t SerialFlashBlockSize_UserCal1                       = 0x18;                                     //!< USERCAL1 領域(スティックのCAL値 + センサーのマジックナンバー)
const uint32_t SerialFlashBlockAddress_UserCal2                    = SerialFlashAddress_UserSensorCal;         //!< USERCAL2 領域(センサーの CAL 値)
const uint32_t SerialFlashBlockSize_UserCal2                       = 0x18;                                     //!< USERCAL2 領域(センサーの CAL 値)

void ParseColor(::nn::util::Color4u8Type* pOutValue, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    pOutValue->v[0] = buffer[0];
    pOutValue->v[1] = buffer[1];
    pOutValue->v[2] = buffer[2];
    pOutValue->v[3] = 0xff;
}

void GenerateColor(uint8_t* pOutBuffer, const ::nn::util::Color4u8Type& color)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);

    pOutBuffer[0] = color.v[0];
    pOutBuffer[1] = color.v[1];
    pOutBuffer[2] = color.v[2];
}

bool IsUserCalAvailable(const uint8_t* buffer)
{
    return (buffer[0] == (UserCalMagicNumber & 0xff))
        && (buffer[1] == ((UserCalMagicNumber >> 8) & 0xff));
}

void SetMagicNumber(uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    buffer[0] = UserCalMagicNumber & 0xff;
    buffer[1] = (UserCalMagicNumber >> 8) & 0xff;
}

void ParseSensorCal(SensorCalibrationValue* pOutValue, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ParseSensorData(&pOutValue->accelerometerOrigin,      &buffer[SerialFlashOffset_Acc0g]);
    ParseSensorData(&pOutValue->accelerometerSensitivity, &buffer[SerialFlashOffset_Acc1g]);
    ParseSensorData(&pOutValue->gyroscopeOrigin,          &buffer[SerialFlashOffset_Gyro0rpm]);
    ParseSensorData(&pOutValue->gyroscopeSensitivity,     &buffer[SerialFlashOffset_Gyro78rpm]);
}

void GenerateSensorCal(uint8_t* buffer, const SensorCalibrationValue& value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    GenerateFromSensorState(&buffer[SerialFlashOffset_Acc0g], value.accelerometerOrigin);
    GenerateFromSensorState(&buffer[SerialFlashOffset_Acc1g], value.accelerometerSensitivity);
    GenerateFromSensorState(&buffer[SerialFlashOffset_Gyro0rpm], value.gyroscopeOrigin);
    GenerateFromSensorState(&buffer[SerialFlashOffset_Gyro78rpm], value.gyroscopeSensitivity);
}

void ParseStickLeftCal(AnalogStickValidRange* pOutValue, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ParseAnalogStick(&pOutValue->origin,     &buffer[SerialFlashOffset_StickLeftZero]);
    ParseAnalogStick(&pOutValue->circuitMin, &buffer[SerialFlashOffset_StickLeftMin]);
    ParseAnalogStick(&pOutValue->circuitMax, &buffer[SerialFlashOffset_StickLeftMax]);
}

void GenerateStickLeftCal(uint8_t* buffer, const AnalogStickValidRange& value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickLeftZero],value.origin);
    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickLeftMin], value.circuitMin);
    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickLeftMax], value.circuitMax);
}

void ParseStickRightCal(AnalogStickValidRange* pOutValue, const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ParseAnalogStick(&pOutValue->origin,     &buffer[SerialFlashOffset_StickRightZero]);
    ParseAnalogStick(&pOutValue->circuitMin, &buffer[SerialFlashOffset_StickRightMin]);
    ParseAnalogStick(&pOutValue->circuitMax, &buffer[SerialFlashOffset_StickRightMax]);
}

void GenerateStickRightCal(uint8_t* buffer, const AnalogStickValidRange& value)
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickRightZero],value.origin);
    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickRightMin], value.circuitMin);
    GenerateFromAnalogStick(&buffer[SerialFlashOffset_StickRightMax], value.circuitMax);
}

void ParseStickModelValue(AnalogStickDeviceParameter* pOutParam,
                          uint16_t* pOutOriginPlay,
                          uint16_t* pOutCiricuitValidRatio,
                          bool* pOutIsXScalingRequired,
                          const uint8_t* buffer)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParam);
    NN_SDK_REQUIRES_NOT_NULL(pOutCiricuitValidRatio);
    NN_SDK_REQUIRES_NOT_NULL(pOutOriginPlay);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    AnalogStickState temp;
    ParseAnalogStick(&temp, &buffer[SerialFlashOffset_StickNoiseAndStrokeTyp]);
    // 0 であれば Scalingが有効
    *pOutIsXScalingRequired = ((temp.x & SerialFlashBit_StickXScalingRequired) == 0x00);
    ParseAnalogStick(&temp, &buffer[SerialFlashOffset_StickPlay]);
    *pOutOriginPlay = temp.x;
    *pOutCiricuitValidRatio = temp.y;
    ParseAnalogStick(&pOutParam->minimumStrokePositive, &buffer[SerialFlashOffset_StickMinimumStrokePos]);
    ParseAnalogStick(&pOutParam->minimumStrokeNegative, &buffer[SerialFlashOffset_StickMinimumStrokeNeg]);
    ParseAnalogStick(&pOutParam->originRangeMax, &buffer[SerialFlashOffset_StickZeroRangeMax]);
    ParseAnalogStick(&pOutParam->originRangeMin, &buffer[SerialFlashOffset_StickZeroRangeMin]);
}

}

CommandHandler::CommandHandler() NN_NOEXCEPT
    : m_CommandCount(0)
    , m_CommandIndex(0)
    , m_Busy(0)
    , m_IsOnSerialFlashWrite(false)
    , m_IsOnSerialFlashManualWrite(false)
    , m_IndicatorPattern(0)
    , m_Activated(false)
    , m_BluetoothFirmwareVersion({0, 0})
{
    // 何もしない
}

CommandHandler::~CommandHandler() NN_NOEXCEPT
{
    // 何もしない
}

void CommandHandler::Activate() NN_NOEXCEPT
{
    // コマンドキューを初期化
    ClearCommandToQueue();
    m_Busy = 0;

    m_IsOnSerialFlashWrite = false;
    m_IsOnSerialFlashManualWrite = false;

    m_IndicatorPattern = 0;

    m_Activated = true;
}

void CommandHandler::Deactivate() NN_NOEXCEPT
{
    m_Activated = false;
}

size_t CommandHandler::GetOutputReport(uint8_t* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(size > CommandOutSize_Payload);

    // キューにコマンドが1つもない
    if(m_CommandCount == 0)
    {
        return 0;
    }

    // コマンド送信中は別のコマンドを送らない
    // ただしBusyが10 sniff cycle以上続くようであればパケロスと判断し、再送する(FWのWorkaround)
    if(m_Busy > 0 && m_Busy < 10)
    {
        m_Busy++;
        return 0;
    }

    // 未送信のコマンドを探索する
    for(int i = m_CommandIndex + MaxCommandCount - m_CommandCount + 1;
        i <= m_CommandIndex + MaxCommandCount; i++)
    {
        auto& command = m_CommandQueue[i % MaxCommandCount];

        // 非Busy状態のときはキューされているコマンドを送信
        // Busy状態が続いているときは送信済みコマンドを再送する (FWのWorkaround)
        if((m_Busy == 0 && command.status == CommandStatus_Queued) ||
           (m_Busy >= 10 && command.status == CommandStatus_Sent) )
        {
            // 未送信のコマンドがみつかった
            command.status = CommandStatus_Sent;
            pOutValue[CommandByte_CommandId] = command.commandId;
            std::memcpy(&pOutValue[CommandOutByte_Payload], command.payload, CommandOutSize_Payload);
            m_Busy = 1;
            m_SentCommandIndex = i % MaxCommandCount;
            return Report_CommandOut::Size;
        }
    }

    // 送信するコマンドがみつからなかった
    return 0;
}

Result CommandHandler::PushCommandToQueue(CommandDescryptor descrypter, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_DO(PushCommandToQueueSerialFlashRead(descrypter, payload, pListener, false, true, false));
    NN_RESULT_SUCCESS;
}

Result CommandHandler::PushCommandToQueueWithoutDuplication(CommandDescryptor descrypter, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_RESULT_DO(PushCommandToQueueSerialFlashRead(descrypter, payload, pListener, false, false, false));
    NN_RESULT_SUCCESS;
}

Result CommandHandler::PushCommandToQueueSerialFlashRead(CommandDescryptor descrypter, uint8_t* payload, ICommandListener* pListener, bool isVerify, bool isDuplicatable, bool isRawAccess) NN_NOEXCEPT
{
    int nextCommandIndex = (m_CommandIndex == MaxCommandCount - 1) ? 0 : (m_CommandIndex + 1);
    auto& command = m_CommandQueue[nextCommandIndex];

    NN_RESULT_THROW_UNLESS(m_CommandCount < MaxCommandCount, ResultCommandQueueFull());
    NN_RESULT_THROW_UNLESS(command.status == CommandStatus_Empty, ResultCommandQueueFull());

    command.status = CommandStatus_Queued;
    command.commandId = descrypter.Id;

    std::memset(&command.payload, 0x00, CommandOutSize_Payload);
    if (descrypter.Size > 0 && payload != nullptr)
    {
        memcpy(command.payload, payload, descrypter.Size);
    }
    command.pListener = pListener;

    command.isSerialFlashVerify = isVerify;
    command.isDuplicatable = isDuplicatable;
    command.isRawSerialFlashRead = isRawAccess;

    m_CommandIndex = nextCommandIndex;
    m_CommandCount++;

    NN_RESULT_SUCCESS;
}

void CommandHandler::ClearCommandToQueue() NN_NOEXCEPT
{
    for (auto& command : m_CommandQueue)
    {
        command.status = CommandStatus_Empty;
        command.isDuplicatable = true;
    }
    m_CommandCount = 0;
    m_CommandIndex = 0;
}

void CommandHandler::RemoveCommandFromTail(int commandCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(commandCount, 0, m_CommandCount);
    for (auto i = 0; i < commandCount; i++)
    {
        m_CommandQueue[m_CommandIndex].status = CommandStatus_Empty;
        m_CommandQueue[m_CommandIndex].isDuplicatable = true;
        m_CommandIndex = (m_CommandIndex + MaxCommandCount - 1) % MaxCommandCount;
        m_CommandCount--;
    }
}

void CommandHandler::ReadSerialFlashImpl(uint32_t address, uint8_t size, ICommandListener* pListener, bool isVerify, bool isRawAccess) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_RANGE(size, 1, SerialFlashReadSize_Data + 1);

    uint8_t buffer[5];
    Uint32ToLittleEndian(&buffer[0], address);
    buffer[4] = size;

    PushCommandToQueueSerialFlashRead(Command_SerialFlashRead, buffer, pListener, isVerify, true, isRawAccess);
}

Result CommandHandler::PushSerialFlashWriteCommandToQueue(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    m_IsOnSerialFlashWrite = true;
    PushCommandToQueue(Command_SerialFlashWrite, m_BufferForSerialFlashWrite, pListener);

    NN_RESULT_SUCCESS;
}

NN_STATIC_ASSERT(PairingSize_DeviceName >= sizeof(BdName));
void CommandHandler::Pairing(const BluetoothDeviceInfo& deviceInfo, ICommandListener* pListener, bool pairingInfoExist) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    CMD_DUMP_LOG("[xcd] Start Pairing\n");

    uint8_t buffer[PairingOutSize_Data];

    if (pairingInfoExist == true)
    {
        m_IsPairingWithRegisteredDevice = true;
        buffer[PairingByteOffset_Status] = PairingStatus_InitiateWithPairedDevice;
    }
    else
    {
        m_IsPairingWithRegisteredDevice = false;
        buffer[PairingByteOffset_Status] = PairingStatus_Initiate;
    }

    CMD_DUMP_LOG("    -> Master BdAddr ");
    for (int i = 0; i < PairingSize_BdAddr; i++)
    {
        buffer[PairingByteOffset_BdAddr + PairingSize_BdAddr - i - 1] = deviceInfo.address.address[i];
        CMD_DUMP_LOG(":%02x", buffer[PairingByteOffset_BdAddr + PairingSize_BdAddr - i - 1]);
    }
    CMD_DUMP_LOG("\n");
    CMD_DUMP_LOG("    -> Master BdName ");

    for (int i = 0; i < PairingSize_DeviceName; i++)
    {
        buffer[PairingByteOffset_DeviceName + i] = deviceInfo.bdName.name[i];
        CMD_DUMP_LOG("%c", buffer[PairingByteOffset_DeviceName + i]);
    }
    CMD_DUMP_LOG("\n");
    memcpy(&buffer[PairingByteOffset_ClassOfDevice], &deviceInfo.classOfDevice, PairingSize_ClassOfDevice);
    buffer[PairingByteOffset_FeatureSet] = deviceInfo.featureSet;

    PushCommandToQueue(Command_PairingOut, buffer, pListener);
}

void CommandHandler::GetDeviceInfo(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    PushCommandToQueue(Command_GetDeviceInfo, nullptr, pListener);
}

void CommandHandler::Reset(bool page) NN_NOEXCEPT
{
    uint8_t buffer;
    buffer = static_cast<uint8_t>((page == true) ? ResetPageType_Enabled : ResetPageType_Disabled);

    PushCommandToQueue(Command_Reset, &buffer, nullptr);
}

void CommandHandler::SetDataFormat(PeriodicDataFormat format, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer;

    switch(format)
    {
    case PeriodicDataFormat_Basic:
        buffer = Report_BasicIn::Id;
        break;
    case PeriodicDataFormat_MCU:
        buffer = Report_McuIn::Id;
        break;
    case PeriodicDataFormat_Attachment:
        buffer = Report_AttachmentIn::Id;
        break;
    case PeriodicDataFormat_Audio:
        buffer = Report_AudioIn::Id;
        break;
    case PeriodicDataFormat_Generic:
        buffer = Report_GenericIn::Id;
        break;
    case PeriodicDataFormat_McuUpdate:
        buffer = Report_McuUpdateIn::Id;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (!IsSameAsTailCommand(Command_SetDataFormat, &buffer, pListener))
    {
        PushCommandToQueue(Command_SetDataFormat, &buffer, pListener);
    }
}

bool CommandHandler::IsSameAsTailCommand(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(command.Size <= CommandOutSize_Payload);

    auto& tailCommand = m_CommandQueue[m_CommandIndex];

    return (tailCommand.status != CommandStatus_Empty)
        && (tailCommand.commandId == command.Id)
        && (tailCommand.pListener == pListener)
        && (::std::memcmp(tailCommand.payload, payload, command.Size) == 0);
}

int CommandHandler::FindSameCommandFromTail(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener, uint8_t compareSizeMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(command.Size <= CommandOutSize_Payload);

    // 対象コマンドと同じものを指定した深さまで探す
    auto commandDepth = -1;
    for (auto i = 0; i < compareSizeMax; i++)
    {
        auto index = (m_CommandIndex + MaxCommandCount - i) % MaxCommandCount;
        if (i >= m_CommandCount) break;
        auto& queuedCommand = m_CommandQueue[index];

        if ((queuedCommand.status != CommandStatus_Empty)
            && (queuedCommand.isDuplicatable == false)
            && (queuedCommand.commandId == command.Id)
            && (queuedCommand.pListener == pListener)
            && (::std::memcmp(queuedCommand.payload, payload, command.Size) == 0))
        {
            commandDepth = i;
            break;
        }
    }
    NN_SDK_ASSERT_LESS_EQUAL(commandDepth, compareSizeMax);
    return commandDepth;
}

int CommandHandler::FindSameSequenceFromTail(CommandDescryptor command, uint8_t* payload, ICommandListener* pListener, uint8_t compareSizeMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(command.Size <= CommandOutSize_Payload);

    // 比較対象のコマンドが存在するかチェックする
    auto sequenceSize = FindSameCommandFromTail(command, payload, pListener, compareSizeMax);
    if (sequenceSize <= 0) return sequenceSize;

    // 同じコマンド以前のシーケンスと、以後のシーケンスが同じかどうかチェックし、一致していればそのサイズを返す
    for (auto i = 0; i < sequenceSize ; i++)
    {
        auto olderIndex = (m_CommandIndex + MaxCommandCount - (sequenceSize + 1) - i) % MaxCommandCount;
        auto newerIndex = (m_CommandIndex + MaxCommandCount - i) % MaxCommandCount;

        if (m_CommandCount < i + sequenceSize + 1)
        {
            sequenceSize = -1;
            break;
        }

        auto& olderCommand = m_CommandQueue[olderIndex];
        auto& newerCommand = m_CommandQueue[newerIndex];

        if ((olderCommand.status == CommandStatus_Queued)
            && (olderCommand.commandId == newerCommand.commandId)
            && (olderCommand.pListener == newerCommand.pListener)
            && (olderCommand.isDuplicatable == false)
            && (newerCommand.isDuplicatable == false)
            && (::std::memcmp(olderCommand.payload, newerCommand.payload, sizeof(olderCommand.payload)) == 0))
        {
            continue;
        }
        else
        {
            sequenceSize = -1;
            break;
        }
    }
    NN_SDK_ASSERT_RANGE(sequenceSize, -1, compareSizeMax);
    return sequenceSize;
}

void CommandHandler::SensorSleep(bool sleep, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(sleep ? SensorSleepValueType_Sleep : SensorSleepValueType_Active);

    // TODO FIXME Maybe this should be somewhere else
    FirmwareVersionUnit const dscaleMinVersion = { 0x3, 0x8F };
    if (m_BluetoothFirmwareVersion >= dscaleMinVersion)
    {
        buffer = static_cast<uint8_t>(sleep ? SensorSleepValueType_Sleep : SensorSleepValueType_Dscale1);
    }

    PushCommandToQueue(Command_SensorSleep, &buffer, pListener);
}

void CommandHandler::SensorConfig(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyroscopeFsr, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer[2] = {SensorConfigGyroscopeType_2000dps, SensorConfigAccelerometerType_8G};
    switch(gyroscopeFsr)
    {
    case GyroscopeFsr_250dps:
        buffer[0] = SensorConfigGyroscopeType_250dps;
        break;

    case GyroscopeFsr_500dps:
        buffer[0] = SensorConfigGyroscopeType_500dps;
        break;

    case GyroscopeFsr_1000dps:
        buffer[0] = SensorConfigGyroscopeType_1000dps;
        break;

    case GyroscopeFsr_2000dps:
        buffer[0] = SensorConfigGyroscopeType_2000dps;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    switch(accelerometerFsr)
    {
    case AccelerometerFsr_2G:
        buffer[1] = SensorConfigAccelerometerType_2G;
        break;

    case AccelerometerFsr_4G:
        buffer[1] = SensorConfigAccelerometerType_4G;
        break;

    case AccelerometerFsr_8G:
        buffer[1] = SensorConfigAccelerometerType_8G;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    PushCommandToQueue(Command_SensorConfig, buffer, pListener);
}

void CommandHandler::SensorPayload(bool enable, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(SensorSleepValueType_Active);

    FirmwareVersionUnit const dscaleMinVersion = { 0x3, 0x8F };
    if (m_BluetoothFirmwareVersion >= dscaleMinVersion)
    {
        buffer = static_cast<uint8_t>(enable ? SensorSleepValueType_Dscale1Payload6 : SensorSleepValueType_Dscale1);
    }

    PushCommandToQueue(Command_SensorSleep, &buffer, pListener);
}

void CommandHandler::McuReset(ICommandListener* pListener) NN_NOEXCEPT
{
    PushCommandToQueue(Command_McuReset, nullptr, pListener);
}

void CommandHandler::McuWrite(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_LESS_EQUAL(size, McuWriteSize_Data);

    uint8_t buffer[McuWriteSize_Data] = {0};

    std::memcpy(buffer, pBuffer, size);

    PushCommandToQueue(Command_McuWrite, buffer, pListener);
}

void CommandHandler::McuWriteWithoutDuplication(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_LESS_EQUAL(size, McuWriteSize_Data);

    uint8_t buffer[McuWriteSize_Data] = {0};

    std::memcpy(buffer, pBuffer, size);

    auto sameSequenceSize = FindSameSequenceFromTail(Command_McuWrite, buffer, pListener, MaxCompareSequenceCount);
    if (sameSequenceSize < 0)
    {
        PushCommandToQueueWithoutDuplication(Command_McuWrite, buffer, pListener);
    }
    else
    {
        RemoveCommandFromTail(sameSequenceSize);
    }
}

void CommandHandler::McuResume(McuResumeValueType resumeType, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(resumeType, McuResumeValueType_Suspend, McuResumeValueType_ResumeForUpdate);

    uint8_t buffer = static_cast<uint8_t>(resumeType);

    PushCommandToQueue(Command_McuResume, &buffer, pListener);
}

void CommandHandler::McuPollingEnable(uint8_t* pBuffer, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_LESS_EQUAL(size, McuWriteSize_Data);

    uint8_t buffer[McuWriteSize_Data] = {0};

    std::memcpy(buffer, pBuffer, size);

    auto sameSequenceSize = FindSameSequenceFromTail(Command_McuPollingEnable, buffer, pListener, MaxCompareSequenceCount);
    if (sameSequenceSize < 0)
    {
        PushCommandToQueueWithoutDuplication(Command_McuPollingEnable, buffer, pListener);
    }
    else
    {
        RemoveCommandFromTail(sameSequenceSize);
    }
}

void CommandHandler::McuPollingDisable(ICommandListener* pListener) NN_NOEXCEPT
{
    auto sameSequenceSize = FindSameSequenceFromTail(Command_McuPollingDisable, nullptr, pListener, MaxCompareSequenceCount);
    if (sameSequenceSize < 0)
    {
        PushCommandToQueueWithoutDuplication(Command_McuPollingDisable, nullptr, pListener);
    }
    else
    {
        RemoveCommandFromTail(sameSequenceSize);
    }
}

void CommandHandler::AttachmentEnable(bool enabled, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(enabled ? AttachmentEnableType_Enable : AttachmentEnableType_Disable);

    PushCommandToQueue(Command_AttachmentEnable, &buffer, pListener);
}

void CommandHandler::GetExtDevInfo(ICommandListener* pListener) NN_NOEXCEPT
{
    PushCommandToQueue(Command_GetExtDevInfo, nullptr, pListener);
}

void CommandHandler::SendCommandToAttachmentDevice(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size <= nn::xcd::AttachmentSendCommandSizeMax);
    uint8_t command[nn::xcd::AttachmentSendCommandSizeMax + 1] = {};

    // 送信するコマンドの頭にサイズを追加する
    command[0] = static_cast<uint8_t>(size);
    std::memcpy(&command[1], pInData, size);

    PushCommandToQueue(Command_ExtDevWrite, command, pListener);
}

void CommandHandler::ConfigBasicInFormatForAttachment(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    const size_t FormatConfigMaxSize = 38;
    NN_SDK_REQUIRES_NOT_NULL(pInData);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_LESS_EQUAL(size, FormatConfigMaxSize);

    uint8_t command[FormatConfigMaxSize];
    memcpy(command, pInData, size);

    PushCommandToQueue(Command_ExtDevInFormatConfig, command, pListener);
}

void CommandHandler::EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInData, size_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInData);
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_LESS_EQUAL(size, nn::xcd::AttachmentSendCommandSizeMax);

    uint8_t command[nn::xcd::AttachmentSendCommandSizeMax] = { 0 };

    // 送信するコマンドの頭にサイズを追加する
    command[0] = static_cast<uint8_t>(size);
    std::memcpy(&command[1], pInData, size);

    PushCommandToQueue(Command_ExtDevPollingEnable, command, pListener);
}

void CommandHandler::DisablePollingReceiveModeForAttachmentDevice(ICommandListener* pListener) NN_NOEXCEPT
{
    PushCommandToQueue(Command_ExtDevPollingDisable, nullptr, pListener);
}

void CommandHandler::SetIndicatorLed(uint8_t pattern, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(pattern, 0x00, 0xf1);

    if (!IsSameAsTailCommand(Command_SetIndicatorLed, &pattern, pListener) && m_IndicatorPattern != pattern)
    {
        if (PushCommandToQueue(Command_SetIndicatorLed, &pattern, pListener).IsSuccess())
        {
            m_IndicatorPattern = pattern;
        }
    }
}

void CommandHandler::MotorEnable(bool enabled, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(enabled ? MotorEnableValueType_Enable : MotorEnableValueType_Disable);

    PushCommandToQueue(Command_MotorEnable, &buffer, pListener);
}

void CommandHandler::WriteChargeSetting(bool chargeEnabled, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(chargeEnabled ? ChargeEnableType_Enable : ChargeEnableType_Disable);

    PushCommandToQueue(Command_WriteChargeSetting, &buffer, pListener);
}

void CommandHandler::SetShipment(bool enabled, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t buffer = static_cast<uint8_t>(enabled ? ShipmentEnableValueType_Enable : ShipmentEnableValueType_Disable);

    PushCommandToQueue(Command_Shipment, &buffer, pListener);
}

void CommandHandler::GetButtonTriggerElapsedTime(ICommandListener* pListener) NN_NOEXCEPT
{
    PushCommandToQueue(Command_LRButtonDetection, nullptr, pListener);
}

void CommandHandler::ReadIdentificationCode(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_IdentificationCode,
                    SerialFlashBlockSize_IdentificationCode,
                    pListener, false, false);
}

Result CommandHandler::UpdateSixAxisSensorUserCal(const SensorCalibrationValue& value, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserSensorMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserSensorMagicNum + SerialFlashSize_SensorCal;

    SetMagicNumber(&m_BufferForSerialFlashWrite[5]);

    GenerateSensorCal(&m_BufferForSerialFlashWrite[5 + (SerialFlashAddress_UserSensorCal - SerialFlashAddress_UserSensorMagicNum)],
                      value);

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::DeleteSixAxisSensorUserCal(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0xff, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserSensorMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserSensorMagicNum + SerialFlashSize_SensorCal;

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::UpdateLeftAnalogStickUserCal(const AnalogStickValidRange& value, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserStickLeftMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserStickLeftMagicNum + SerialFlashSize_UserStickLeftCal;

    SetMagicNumber(&m_BufferForSerialFlashWrite[5]);

    GenerateStickLeftCal(&m_BufferForSerialFlashWrite[5 + (SerialFlashAddress_UserStickLeftCal - SerialFlashAddress_UserStickLeftMagicNum)],
                         value);

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::DeleteLeftAnalogStickUserCal(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0xff, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserStickLeftMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserStickLeftMagicNum + SerialFlashSize_UserStickLeftCal;

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;

}

Result CommandHandler::UpdateRightAnalogStickUserCal(const AnalogStickValidRange& value, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserStickRightMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserStickRightMagicNum + SerialFlashSize_UserStickRightCal;

    SetMagicNumber(&m_BufferForSerialFlashWrite[5]);

    GenerateStickRightCal(&m_BufferForSerialFlashWrite[5 + (SerialFlashAddress_UserStickRightCal - SerialFlashAddress_UserStickRightMagicNum)],
                         value);

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::DeleteRightAnalogStickUserCal(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0xff, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_UserStickRightMagicNum);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_UserStickRightMagicNum + SerialFlashSize_UserStickRightCal;

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}


void CommandHandler::ReadCal1(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_Cal1,
                    SerialFlashBlockSize_Cal1,
                    pListener, false, false);
}

void CommandHandler::ReadCal2(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_Cal2,
                    SerialFlashBlockSize_Cal2,
                    pListener, false, false);
}

void CommandHandler::ReadDesign(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_Design,
                    SerialFlashBlockSize_Design,
                    pListener, false, false);
}

void CommandHandler::ReadModel1(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_Model1,
                    SerialFlashBlockSize_Model1,
                    pListener, false, false);
}

void CommandHandler::ReadModel2(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_Model2,
                    SerialFlashBlockSize_Model2,
                    pListener, false, false);
}

void CommandHandler::ReadUserCal1(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_UserCal1,
                    SerialFlashBlockSize_UserCal1,
                    pListener, false, false);
}

void CommandHandler::ReadUserCal2(ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(SerialFlashBlockAddress_UserCal2,
                    SerialFlashBlockSize_UserCal2,
                    pListener, false, false);
}

Result CommandHandler::UpdateControllerColor(const nn::util::Color4u8Type& mainColor, const nn::util::Color4u8Type& subColor, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_ColorMain);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_ColorMain + SerialFlashSize_ColorSub;

    GenerateColor(&m_BufferForSerialFlashWrite[5],
                  mainColor);

    GenerateColor(&m_BufferForSerialFlashWrite[5 + (SerialFlashAddress_ColorSub - SerialFlashAddress_ColorMain)],
                  subColor);

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::UpdateDesignInfo(const DeviceColor& color, uint8_t variation, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_ColorMain);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_ColorMain + SerialFlashSize_ColorSub + SerialFlashSize_ColorThird + SerialFlashSize_ColorForth + SerialFlashSize_DesignVariation;

    auto index = 5;
    GenerateColor(&m_BufferForSerialFlashWrite[index],
                  color.color[0]);
    index += SerialFlashSize_ColorMain;
    GenerateColor(&m_BufferForSerialFlashWrite[index],
                  color.color[1]);
    index += SerialFlashSize_ColorSub;
    GenerateColor(&m_BufferForSerialFlashWrite[index],
                  color.color[2]);
    index += SerialFlashSize_ColorThird;
    GenerateColor(&m_BufferForSerialFlashWrite[index],
                  color.color[3]);
    index += SerialFlashSize_ColorForth;
    m_BufferForSerialFlashWrite[index] = variation;

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

Result CommandHandler::UpdateFormatVersion(uint8_t version, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], SerialFlashAddress_FormatVersion);
    m_BufferForSerialFlashWrite[4] = SerialFlashSize_FormatVersion;

    m_BufferForSerialFlashWrite[5] = version;

    NN_RESULT_DO(PushSerialFlashWriteCommandToQueue(pListener));

    NN_RESULT_SUCCESS;
}

int CommandHandler::GetSerialFlashReadUnitSize() NN_NOEXCEPT
{
    return SerialFlashReadSize_Data;
}

int CommandHandler::GetSerialFlashWriteUnitSize() NN_NOEXCEPT
{
    return SerialFlashWriteSize_Data;
}

void CommandHandler::ReadSerialFlash(const uint32_t address, uint8_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    ReadSerialFlashImpl(address,
                    size,
                    pListener, false, true);
}

Result CommandHandler::WriteSerialFlash(const uint32_t address, const uint8_t* pBuffer, uint8_t size, ICommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);
    NN_SDK_REQUIRES_MINMAX(size, 1, SerialFlashWriteSize_Data);

    if (m_IsOnSerialFlashWrite)
    {
        NN_RESULT_THROW(ResultSerialFlashOnWrite());
    }

    std::memset(m_BufferForSerialFlashWrite, 0x0, sizeof(m_BufferForSerialFlashWrite));

    Uint32ToLittleEndian(&m_BufferForSerialFlashWrite[0], address);
    m_BufferForSerialFlashWrite[4] = size;
    std::memcpy(&m_BufferForSerialFlashWrite[5], pBuffer, size);
    m_IsOnSerialFlashManualWrite = true;
    auto result = PushSerialFlashWriteCommandToQueue(pListener);
    if (result.IsFailure())
    {
        m_IsOnSerialFlashManualWrite = false;
        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

void CommandHandler::ParseInputReport(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);

    if(m_Busy > 0)
    {
        auto& iCommand = m_CommandQueue[m_SentCommandIndex];
        if(iCommand.commandId == pBuffer[CommandInByte_OutputId] && iCommand.status == CommandStatus_Sent)
        {
            bool isCommandFound = false;

            // Ack
            if (pBuffer[CommandByte_CommandId] == Command_Ack.Id)
            {
                HandleAck(pBuffer, iCommand.pListener, iCommand.payload);
                isCommandFound = true;
            }
            // ペアリング
            else if (pBuffer[CommandByte_CommandId] == Command_PairingIn.Id)
            {
                HandlePairing(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }
            // DeviceInfo
            else if (pBuffer[CommandByte_CommandId] == Command_DeviceInfo.Id)
            {
                HandleDeviceInfo(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }
            // L/R Button Elapsed Time
            else if (pBuffer[CommandByte_CommandId] == Command_LRButtonElapsedTime.Id)
            {
                HandleGetButtonTriggerElapsedTime(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }
            // Mcu Read
            else if (pBuffer[CommandByte_CommandId] == Command_McuData.Id)
            {
                HandleMcuRead(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }
            // SerialFlashData
            else if (pBuffer[CommandByte_CommandId] == Command_SerialFlashData.Id)
            {
                isCommandFound = HandleSerialFlashData(pBuffer, iCommand.pListener, iCommand.isSerialFlashVerify, iCommand.isRawSerialFlashRead, iCommand.payload);
            }
            // ExtDevInfo
            else if (pBuffer[CommandByte_CommandId] == Command_ExtDevInfo.Id)
            {
                HandleExtDevInfo(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }
            // ExtDevRead
            else if (pBuffer[CommandByte_CommandId] == Command_ExtDevRead.Id)
            {
                HandleExtDevRead(pBuffer, iCommand.pListener);
                isCommandFound = true;
            }

            if (isCommandFound == true)
            {
                // コマンドをキューから削除
                iCommand.status = CommandStatus_Empty;
                m_CommandCount--;
                m_Busy = 0;
            }
            else
            {
                // 対応するコマンドではない。主にSerialFlashRead の再送
            }
        }
        else
        {
            // 送信中のコマンドと cmd_id が一致しない
        }
    }
    else
    {
        // 送信中のコマンドが存在しない
    }
}

void CommandHandler::HandleAck(const uint8_t* pBuffer, ICommandListener* pListener, uint8_t* pOutputReportBuffer) NN_NOEXCEPT
{
    // pListener は nullptr の可能性もある

    auto result = nn::ResultSuccess();

    // SerialFlashWrite の時は、Verify をする
    if (pBuffer[CommandInByte_OutputId] == Command_SerialFlashWrite.Id)
    {
        ReadSerialFlashImpl(LittleEndianToUint32(&m_BufferForSerialFlashWrite[0]),
                        m_BufferForSerialFlashWrite[4],
                        pListener, true, false);
    }
    else if (pBuffer[CommandInByte_OutputId] == Command_SetDataFormat.Id)
    {
        PeriodicDataFormat format = PeriodicDataFormat_Basic;
        if (pOutputReportBuffer[0] == Report_BasicIn::Id)
        {
            format = PeriodicDataFormat_Basic;
        }
        else if (pOutputReportBuffer[0] == Report_McuIn::Id)
        {
            format = PeriodicDataFormat_MCU;
        }
        else if (pOutputReportBuffer[0] == Report_AttachmentIn::Id)
        {
            format = PeriodicDataFormat_Attachment;
        }
        else if (pOutputReportBuffer[0] == Report_AudioIn::Id)
        {
            format = PeriodicDataFormat_Audio;
        }
        else if (pOutputReportBuffer[0] == Report_GenericIn::Id)
        {
            format = PeriodicDataFormat_Generic;
        }
        else if (pOutputReportBuffer[0] == Report_McuUpdateIn::Id)
        {
            format = PeriodicDataFormat_McuUpdate;
        }
        else
        {
            // Data Format が正しく解釈できなかった
            return;
        }
        // Hid Commandの完了を通知
        if (pListener != nullptr)
        {
            pListener->NotifySetDataFormat(result, format);
        }
    }
    else
    {
        switch (pBuffer[AckByte_Result])
        {
        case AckResultType_Success:
            // Success
            result = nn::ResultSuccess();
            break;
        case AckResultType_Fail:
        case AckResultType_NoDevicePaired:
        case AckResultType_NotSupported:
            // Hid Command の Ack については製品ではない
            // 古いFWとの互換性担保のため、Ack の通知を上位には行わず捨てる
            return;
        default:
            ;
        }

        // Hid Commandの完了を通知
        if (pListener != nullptr)
        {
            pListener->NotifyAck(result, pBuffer[CommandInByte_OutputId]);
        }
    }
}

void CommandHandler::HandlePairing(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    CMD_DUMP_LOG("[xcd] Pairing Result\n");
    Result result;
    uint8_t buffer[PairingOutSize_Data];

    const uint8_t* pBufferPairing = &pBuffer[PairingInByte_Payload];
    switch (pBufferPairing[PairingByteOffset_Status])
    {
    case PairingStatus_Initiate:
    {
        // ペアリング情報の解析
        CMD_DUMP_LOG("    -> Bd Addr : ");
        for (int i = 0; i < ::nn::bluetooth::AddressLength; i++)
        {
            m_PairingDeviceInfo.address.address[i] = pBufferPairing[PairingByteOffset_BdAddr + ::nn::bluetooth::AddressLength - i - 1];
            CMD_DUMP_LOG("%02x ", m_PairingDeviceInfo.address.address[i]);
        }
        CMD_DUMP_LOG("\n");

        memcpy(&m_PairingDeviceInfo.bdName, &pBufferPairing[PairingByteOffset_DeviceName], sizeof(m_PairingDeviceInfo.bdName));
        CMD_DUMP_LOG("    -> DeviceName :%s\n", m_PairingDeviceInfo.bdName.name);
        memcpy(&m_PairingDeviceInfo.classOfDevice, &pBufferPairing[PairingByteOffset_ClassOfDevice], sizeof(m_PairingDeviceInfo.classOfDevice));

        // LinkKey シーケンス
        buffer[PairingByteOffset_Status] = PairingStatus_LinkKey;
        PushCommandToQueue(Command_PairingOut, buffer, pListener);

        CMD_DUMP_LOG("    -> Sending Pairing LinkKey\n");
        return;
    }

    case PairingStatus_LinkKey:
    {
        // ペアリング情報の解析
        CMD_DUMP_LOG("    -> Link Key :");
        for (int i = 0; i < LinkkeyLength; i++)
        {
            m_PairingDeviceInfo.linkKey.key[i] = pBufferPairing[PairingByteOffset_LinkKey + i] ^ 0xAA;
            CMD_DUMP_LOG("%02x ", m_PairingDeviceInfo.linkKey.key[i]);
        }
        CMD_DUMP_LOG("\n");

        // 未ペアリングデバイス
        m_IsPairingWithRegisteredDevice = false;

        // 成功の場合は、デバイスより先にペアリング情報を更新する必要があるためここで Hid Commandの完了を通知
        if (pListener != nullptr)
        {
            m_PairingDeviceInfo.hid.vid = 0x57E;
            m_PairingDeviceInfo.hid.pid = 0x2007;
            pListener->NotifyPairingDatabaseUpdate(m_PairingDeviceInfo);
        }

        // Ack の送信
        buffer[PairingByteOffset_Status] = PairingStatus_Ack;
        PushCommandToQueue(Command_PairingOut, buffer, pListener);

        CMD_DUMP_LOG("    -> Sending Pairing Ack\n");
        return;
    }

    case PairingStatus_Ack:
        CMD_DUMP_LOG("    -> Received Ack\n");
        if (m_IsPairingWithRegisteredDevice)
        {
            CMD_DUMP_LOG("    -> Device already pairied\n");
            // 失敗の場合は Ack 受信時に完了を通知
            if (pListener != nullptr)
            {
                pListener->NotifyPairingComplete(nn::xcd::ResultPairingRegisteredDevice());
            }
        }
        else
        {
            CMD_DUMP_LOG("    -> Device paired\n");
            pListener->NotifyPairingComplete(nn::ResultSuccess());
        }
        break;

    default:
        CMD_DUMP_LOG("    -> Pairing Result Err %02x\n", pBufferPairing[PairingByteOffset_Status]);
        // 失敗の場合は Ack 受信時に完了を通知
        if (pListener != nullptr)
        {
            pListener->NotifyPairingComplete(nn::xcd::ResultPairingUnknownError());
        }
        break;
    }

    m_IsOnWiredPairing = false;
    m_PairingDeviceInfo = BluetoothDeviceInfo();
}

void CommandHandler::HandleDeviceInfo(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    DeviceInfo deviceInfo;

    // DeviceType
    switch (pBuffer[DeviceInfoByte_DeviceType])
    {
    case DeviceInfoDeviceType_LeftJoy:
        deviceInfo.deviceType = DeviceType_Left;
        break;

    case DeviceInfoDeviceType_RightJoy:
        deviceInfo.deviceType = DeviceType_Right;
        break;

    case DeviceInfoDeviceType_Fullkey:
        deviceInfo.deviceType = DeviceType_FullKey;
        break;

    case DeviceInfoDeviceType_MiyabiLeft:
        deviceInfo.deviceType = DeviceType_MiyabiLeft;
        break;

    case DeviceInfoDeviceType_MiyabiRight:
        deviceInfo.deviceType = DeviceType_MiyabiRight;
        break;

    case DeviceInfoDeviceType_Tarragon:
        deviceInfo.deviceType = DeviceType_Tarragon;
        break;

    case DeviceInfoDeviceType_Reserved:
    default:
        deviceInfo.deviceType = DeviceType_Unknown;
    }

    // BD_ADDR
    std::memcpy(deviceInfo.address.address, &pBuffer[DeviceInfoByte_BdAddr], sizeof(::nn::bluetooth::Address));

    // Firmware Version
    m_BluetoothFirmwareVersion.major = pBuffer[DeviceInfoByte_FwVersion];
    m_BluetoothFirmwareVersion.minor = pBuffer[DeviceInfoByte_FwVersion + 1];

    // Serial Flash のフォーマットバージョン
    int serialFlashFormatVersion = pBuffer[DeviceInfoByte_FlashFormatVersion];
    if (serialFlashFormatVersion == 0xff)
    {
        serialFlashFormatVersion = 0;
    }

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyDeviceInfo(deviceInfo, m_BluetoothFirmwareVersion, serialFlashFormatVersion);
    }
}

void CommandHandler::HandleGetButtonTriggerElapsedTime(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    ButtonTriggerElapsedTime elapsedTime;

    for (uint16_t i = 0; i < 6; i++)
    {
        uint16_t index = LRButtonElapsedTimeByte_L + i * 2;
        elapsedTime.time[i] = (pBuffer[index] | ((pBuffer[index + 1] << 8) & 0xff00));
    }

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyButtonTriggerElapsedTime(elapsedTime);
    }
}

void CommandHandler::HandleMcuRead(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t mcuData[McuReadSize_Data];
    std::memcpy(mcuData, &pBuffer[McuReadByte_Data], McuReadSize_Data);

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyMcuRead(mcuData, McuReadSize_Data);
    }
}

bool CommandHandler::HandleSerialFlashData(const uint8_t* pBuffer, ICommandListener* pListener, bool isVerify, bool isRawSerialFlashRead, uint8_t* pOutputReportBuffer) NN_NOEXCEPT
{
    uint32_t address = LittleEndianToUint32(&pBuffer[SerialFlashDataByte_Address]);
    uint32_t addressFromOuput = LittleEndianToUint32(&pOutputReportBuffer[0]);
    uint32_t size = m_BufferForSerialFlashWrite[4];

    if (address != addressFromOuput)
    {
        // アドレスが一致しない場合は送信したコマンドではない
        return false;
    }

    const uint8_t* pSerialFlashBuffer = &pBuffer[SerialFlashDataByte_Data];

    if (isVerify == true)
    {
        Result result;
        if (std::memcmp(pBuffer, m_BufferForSerialFlashWrite, m_BufferForSerialFlashWrite[4] + 5))
        {
            result = ResultSuccess();
        }
        else
        {
            result = ResultSerialFlashVerifyError();
        }

        m_IsOnSerialFlashWrite = false;

        if (m_IsOnSerialFlashManualWrite)
        {
            NN_SDK_REQUIRES_NOT_NULL(pListener);
            m_IsOnSerialFlashManualWrite = false;
            pListener->NotifyAck(result, Command_SerialFlashWrite.Id);
        }
        else
        {
            switch (address)
            {
            case SerialFlashAddress_UserSensorMagicNum:
                pListener->NotifyUpdateSixAxisSensorUserCal(result);
                break;

            case SerialFlashAddress_UserStickLeftMagicNum:
                pListener->NotifyUpdateLeftAnalogStickUserCal(result);
                break;

            case SerialFlashAddress_UserStickRightMagicNum:
                pListener->NotifyUpdateRightAnalogStickUserCal(result);
                break;

            case SerialFlashAddress_ColorMain:
                if (size == SerialFlashSize_ColorMain + SerialFlashSize_ColorSub + SerialFlashSize_ColorThird + SerialFlashSize_ColorForth + SerialFlashSize_DesignVariation)
                {
                    pListener->NotifyUpdateDesignInfo(result);
                }
                else if (size == SerialFlashSize_ColorMain + SerialFlashSize_ColorSub)
                {
                    pListener->NotifyUpdateControllerColor(result);
                }
                break;

            case SerialFlashAddress_FormatVersion:
                pListener->NotifyUpdateFormatVersion(result);
                break;

            default:
                ;
                // 指定したアドレスに対する Command がない場合は何もしない
            }
        }
    }
    else
    {
        if (isRawSerialFlashRead)
        {
            pListener->NotifySerialFlashRead(pSerialFlashBuffer, 0);
        }
        else
        {
            switch (address)
            {
            case SerialFlashBlockAddress_IdentificationCode:
                HandleReadIdentificationCode(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_Cal1:
                HandleReadCal1(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_Cal2:
                HandleReadCal2(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_Design:
                HandleReadDesign(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_Model1:
                HandleReadModel1(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_Model2:
                HandleReadModel2(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_UserCal1:
                HandleReadUserCal1(pSerialFlashBuffer, address, pListener);
                break;

            case SerialFlashBlockAddress_UserCal2:
                HandleReadUserCal2(pSerialFlashBuffer, address, pListener);
                break;

            default:
                ;
                // 指定したアドレスに対する Command がない場合は何もしない
            }
        }
    }

    return true;
}

void CommandHandler::HandleReadIdentificationCode(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    IdentificationCode code;
    size_t idLength = 0;
    for (int i = 0; i < IdentificationCodeLength; i++)
    {
        if (pBuffer[SerialFlashAddress_IdentificationCode - baseAddress + i] != 0)
        {
            code._code[idLength] = static_cast<char>(pBuffer[SerialFlashAddress_IdentificationCode - baseAddress + i]);
            idLength++;
        }
    }

    // IdentificationCodeLength と一致している場合は、最終バイトを null終端する
    // 実製品では発生しない
    if (idLength == IdentificationCodeLength)
    {
        idLength--;
    }

    // 終端
    code._code[idLength] = '\0';

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyIdentificationCode(code);
    }
}

void CommandHandler::HandleReadCal1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    SensorCalibrationValue sensorCal;

    ParseSensorCal(&sensorCal, &pBuffer[SerialFlashAddress_SensorCal - baseAddress]);
    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyCal1(sensorCal);
    }
}

void CommandHandler::HandleReadCal2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    AnalogStickValidRange leftStickValidRange;
    AnalogStickValidRange rightStickValidRange;
    ::nn::util::Color4u8Type mainColor;
    ::nn::util::Color4u8Type subColor;

    ParseStickLeftCal(&leftStickValidRange, &pBuffer[SerialFlashAddress_StickLeftCal - baseAddress]);
    ParseStickRightCal(&rightStickValidRange, &pBuffer[SerialFlashAddress_StickRightCal - baseAddress]);
    ParseColor(&mainColor, &pBuffer[SerialFlashAddress_ColorMain - baseAddress]);
    ParseColor(&subColor, &pBuffer[SerialFlashAddress_ColorSub - baseAddress]);

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyCal2(leftStickValidRange, rightStickValidRange, mainColor, subColor);
    }
}

void CommandHandler::HandleReadDesign(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    DeviceColor color;

    ParseColor(&color.color[0], &pBuffer[SerialFlashAddress_ColorMain - baseAddress]);
    ParseColor(&color.color[1], &pBuffer[SerialFlashAddress_ColorSub - baseAddress]);
    ParseColor(&color.color[2], &pBuffer[SerialFlashAddress_ColorThird - baseAddress]);
    ParseColor(&color.color[3], &pBuffer[SerialFlashAddress_ColorForth - baseAddress]);


    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyDesign(color, pBuffer[SerialFlashAddress_DesignVariation - baseAddress]);
    }
}

void CommandHandler::HandleReadModel1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    SensorState sensorHorizontalOffset;
    AnalogStickDeviceParameter stickDeviceParameter;
    uint16_t stickOriginPlay;
    uint16_t stickCircuitValidRatio;
    bool isStickXScalingRequired;

    ParseStickModelValue(&stickDeviceParameter,
        &stickOriginPlay,
        &stickCircuitValidRatio,
        &isStickXScalingRequired,
        &pBuffer[SerialFlashAddress_ModelStickLeft - baseAddress]);
    ParseSensorData(&sensorHorizontalOffset,
        &pBuffer[SerialFlashAddress_SensorHorizontal - baseAddress]);
    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        // モデル値カウントを FSR = 8G から FSR = 2G の単位系に変換
        sensorHorizontalOffset.x *= 4;
        sensorHorizontalOffset.y *= 4;
        sensorHorizontalOffset.z *= 4;

        pListener->NotifyModel1(sensorHorizontalOffset,
            stickDeviceParameter,
            stickOriginPlay,
            stickCircuitValidRatio,
            isStickXScalingRequired);
    }
}

void CommandHandler::HandleReadModel2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    AnalogStickDeviceParameter stickDeviceParameter;
    uint16_t stickOriginPlay;
    uint16_t stickCircuitValidRatio;
    bool isStickXScalingRequired;

    ParseStickModelValue(&stickDeviceParameter,
        &stickOriginPlay,
        &stickCircuitValidRatio,
        &isStickXScalingRequired,
        &pBuffer[SerialFlashAddress_ModelStickRight - baseAddress]);
    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyModel2(stickDeviceParameter,
            stickOriginPlay,
            stickCircuitValidRatio,
            isStickXScalingRequired);
    }
}

void CommandHandler::HandleReadUserCal1(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    AnalogStickValidRange leftStickValidRange;
    AnalogStickValidRange rightStickValidRange;
    bool isLeftStickUserCalValid;
    bool isRightStickUserCalValid;
    bool isSensorUserCalValid;

    ParseStickLeftCal(&leftStickValidRange, &pBuffer[SerialFlashAddress_UserStickLeftCal - baseAddress]);
    ParseStickRightCal(&rightStickValidRange, &pBuffer[SerialFlashAddress_UserStickRightCal - baseAddress]);
    isLeftStickUserCalValid = IsUserCalAvailable(
        &pBuffer[SerialFlashAddress_UserStickLeftMagicNum - baseAddress]);
    isRightStickUserCalValid = IsUserCalAvailable(
        &pBuffer[SerialFlashAddress_UserStickRightMagicNum - baseAddress]);
    isSensorUserCalValid = IsUserCalAvailable(
        &pBuffer[SerialFlashAddress_UserSensorMagicNum - baseAddress]);

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyUserCal1(isLeftStickUserCalValid,
            leftStickValidRange,
            isRightStickUserCalValid,
            rightStickValidRange,
            isSensorUserCalValid);
    }
}

void CommandHandler::HandleReadUserCal2(const uint8_t* pBuffer, uint32_t baseAddress, ICommandListener* pListener) NN_NOEXCEPT
{
    SensorCalibrationValue sensorCal;

    ParseSensorCal(&sensorCal, &pBuffer[SerialFlashAddress_UserSensorCal - baseAddress]);

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyUserCal2(sensorCal);
    }
}

void CommandHandler::HandleExtDevInfo(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t extDevType;

    extDevType = pBuffer[ExtDevInfoOffset_DeviceType];
    NN_DETAIL_XCD_INFO("ExtDevType = %x\n", extDevType);

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyExtDevInfo(extDevType);
    }
}

void CommandHandler::HandleExtDevRead(const uint8_t* pBuffer, ICommandListener* pListener) NN_NOEXCEPT
{
    uint8_t status = pBuffer[ExtDevReadOffset_Status];
    uint8_t dataSize = pBuffer[ExtDevReadOffset_Size];
    const uint8_t* pData = &pBuffer[ExtDevReadOffset_Data];

    // Hid Commandの完了を通知
    if (pListener != nullptr)
    {
        pListener->NotifyExtDevRead(status, pData, dataSize);
    }
}

}} // namespace nn::xcd
