﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/xcd/detail/xcd_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TimerEvent.h>
#include "xcd_BleCommandHandler.h"
#include "xcd_TaskManager.h"

namespace nn { namespace xcd {

namespace
{
    const ::nn::TimeSpan BleCommandResponseTimeout = ::nn::TimeSpan::FromMilliSeconds(5000);
}

BleCommandHandler::BleCommandHandler() NN_NOEXCEPT
    : m_IsActivated(false)
    , m_pListener(nullptr)
    , m_CommandHandlerType(CommandHandlerType_NotSupported)
    , m_ResponseCount(0)
    , m_ResponseIndex(0)
    , m_CompletedCount(0)
    , m_IsSystem(false)
{
    m_ResponseMutex = NN_OS_SDK_MUTEX_INITIALIZER();
}

BleCommandHandler::~BleCommandHandler() NN_NOEXCEPT
{
    // 何もしない
}

void BleCommandHandler::NotifyCommandCompleted(BleDeviceOperationType operationType, BleDeviceOperationResultType resultType) NN_NOEXCEPT
{
    if (m_IsSystem)
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pListener);
        m_IsSystem = false;
        m_pListener->NotifyCommandCompleted(operationType, resultType);
        return;
    }

    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(operationType);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyReadProductType(const BleDeviceProductType& type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    m_pListener->NotifyReadProductType(type);
}

void BleCommandHandler::NotifyReadModelInformation(const BleDeviceModelInformation& info) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    m_pListener->NotifyReadModelInformation(info);
}

void BleCommandHandler::NotifyReadSensorCalibration(const SensorCalibrationValue& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    m_pListener->NotifyReadSensorCalibration(value);
}

void BleCommandHandler::NotifyReadAnalogStickCalibration(const AnalogStickValidRange& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pListener);
    m_pListener->NotifyReadAnalogStickCalibration(value);
}

void BleCommandHandler::NotifyReadContentUniqueCode(BleDeviceOperationResultType resultType, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_ReadContentUniqueCode);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    // UniqueCode が無効化されている場合は、ResultType を UniqueCodeInvalid にする
    if (pResp->result.individual.contentUniqueCode.isValid == false)
    {
        pResp->result.result = BleDeviceOperationResultType_UniqueCodeInvalid;
    }
    else
    {
        pResp->result.result = resultType;
    }

    NN_SDK_REQUIRES_EQUAL(size, BleDeviceContentUniqueCodeSize + BleDeviceContentIdSize);
    memcpy(&pResp->result.individual.contentUniqueCode.contentId, &buffer[0], BleDeviceContentIdSize);
    memcpy(&pResp->result.individual.contentUniqueCode.uniqueCode, &buffer[BleDeviceContentIdSize], BleDeviceContentUniqueCodeSize);

    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyReadStep(BleDeviceOperationResultType resultType, uint32_t step) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_GetStepCount);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->result.individual.stepCount.step = step;
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyReadContentState(BleDeviceOperationResultType resultType, bool isValid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_ReadContentUniqueCode);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            auto result = m_PalmaCommandHandler.GetContentUniqueCode();
            if (result.IsFailure())
            {
                pResp->result.result = BleDeviceOperationResultType_Busy;
                pResp->status = BleResponseStatus_Completed;

                NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
                nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
            }
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    pResp->result.result = resultType;
    pResp->result.individual.contentUniqueCode.isValid = isValid;
}

void BleCommandHandler::NotifyReadDatabaseVersion(BleDeviceOperationResultType resultType, int32_t version) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_GetDatabaseIdentificationVersion);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->result.individual.databaseIdentificationVersion.version = version;
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyReadApplicationSection(BleDeviceOperationResultType resultType, const uint8_t* buffer, int32_t size, int32_t address) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_ReadApplicationSection);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->result.individual.readApplicationSection.address = address;
    pResp->result.individual.readApplicationSection.size = size;
    NN_SDK_REQUIRES_LESS_EQUAL(static_cast<size_t>(size), BleDeviceApplicationSectionAccessUnitSizeMax);
    memcpy(pResp->result.individual.readApplicationSection.data, buffer, size);
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyRespSuspendFeatureSet(BleDeviceOperationResultType resultType, uint32_t featureFlagSet) NN_NOEXCEPT
{
    if (m_IsSystem)
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pListener);
        m_IsSystem = false;
        m_pListener->NotifyRespSuspendFeatureSet(resultType, featureFlagSet);
        return;
    }

    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_SuspendFeature);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->result.individual.suspendFeature.flagSet = featureFlagSet;
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::NotifyReadPlayLog(BleDeviceOperationResultType resultType, uint16_t index, const uint8_t* buffer, int32_t size) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleDeviceOperationType_ReadPlayLog);
    NN_SDK_REQUIRES_NOT_NULL(pResp);

    pResp->result.result = resultType;
    pResp->result.individual.playLog.index = index;
    pResp->result.individual.playLog.size = size;
    NN_SDK_REQUIRES_LESS_EQUAL(static_cast<size_t>(size), sizeof(pResp->result.individual.playLog.raw));
    memcpy(pResp->result.individual.playLog.raw, buffer, size);
    pResp->status = BleResponseStatus_Completed;
    m_CompletedCount++;

    NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
    nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
}

void BleCommandHandler::Activate(detail::BleHidAccessor* pAccessor, BleOutputCommandFormatVersion commandVer, BleCommandResponseFormatVersion responseVer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    if (m_PalmaCommandHandler.IsSupportedVersion(commandVer, responseVer))
    {
        m_CommandHandlerType = CommandHandlerType_Palma;
        m_PalmaCommandHandler.SetGattClient(pAccessor->GetBleNhogClient());
        m_PalmaCommandHandler.SetListener(this);
        m_IsActivated = true;
    }
    else
    {
        m_CommandHandlerType = CommandHandlerType_NotSupported;
    }

    // サポートするコマンドハンドラが見つかったら CommandResponse を有効にする
    if (m_IsActivated)
    {
        pAccessor->ActivateCommandResponse(CommandResponseHandler, this);

        // キューを初期化する
        for (auto& response : m_ResponseArray)
        {
            response.status = BleResponseStatus_Empty;
        }
        m_ResponseCount = 0;
        m_ResponseIndex = 0;
        m_CompletedCount = 0;

        ::nn::os::InitializeTimerEvent(&m_ResponseTimer, ::nn::os::EventClearMode_AutoClear);
    }
}

void BleCommandHandler::Deactivate(detail::BleHidAccessor* pAccessor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAccessor);

    if (m_IsActivated)
    {
        pAccessor->DeactivateCommandResponse();
        m_PalmaCommandHandler.Close();
        ::nn::os::FinalizeTimerEvent(&m_ResponseTimer);
    }

    m_IsActivated = false;
    m_pListener = nullptr;
}

void BleCommandHandler::SetListener(IBleCommandListener* pListener) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pListener);

    m_pListener = pListener;
}

Result BleCommandHandler::RegisterOperationEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotConnected());

    m_pOperationCompletedEvent = pSystemEvent;

    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetOperationResult(BleDeviceOperationResult* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    auto* pResp = FindResponseFromArray(BleResponseStatus_Completed);
    NN_RESULT_THROW_UNLESS(pResp != nullptr, nn::xcd::ResultBluetoothNoOperationResult());

    pOutValue->type = pResp->result.type;
    pOutValue->result = pResp->result.result;
    memset(pOutValue->individual.raw, 0, sizeof(pOutValue->individual.raw));
    memcpy(pOutValue->individual.raw, pResp->result.individual.raw, sizeof(pOutValue->individual.raw));

    RemoveResponseFromArray(pResp);
    m_CompletedCount--;
    // ほかにも取得できる結果が残っていたら、イベントを通知する
    if (m_CompletedCount > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pOperationCompletedEvent);
        nn::os::SignalSystemEvent(m_pOperationCompletedEvent);
    }
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetProductType() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
            NN_RESULT_DO(m_PalmaCommandHandler.GetProductType());
            break;
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetModelInformation() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
            NN_RESULT_DO(m_PalmaCommandHandler.GetModelInformation());
            break;
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetSensorCalibrationValue() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
            NN_RESULT_DO(m_PalmaCommandHandler.GetSensorCalibrationValue());
            break;
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetAnalogStickCalibrationValue() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
            NN_RESULT_DO(m_PalmaCommandHandler.GetAnalogStickCalibrationValue());
            break;
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::PlayActivity(uint16_t slotNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.PlayActivity(slotNumber));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_PlayDeviceActivity;
            pResponse->result.individual.activityIndex.index = slotNumber;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::PlayActivitySystem(uint16_t slotNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.PlayActivity(slotNumber));
            m_IsSystem = true;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::SetFrModeType(uint32_t frModeType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.SetFrModeType(frModeType));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_SetFrModeType;
            pResponse->result.individual.frModeType.type = frModeType;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.WriteDatabaseEntry(config, buffer, size));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_WriteDatabaseEntry;
            pResponse->result.individual.writeDatabaseEntry.type = config.type;
            pResponse->result.individual.writeDatabaseEntry.index = config.index;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::SetDatabaseIdentificationVersion(int32_t version) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.SetDatabaseIdentificationVersion(version));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_SetDatabaseIdentificationVersion;
            pResponse->result.individual.databaseIdentificationVersion.version = version;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetDatabaseIdentificationVersion() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.GetDatabaseIdentificationVersion());
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_GetDatabaseIdentificationVersion;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::ResetStepCount() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.ResetStepCount());
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_ResetStepCount;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::GetStepCount() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.GetStepCount());
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_GetStepCount;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::EnableStepCounter(bool isEnable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.EnableStepCounter(isEnable));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_EnableStep;
            pResponse->result.individual.stepCounterState.isEnabled = isEnable;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::ReadApplicationSection(int32_t address, size_t readSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.ReadApplicationSection(address, static_cast<int32_t>(readSize)));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_ReadApplicationSection;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::WriteApplicationSection(int32_t address, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.WriteApplicationSection(address, buffer, static_cast<int32_t>(size)));
            BleResponse* pResponse;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_WriteApplicationSection;
            pResponse->result.individual.writeApplicationSection.address = address;
            pResponse->result.individual.writeApplicationSection.size = static_cast<int32_t>(size);
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::ReadContentUniqueCode() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.GetContentState());
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_ReadContentUniqueCode;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::SetContentUniqueCodeInvalid() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.SetContentState(false));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_SetContentUniqueCodeInvalid;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::SuspendFeature(uint32_t featureFlagSet) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.SuspendFeature(featureFlagSet));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_SuspendFeature;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::SuspendFeatureSystem(uint32_t featureFlagSet) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.SuspendFeature(featureFlagSet));
            m_IsSystem = true;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::ReadPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.ReadPlayLog(index));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_ReadPlayLog;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

Result BleCommandHandler::ResetPlayLog(uint16_t index) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsActivated == true, nn::xcd::ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_ResponseCount < MaxBleOperationResultCount, nn::xcd::ResultBluetoothLeBusy());
    ::std::lock_guard<decltype(m_ResponseMutex)
                      > locker(m_ResponseMutex);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            NN_RESULT_DO(m_PalmaCommandHandler.ResetPlayLog(index));
            BleResponse* pResponse = nullptr;
            NN_RESULT_DO(GetEmptyResponseFromArray(&pResponse));
            pResponse->result.type = BleDeviceOperationType_ResetPlayLog;
            pResponse->result.individual.playLog.index = index;
            break;
        }
        case CommandHandlerType_NotSupported:
            NN_RESULT_THROW(nn::xcd::ResultNotSupported());
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
    NN_RESULT_SUCCESS;
}

void BleCommandHandler::PeriodicOperation() NN_NOEXCEPT
{
    if (m_IsActivated == false)
    {
        return;
    }

    if (nn::os::TryWaitTimerEvent(&m_ResponseTimer))
    {
        switch (m_CommandHandlerType)
        {
            case CommandHandlerType_Palma:
            {
                m_PalmaCommandHandler.HandleResponseTimeout();
                break;
            }
            case CommandHandlerType_NotSupported:
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}

void BleCommandHandler::CancelWriteDatabaseEntry() NN_NOEXCEPT
{
    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            m_PalmaCommandHandler.CancelWriteDatabaseEntry();
            break;
        }
        case CommandHandlerType_NotSupported:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void BleCommandHandler::CommandResponseHandler(void* pParser, const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    // 受信した Report の Parse
    reinterpret_cast<BleCommandHandler*>(pParser)->ParseCommnadResponse(buffer, size);
}

void BleCommandHandler::ParseCommnadResponse(const uint8_t* buffer, size_t size) NN_NOEXCEPT
{
    nn::os::StopTimerEvent(&m_ResponseTimer);

    switch (m_CommandHandlerType)
    {
        case CommandHandlerType_Palma:
        {
            auto result = m_PalmaCommandHandler.HandleResponseReceivedEvent(buffer, size);
            // コマンドが進行中の場合、タイムアウトを延長するために応答のタイマーを更新する
            if (ResultBluetoothOtherTriggerEventOnProgress::Includes(result))
            {
                ::nn::os::StartOneShotTimerEvent(&m_ResponseTimer, BleCommandResponseTimeout);
            }
            break;
        }
        case CommandHandlerType_NotSupported:
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

Result BleCommandHandler::GetEmptyResponseFromArray(BleResponse** pOutPtr) NN_NOEXCEPT
{
    // レスポンス待ちを探索する
    int nextIndex = m_ResponseIndex + MaxBleOperationResultCount + 1;
    for (int i = 0; i < MaxBleOperationResultCount; i++)
    {
        auto& response = m_ResponseArray[nextIndex % MaxBleOperationResultCount];

        if (response.status == BleResponseStatus_Empty)
        {
            response.status = BleResponseStatus_Waiting;
            m_ResponseIndex = nextIndex % MaxBleOperationResultCount;
            m_ResponseCount++;
            *pOutPtr = &response;
            NN_RESULT_SUCCESS;
        }
        nextIndex++;
    }
    NN_SDK_ASSERT(false);
    NN_RESULT_THROW(ResultBluetoothLeBusy());
}

BleCommandHandler::BleResponse* BleCommandHandler::FindResponseFromArray(BleDeviceOperationType operation) NN_NOEXCEPT
{
    if (m_ResponseCount == 0)
    {
        return nullptr;
    }

    // レスポンス待ちを探索する
    int index = m_ResponseIndex + MaxBleOperationResultCount;
    for (int i = 0; i < MaxBleOperationResultCount; i++)
    {
        auto& response = m_ResponseArray[index % MaxBleOperationResultCount];

        if (response.status == BleResponseStatus_Waiting
            && response.result.type == operation)
        {
            // レスポンスが見つかった
            return &response;
        }
        index--;
    }
    return nullptr;
}

BleCommandHandler::BleResponse* BleCommandHandler::FindResponseFromArray(BleResponseStatus status) NN_NOEXCEPT
{
    if (m_ResponseCount == 0)
    {
        return nullptr;
    }

    int index = m_ResponseIndex + MaxBleOperationResultCount;
    for (int i = 0; i < MaxBleOperationResultCount; i++)
    {
        auto& response = m_ResponseArray[index % MaxBleOperationResultCount];

        if (response.status == status)
        {
            // レスポンスが見つかった
            return &response;
        }
        index--;
    }
    return nullptr;
}

void BleCommandHandler::RemoveResponseFromArray(BleResponse* pResponse) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_ResponseCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(pResponse);

    pResponse->status = BleResponseStatus_Empty;
    memset(&pResponse->result.individual, 0, sizeof(BleDeviceOperationResult::IndividualInfo));

    m_ResponseCount--;
}

}} // namespace nn::xcd
