﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/fs.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "xcd_Attachmentbase.h"
#include "xcd_Bluetooth.h"
#include "xcd_DeviceListManager.h"
#include "xcd_PadInput.h"
#include "xcd_NfcProcessor.h"
#include "xcd_IrsensorBase.h"
#include "detail/xcd_SystemHipcServer.h"
#include "detail/xcd_TeraCommon.h"

namespace nn { namespace xcd {

namespace {

DeviceListManager g_DeviceListManager;

// サーバー処理スレッド用
const size_t ServerThreadStackSize = nn::os::ThreadStackAlignment * 2;
NN_OS_ALIGNAS_THREAD_STACK char g_ServerThreadStack[ServerThreadStackSize];
nn::os::ThreadType g_ServerThread;

// サーバースレッド
void ServerThreadFunction(void* pArg) NN_NOEXCEPT
{
    NN_UNUSED(pArg);

    // サーバーを開始
    detail::InitializeSystemServer();
    detail::LoopSystemServer();
}

// サーバースレッドを開始
void StartServerThread() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(
        &g_ServerThread,
        ServerThreadFunction,
        nullptr,
        g_ServerThreadStack,
        ServerThreadStackSize,
        NN_SYSTEM_THREAD_PRIORITY(hid, XcdIpcServer)));
    os::SetThreadNamePointer(&g_ServerThread, NN_SYSTEM_THREAD_NAME(hid, XcdIpcServer));
    os::StartThread(&g_ServerThread);
}

// データフォーマットを変更
void SetDataFormatInternal(PeriodicDataFormat format, void* pArgument) NN_NOEXCEPT
{
    auto* pHandler = reinterpret_cast<DeviceHandler*>(pArgument);
    pHandler->SetDataFormat(format);
}

}  // anonymous

DeviceListManager* GetDeviceListManager() NN_NOEXCEPT
{
    return &g_DeviceListManager;
}

void Initialize(nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMultiWait);

    GetTaskManager().Activate(pMultiWait);
    g_DeviceListManager.Activate();
    ActivateBluetoothDriver();
    StartServerThread();

    return;
}

void Finalize() NN_NOEXCEPT
{
    g_DeviceListManager.Deactivate();
    GetTaskManager().Deactivate();
}

void SuspendForSleep() NN_NOEXCEPT
{
    g_DeviceListManager.Suspend();
    SuspendBluetooth();
}

void ResumeFromSleep() NN_NOEXCEPT
{
    ResumeBluetooth();
    g_DeviceListManager.Resume();
}

bool Proceed(const nn::os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMultiWaitHolder);

    return GetTaskManager().Proceed(pMultiWaitHolder);
}

void ProceedPeriodic() NN_NOEXCEPT
{
    GetTaskManager().ProceedPeriodic();
}

nn::os::MultiWaitHolderType* WaitEvent() NN_NOEXCEPT
{
    return GetTaskManager().WaitEvent();
}

void BindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    g_DeviceListManager.BindLinkUpdateEvent(pSystemEvent);
}

void UnbindLinkUpdateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    g_DeviceListManager.UnbindLinkUpdateEvent(pSystemEvent);
}

Result ListDevices(DeviceList* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_THROW(g_DeviceListManager.ListDevices(pOutValue));
}

Result Detach(DeviceHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(handle);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->Detach();

    NN_RESULT_SUCCESS;
}

Result Reboot(DeviceHandle handle, bool reconnect, bool isFlash) NN_NOEXCEPT
{
    NN_UNUSED(handle);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->Reboot(reconnect, isFlash);

    NN_RESULT_SUCCESS;
}

Result Reboot(DeviceHandle handle, bool reconnect) NN_NOEXCEPT
{
    return Reboot(handle, reconnect, false);
}

Result McuResume(DeviceHandle handle, bool isResume) NN_NOEXCEPT
{
    NN_UNUSED(handle);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->McuResume(isResume);

    NN_RESULT_SUCCESS;
}

Result GetDeviceInfo(DeviceInfo* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetDeviceInfo(pOutValue);

    NN_RESULT_SUCCESS;
}

Result SetSamplingEvent(nn::os::SystemEventType* pSamplingEvent, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSamplingEvent);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->SetSamplingEvent(pSamplingEvent);

    NN_RESULT_SUCCESS;
}

Result GetIdentificationCode(IdentificationCode* pOutCode, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCode);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutCode = pHandler->GetIdentificationCode();

    NN_RESULT_SUCCESS;
}

Result GetDeviceColor(DeviceColor* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetColor();

    NN_RESULT_SUCCESS;
}

Result UpdateDeviceColor(const ::nn::util::Color4u8Type& mainColor, const ::nn::util::Color4u8Type& subColor, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->SetColor(mainColor, subColor);

    NN_RESULT_SUCCESS;
}

Result UpdateDesignInfo(const DeviceColor& color, uint8_t variation, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->SetDesignInfo(color, variation);

    NN_RESULT_SUCCESS;
}

Result GetDeviceStatus(DeviceStatus* pDeviceStatus, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDeviceStatus);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetDeviceStatus(pDeviceStatus);

    NN_RESULT_SUCCESS;
}

Result GetPadState(PadState* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetPadState(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetSensorStates(pOutCount, pOutValue, length);

    NN_RESULT_SUCCESS;
}

Result GetButtonTriggerElapsedTime(nn::os::Tick* pOutValue, int buttonIndex, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetPadInput()->GetButtonTriggerElapsedTime(pOutValue, buttonIndex);
}

Result GetLeftAnalogStickValidRange(AnalogStickValidRange* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetLeftAnalogStickValidRange(pOutValue);

    NN_RESULT_SUCCESS;
}

Result UpdateLeftAnalogStickValidRange(const AnalogStickValidRange& value, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->UpdateLeftAnalogStickValidRange(value));

    NN_RESULT_SUCCESS;
}

Result ResetLeftAnalogStickValidRange(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->ResetLeftAnalogStickValidRange());

    NN_RESULT_SUCCESS;
}

Result GetLeftAnalogStickValidRangeUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetLeftAnalogStickValidRangeUpdateStatus();

    NN_RESULT_SUCCESS;
}

Result GetRightAnalogStickValidRange(AnalogStickValidRange* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetRightAnalogStickValidRange(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetLeftAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetLeftAnalogStickDeviceParameter();

    NN_RESULT_SUCCESS;
}

Result UpdateRightAnalogStickValidRange(const AnalogStickValidRange& value, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->UpdateRightAnalogStickValidRange(value));

    NN_RESULT_SUCCESS;
}

Result ResetRightAnalogStickValidRange(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->ResetRightAnalogStickValidRange());

    NN_RESULT_SUCCESS;
}

Result GetRightAnalogStickValidRangeUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetRightAnalogStickValidRangeUpdateStatus();

    NN_RESULT_SUCCESS;
}

Result GetRightAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetRightAnalogStickDeviceParameter();

    NN_RESULT_SUCCESS;
}

Result GetSensorCalibrationValue(SensorCalibrationValue* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetSensorCalibrationValue(pOutValue);

    NN_RESULT_SUCCESS;
}

Result UpdateSensorCalibrationValue(const SensorCalibrationValue& value, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->UpdateSensorCalibrationValue(value));

    NN_RESULT_SUCCESS;
}

Result ResetSensorCalibrationValue(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetPadInput()->ResetSensorCalibrationValue());

    NN_RESULT_SUCCESS;
}


Result GetSensorCalibrationValueUpdateStatus(CalibrationUpdateStatus* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetSensorCalibrationValueUpdateStatus();

    NN_RESULT_SUCCESS;
}

Result GetSensorHorizontalOffset(SensorState* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetPadInput()->GetSensorHorizontalOffset();

    NN_RESULT_SUCCESS;
}

Result GetSensorFsr(AccelerometerFsr* pOutAccelerometerFsr, GyroscopeFsr* pOutGyrosocopeFsr, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAccelerometerFsr);
    NN_SDK_REQUIRES_NOT_NULL(pOutGyrosocopeFsr);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->GetSensorConfig(pOutAccelerometerFsr, pOutGyrosocopeFsr);

    NN_RESULT_SUCCESS;
}

Result SetSensorFsr(AccelerometerFsr accelerometerFsr, GyroscopeFsr gyrosocopeFsr, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->SetSensorConfig(accelerometerFsr, gyrosocopeFsr);

    NN_RESULT_SUCCESS;
}

Result SleepSensor(bool sleep, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->SleepSensor(sleep);

    NN_RESULT_SUCCESS;
}

Result IsSensorSleep(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetPadInput()->IsSensorSleep(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetPlayerIndicatorPattern(nn::Bit8* pPattern, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPattern);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetIndicatorLedPattern(pPattern);

    NN_RESULT_SUCCESS;
}

Result SetPlayerIndicatorPattern(nn::Bit8 pattern, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(pattern, 0x00, 0xf1);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->ControlLed(pattern);

    NN_RESULT_SUCCESS;
}

Result SetVibratorEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetVibrator()->SetEnabled(isEnabled, position);
}

Result IsVibratorEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetVibrator()->IsEnabled(pOutValue, position);
}

Result GetVibratorBufferStatus(VibratorBufferStatus* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->GetBufferStatus(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetRequiredVibrationValueCount(int* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->GetRequiredVibrationValueCount(pOutValue);

    NN_RESULT_SUCCESS;
}

Result SendVibrationValue(const VibrationValue& value, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    NN_RESULT_DO(pVibratorAgent->SendVibrationValue(value));

    NN_RESULT_SUCCESS;
}

Result GetActualVibrationValue(VibrationValue* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->GetActualVibrationValue(pOutValue);

    NN_RESULT_SUCCESS;
}

Result StartVibrationOnConnect(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->GetVibrator()->StartVibrationOnConnect();

    NN_RESULT_SUCCESS;
}

Result SetVibratorCalibrationValue(const VibratorCalibrationValue& value, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->SetCalibrationValue(value);

    NN_RESULT_SUCCESS;
}

Result GetVibratorCalibrationValue(VibratorCalibrationValue* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->GetCalibrationValue(pOutValue);

    NN_RESULT_SUCCESS;
}

Result SetVibratorAmplitudeLimitEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->SetAmplitudeLimitEnabled(isEnabled);

    NN_RESULT_SUCCESS;
}

Result IsVibratorAmplitudeLimitEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    *pOutValue = pVibratorAgent->IsAmplitudeLimitEnabled();

    NN_RESULT_SUCCESS;
}

Result SetVibratorEqualizerEnabled(bool isEnabled, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    pVibratorAgent->SetEqualizerEnabled(isEnabled);

    NN_RESULT_SUCCESS;
}

Result IsVibratorEqualizerEnabled(bool* pOutValue, VibratorPosition position, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    VibratorAgent* pVibratorAgent;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetVibrator()->GetVibratorAgent(&pVibratorAgent, position));
    *pOutValue = pVibratorAgent->IsEqualizerEnabled();

    NN_RESULT_SUCCESS;
}

void StartButtonPairing() NN_NOEXCEPT
{
    StartBluetoothButtonPairing();
}

void BindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    g_DeviceListManager.BindPairingCompletedEvent(pSystemEvent);
}

void UnbindPairingCompletedEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    g_DeviceListManager.UnbindPairingCompletedEvent(pSystemEvent);
}

size_t GetRegisteredDevices(RegisteredDeviceInfo* pOutValues, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);

    return g_DeviceListManager.GetRegisteredDevices(pOutValues, length);
}

void AddRegisteredDeviceInfo(const RegisteredDeviceInfo& info) NN_NOEXCEPT
{
    g_DeviceListManager.AddRegisteredDeviceInfo(info);
}

nn::Result SetFirmwareUpdatingDevice(DeviceHandle handle) NN_NOEXCEPT
{
    return g_DeviceListManager.SetFirmwareUpdatingDevice(handle);
}

void UnsetFirmwareUpdatingDevice() NN_NOEXCEPT
{
    g_DeviceListManager.UnsetFirmwareUpdatingDevice();
}

nn::Result StartBtFirmwareUpdate(
    DeviceHandle handle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->StartBtFirmwareUpdate(image, pSystemEvent));

    NN_RESULT_SUCCESS;
}

nn::Result AbortBtFirmwareUpdate(
    DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->AbortBtFirmwareUpdate());

    NN_RESULT_SUCCESS;
}

nn::Result GetBtFirmwareVersion(BtFirmwareVersion* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetBtFirmwareVersion();

    NN_RESULT_SUCCESS;
}

bool IsBtFirmwareUpdateThroughBtRequired(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    if (g_DeviceListManager.GetDeviceHandler(&pHandler, handle).IsFailure())
    {
        return false;
    }

    return pHandler->IsBtFirmwareUpdateThroughBtRequired();
}
nn::Result GetBtFirmwareUpdateProgress(DeviceHandle handle, FirmwareUpdateStage* pOutStage, int* pOutProgress) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetBtFirmwareUpdater()->GetFirmwareUpdateProgress(pOutStage, pOutProgress));

    NN_RESULT_SUCCESS;
}

void SetBtFirmwareUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT
{
    BtFirmwareUpdater::SetFirmwareUpdateFailureEmulationEnabled(isEnabled);
}

nn::Result GetMcuVersion(McuVersionData* pOutMcuVersionData, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuVersionData);

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetTeraBase()->GetMcuFirmwareVersion(pOutMcuVersionData);
}

nn::Result RequestMcuVersion(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetTeraBase()->RequestMcuFirmwareVersion();
}

nn::Result StartMcuUpdate(
    DeviceHandle deviceHandle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pFinishEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinishEvent);

    // Tera FW 更新は Horizon のみサポート
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    auto* pTeraBase = pHandler->GetTeraBase();
    pTeraBase->SetChangeDataFormatFunction(SetDataFormatInternal, pHandler);

    return pTeraBase->StartMcuFirmwareUpdate(
        image,
        pFinishEvent,
        ITeraUpdater::UpdateMode::Normal
    );
#else
    NN_UNUSED(deviceHandle);
    NN_UNUSED(image);
    NN_UNUSED(pFinishEvent);

    NN_RESULT_THROW(nn::xcd::ResultNotSupported());
#endif  // if defined(NN_BUILD_CONFIG_OS_HORIZON)
}

nn::Result StartMcuUpdateFull(
    DeviceHandle deviceHandle,
    const FirmwareImage& image,
    nn::os::SystemEventType* pFinishEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFinishEvent);

    // Tera FW 更新は Horizon のみサポート
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    auto* pTeraBase = pHandler->GetTeraBase();
    pTeraBase->SetChangeDataFormatFunction(SetDataFormatInternal, pHandler);

    return pTeraBase->StartMcuFirmwareUpdate(
        image,
        pFinishEvent,
        ITeraUpdater::UpdateMode::Full
    );
#else
    NN_UNUSED(deviceHandle);
    NN_UNUSED(image);
    NN_UNUSED(pFinishEvent);

    NN_RESULT_THROW(nn::xcd::ResultNotSupported());
#endif  // if defined(NN_BUILD_CONFIG_OS_HORIZON)
}

nn::Result AbortMcuUpdate(DeviceHandle handle) NN_NOEXCEPT
{
    // Tera FW 更新は Horizon のみサポート
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetTeraBase()->ProcessUpdater([](ITeraUpdater* pUpdater)
    {
        NN_SDK_ASSERT_NOT_NULL(pUpdater);
        return pUpdater->Abort();
    });
#else
    NN_UNUSED(handle);

    NN_RESULT_THROW(nn::xcd::ResultNotSupported());
#endif  // if defined(NN_BUILD_CONFIG_OS_HORIZON)
}

nn::Result GetMcuUpdateState(McuUpdateStateInfo* pOutMcuUpdateStateInfo, DeviceHandle handle) NN_NOEXCEPT
{
    // Tera FW 更新は Horizon のみサポート
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuUpdateStateInfo);

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetTeraBase()->ProcessUpdater([&pOutMcuUpdateStateInfo](ITeraUpdater* pUpdater)
    {
        NN_SDK_ASSERT_NOT_NULL(pUpdater);
        return pUpdater->GetState(pOutMcuUpdateStateInfo);
    });
#else
    NN_UNUSED(pOutMcuUpdateStateInfo);
    NN_UNUSED(handle);

    NN_RESULT_THROW(nn::xcd::ResultNotSupported());
#endif  // if defined(NN_BUILD_CONFIG_OS_HORIZON)
}

void SetMcuHardwareErrorEmulationEnabled(bool isEnabled) NN_NOEXCEPT
{
    TeraBase::SetMcuBrokenEmulationEnabled(isEnabled);
}

void SetMcuUpdateFailureEmulationEnabled(bool isEnabled) NN_NOEXCEPT
{
    TeraBase::SetFirmwareUpdateFailureEmulationEnabled(isEnabled);
}

Result GetKuinaVersion(KuinaVersionData* pOutMcuVersionData, int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuVersionData);
    NN_RESULT_DO(
        g_DeviceListManager.GetKuinaVersion(pOutMcuVersionData,index));
    NN_RESULT_SUCCESS;
}

Result RequestKuinaVersion(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        g_DeviceListManager.RequestKuinaVersion(index));
    NN_RESULT_SUCCESS;
}

Result SetKuinaToFirmwareUpdateMode(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        g_DeviceListManager.SetKuinaToFirmwareUpdateMode(index));
    NN_RESULT_SUCCESS;
}

Result CheckIrDevicePower(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return detail::CheckIrDevicePowerImpl(pHandler);
}

Result SetIrDataFormat(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->SetDataFormat(::nn::xcd::PeriodicDataFormat_MCU);

    NN_RESULT_SUCCESS;
}

Result ResetDataFormat(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->SetDataFormat(::nn::xcd::PeriodicDataFormat_Basic);

    NN_RESULT_SUCCESS;
}

Result GetIrMcuState(McuState* pOutMcuState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMcuState);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    *pOutMcuState = pHandler->GetTeraBase()->GetMcuState();

    NN_RESULT_SUCCESS;
}

Result SetIrMcuState(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetTeraBase()->SetMcuState(mcuState));

    NN_RESULT_SUCCESS;
}

Result SetIrMcuStateImmediate(McuState mcuState, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetTeraBase()->SetMcuStateImmediate(mcuState));

    NN_RESULT_SUCCESS;
}

Result SetIrControlEvent(nn::os::SystemEventType* pIrSamplingEvent,
                         nn::os::SystemEventType* pIrCommandCompletionEvent,
                         DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrSamplingEvent);
    NN_SDK_REQUIRES_NOT_NULL(pIrCommandCompletionEvent);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrSamplingEvent, pIrCommandCompletionEvent](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetIrControlEvent(pIrSamplingEvent, pIrCommandCompletionEvent);
    });
}

Result ReleaseIrControlEvent(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->ReleaseIrControlEvent();
    });
}

Result SetIrProcessorType(IrProcessorType type, int modeOffset, IrTeraPluginParameter param, IrMcuVersion requiredVersion, IrCommandType commandType, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([type, modeOffset, param, requiredVersion, commandType](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetIrProcessorType(type, modeOffset, param, requiredVersion, commandType);
    });
}

Result GetIrProcessorType(IrProcessorType* pType, IrMcuVersion* pCompatibleVersion, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pType);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pType, pCompatibleVersion](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetIrProcessorType(pType, pCompatibleVersion);
    });
}

Result StartIrSampling(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartSampling();
    });
}

Result StopIrSampling(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StopSampling();
    });
}

Result TeardownIrProcessor(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->TeardownProcessor();
    });
}

Result EnableIrPollingMode(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->EnablePollingMode();
    });
}

Result DisableIrPollingMode(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->DisablePollingMode();
    });
}

Result SetupIrMomentProcessor(IrCommonData* pIrCommonWorkBuffer, IrMomentProcessorState* pMomentProcessorWorkBuffer, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pMomentProcessorWorkBuffer);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrCommonWorkBuffer, pMomentProcessorWorkBuffer](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetupMomentProcessor(pIrCommonWorkBuffer, pMomentProcessorWorkBuffer);
    });
}

Result GetIrMomentStates(IrCommonData* pOutIrCommonData, IrMomentProcessorState* pOutMomentProcessorStates, int* pOutCount, int countMax, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrCommonData);
    NN_SDK_REQUIRES_NOT_NULL(pOutMomentProcessorStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrCommonData, pOutMomentProcessorStates, pOutCount, countMax](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetMomentStates(pOutIrCommonData, pOutMomentProcessorStates, pOutCount, countMax);
    });
}

Result SetupIrClusteringProcessor(IrCommonData* pIrCommonWorkBuffer, IrClusteringProcessorState* pClusteringProcessorWorkBuffer, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pClusteringProcessorWorkBuffer);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrCommonWorkBuffer, pClusteringProcessorWorkBuffer](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetupClusteringProcessor(pIrCommonWorkBuffer, pClusteringProcessorWorkBuffer);
    });
}

Result GetIrClusteringStates(IrCommonData* pOutIrCommonData, IrClusteringProcessorState* pOutClusteringProcessorStates, int* pOutCount, int countMax, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrCommonData);
    NN_SDK_REQUIRES_NOT_NULL(pOutClusteringProcessorStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrCommonData, pOutClusteringProcessorStates, pOutCount, countMax](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetClusteringStates(pOutIrCommonData, pOutClusteringProcessorStates, pOutCount, countMax);
    });
}

Result SetupIrImageTransferProcessor(IrCommonData* pIrCommonWorkBuffer, IrImageTransferProcessorState* pImageTransferProcessorWorkBuffer, IrImageTransferProcessorFormat size, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pImageTransferProcessorWorkBuffer);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrCommonWorkBuffer, pImageTransferProcessorWorkBuffer, size](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetupImageTransferProcessor(pIrCommonWorkBuffer, pImageTransferProcessorWorkBuffer, size);
    });
}

Result GetIrImageTransferState(IrCommonData* pOutIrCommonData, IrImageTransferProcessorState* pOutImageTransferProcessorState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrCommonData);
    NN_SDK_REQUIRES_NOT_NULL(pOutImageTransferProcessorState);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrCommonData, pOutImageTransferProcessorState](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetImageTransferState(pOutIrCommonData, pOutImageTransferProcessorState);
    });
}

Result SetupIrTeraPluginProcessor(IrCommonData* pIrCommonWorkBuffer, IrTeraPluginProcessorState* pTeraPluginProcessorWorkBuffer, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pTeraPluginProcessorWorkBuffer);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrCommonWorkBuffer, pTeraPluginProcessorWorkBuffer](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetupTeraPluginProcessor(pIrCommonWorkBuffer, pTeraPluginProcessorWorkBuffer);
    });
}

Result GetIrTeraPluginState(IrCommonData* pOutIrCommonData, IrTeraPluginProcessorState* pOutTeraPluginProcessorState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrCommonData);
    NN_SDK_REQUIRES_NOT_NULL(pOutTeraPluginProcessorState);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrCommonData, pOutTeraPluginProcessorState](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetTeraPluginState(pOutIrCommonData, pOutTeraPluginProcessorState);
    });
}

Result SetupIrDpdProcessor(IrCommonData* pIrCommonWorkBuffer, IrDpdProcessorState* pDpdProcessorWorkBuffer, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pIrCommonWorkBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pDpdProcessorWorkBuffer);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pIrCommonWorkBuffer, pDpdProcessorWorkBuffer](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SetupDpdProcessor(pIrCommonWorkBuffer, pDpdProcessorWorkBuffer);
    });
}

Result GetIrDpdStates(IrCommonData* pOutIrCommonData, IrDpdProcessorState* pOutDpdProcessorStates, int* pOutCount, int countMax, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrCommonData);
    NN_SDK_REQUIRES_NOT_NULL(pOutDpdProcessorStates);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrCommonData, pOutDpdProcessorStates, pOutCount, countMax](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetDpdStates(pOutIrCommonData, pOutDpdProcessorStates, pOutCount, countMax);
    });
}

Result StartIrReadRegister(const IrReadRegisterSetting& irReadRegisterSetting, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([irReadRegisterSetting](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartReadRegister(irReadRegisterSetting);
    });
}

Result StartIrWriteRegister(const IrWriteRegisterSetting& irWriteRegisterSetting, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([irWriteRegisterSetting](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartWriteRegister(irWriteRegisterSetting);
    });
}

Result StartIrWriteRegisterEx(const IrWriteRegisterSettingEx& irWriteRegisterSetting, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([irWriteRegisterSetting](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartWriteRegisterEx(irWriteRegisterSetting);
    });
}

Result GetIrReadRegisterState(IrReadRegisterState* pOutIrReadRegisterState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrReadRegisterState);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrReadRegisterState](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetReadRegisterState(pOutIrReadRegisterState);
    });
}

Result GetIrWriteRegisterState(IrWriteRegisterState* pOutIrWriteRegisterState, DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutIrWriteRegisterState);

    DeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    return pHandler->GetTeraBase()->ProcessIrCommand([pOutIrWriteRegisterState](IrsensorBase* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetWriteRegisterState(pOutIrWriteRegisterState);
    });
}

void EnableNwcp() NN_NOEXCEPT
{
    g_DeviceListManager.SetNwcpEnabled(true);
}

void DisableNwcp() NN_NOEXCEPT
{
    g_DeviceListManager.SetNwcpEnabled(false);
}

bool IsUsbHidSupported(UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    return g_DeviceListManager.IsUsbHidSupported(deviceInfo);
}

Result AddUsbHidDevice(int index, UsbHidDeviceInfo deviceInfo) NN_NOEXCEPT
{
    NN_RESULT_DO(
        g_DeviceListManager.AddUsbHidDevice(index, deviceInfo));
    NN_RESULT_SUCCESS;
}

Result RemoveUsbHidDevice(int index) NN_NOEXCEPT
{
    NN_RESULT_DO(
        g_DeviceListManager.RemoveUsbHidDevice(index));
    NN_RESULT_SUCCESS;
}

Result SetUsbHidInputReport(int index, uint8_t* pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(
        g_DeviceListManager.SetUsbHidInputReport(index, pBuffer, length));
    NN_RESULT_SUCCESS;
}

size_t GetUsbHidOutputReport(int index, uint8_t* pOutBuffer, size_t length) NN_NOEXCEPT
{
    return g_DeviceListManager.GetUsbHidOutputReport(index, pOutBuffer, length);
}

void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);
    g_DeviceListManager.SetRailUpdateEvent(pEvent);
}

RailUpdateEventInfo GetRailUpdateEventInfo() NN_NOEXCEPT
{
    return g_DeviceListManager.GetRailUpdateEventInfo();
}

bool IsDeviceAttachedOnRail(RailType type) NN_NOEXCEPT
{
    return g_DeviceListManager.IsDeviceAttachedOnRail(type);
}

void EnableWiredDeviceRegistration() NN_NOEXCEPT
{
    g_DeviceListManager.SetWiredDeviceActivationEnabled(true);
}

void DisableWiredDeviceRegistration() NN_NOEXCEPT
{
    g_DeviceListManager.SetWiredDeviceActivationEnabled(false);
}

nn::hid::detail::RxPacketHistory GetRxPacketHistory(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    if (g_DeviceListManager.GetDeviceHandler(&pHandler, handle).IsFailure())
    {
        nn::hid::detail::RxPacketHistory returnValue = {};
        returnValue.samplingCount = 0;
        return returnValue;
    }

    return pHandler->GetRxPacketHistory();
}

Result ResetBatteryChargeTimer(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    pHandler->ResetBatteryChargerTimer();
    NN_RESULT_SUCCESS;
}

void SetFullKeyUsbEnabled(bool enabled) NN_NOEXCEPT
{
    g_DeviceListManager.SetFullKeyUsbEnabled(enabled);
}

void GetFullKeyUsbEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    g_DeviceListManager.GetFullKeyUsbEnabled(pOutEnabled);
}

void GetUsbConnected(bool* pOutValue, DeviceHandle handle) NN_NOEXCEPT
{
    g_DeviceListManager.GetUsbConnected(pOutValue, handle);
}

void SetConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    g_DeviceListManager.SetConnectionTriggerTimeoutEvent(pSystemEvent);
}

Result SendConnectionTrigger(::nn::bluetooth::Address address) NN_NOEXCEPT
{
    NN_RESULT_DO(g_DeviceListManager.SendConnectionTrigger(address));
    NN_RESULT_SUCCESS;
}

int GetMaxBluetoothLinks() NN_NOEXCEPT
{
    return g_DeviceListManager.GetMaxBluetoothLinks();
}

nn::Result AttachmentEnable(bool isEnabled, AttachmentFunctionRequiredVersion version, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    // Required Version の設定
    pHandler->GetAttachmentBase()->SetRequiredVersion(version);

    // Enable 時は BTFW のバージョンをチェックする。Tera は一度 Tera を起こす必要があるので後段でバージョンチェックする。
    if (isEnabled)
    {
        nn::xcd::BtFirmwareVersion btVersion;
        NN_RESULT_DO(nn::xcd::GetBtFirmwareVersion(&btVersion, handle));
        if (!pHandler->GetAttachmentBase()->BtVersionCheck(btVersion))
        {
            return nn::xcd::ResultAttachmentDeviceNeedUpdate();
        }
    }

    NN_RESULT_DO(pHandler->AttachmentPowerEnable(isEnabled));

    if (isEnabled)
    {
        NN_RESULT_DO(pHandler->GetAttachmentBase()->SendInquiryCommand());
    }
    else
    {
        if (pHandler->GetAttachmentBase()->IsExtDevAttached())
        {
            // Attach されている場合、Status を Attached まで戻す
            pHandler->GetAttachmentBase()->SetAttachmentStatus(AttachmentStatus_Attached);
        }
        pHandler->GetAttachmentBase()->SetAttachmentType(0x00);
    }

    NN_RESULT_SUCCESS;
}
Result GetExtDevInfo(ExtDevInfo* pOutExtDevInfo, DeviceHandle handle) NN_NOEXCEPT
{
    uint8_t extDevType;

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    auto result = pHandler->GetAttachmentBase()->GetAttachmentType(&extDevType);

    pOutExtDevInfo->extDevType = extDevType;
    return result;
}

Result SendCommandToAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetAttachmentBase()->SendCommandToAttachmentDevice(pInCommand, inCommandSize);
}

Result GetReceivedCommandFromAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetAttachmentBase()->GetReceivedCommandFromAttachmentDevice(pOutSize, pOutCommand, static_cast<uint8_t>(outCommandSize));
}

nn::Result SetAttachmentEvent(nn::os::SystemEventType* pAttachmentDataReceiveEvent, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetAttachmentBase()->SetAttachmentDataReceiveEvent(pAttachmentDataReceiveEvent);
}

bool IsAttachmentDeviceAttached(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    auto result = g_DeviceListManager.GetDeviceHandler(&pHandler, handle);
    if (result.IsSuccess())
    {
        return pHandler->GetAttachmentBase()->IsExtDevAttached();
    }

    return false;
}

Result EnablePollingReceiveModeForAttachmentDevice(const uint8_t* pInCommand, size_t inCommandSize, DeviceHandle handle, AttachmentPollingMode mode) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    nn::xcd::DeviceInfo deviceInfo;
    pHandler->GetDeviceInfo(&deviceInfo);

    return pHandler->GetAttachmentBase()->EnablePollingReceiveModeForAttachmentDevice(pInCommand, inCommandSize, mode, deviceInfo.deviceType);
}

Result DisablePollingReceiveModeForAttachmentDevice(DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    NN_RESULT_DO(pHandler->GetAttachmentBase()->DisablePollingReceiveModeForAttachmentDevice());

    NN_RESULT_SUCCESS;
}

Result GetPollingDataForAttachmentDevice(size_t* pOutSize, uint8_t* pOutCommand, size_t outCommandSize, DeviceHandle handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));

    return pHandler->GetAttachmentBase()->GetPollingDataForAttachmentDevice(pOutSize, pOutCommand, static_cast<uint8_t>(outCommandSize));
}

Result ListDevices(BleDeviceList* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_THROW(g_DeviceListManager.ListDevices(pOutValue));
}

Result Detach(BleConnectionHandle handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->Detach();

    NN_RESULT_SUCCESS;
}

Result GetDeviceInfo(BleDeviceInfo* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetDeviceInfo(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetDeviceStatus(DeviceStatus* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetDeviceStatus(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetPadState(PadState* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetBlePadInput()->GetPadState(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetSensorStates(int* pOutCount, SixAxisSensorState* pOutValue, int length, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetBlePadInput()->GetSensorStates(pOutCount, pOutValue, length);

    NN_RESULT_SUCCESS;
}

Result GetBleDeviceOperationResult(BleDeviceOperationResult* pOutValue, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->GetOperationResult(pOutValue));

    NN_RESULT_SUCCESS;
}

Result RegisterBleDeviceOperationEvent(nn::os::SystemEventType* pSystemEvent, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->RegisterOperationEvent(pSystemEvent));

    NN_RESULT_SUCCESS;
}

Result PlayDeviceActivity(uint16_t number, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->PlayActivity(number));

    NN_RESULT_SUCCESS;
}

Result SetFrModeType(uint32_t frModeType, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->SetFrModeType(frModeType));

    NN_RESULT_SUCCESS;
}

Result WriteDatabaseEntry(const BleDeviceDatabaseEntryConfig& config, size_t size, const void* pEntryBuffer, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEntryBuffer);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->WriteDatabaseEntry(config, reinterpret_cast<const uint8_t*>(pEntryBuffer), size));

    NN_RESULT_SUCCESS;
}

Result SetDatabaseIdentificationVersion(int32_t version, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->SetDatabaseIdentificationVersion(version));

    NN_RESULT_SUCCESS;
}

Result GetDatabaseIdentificationVersion(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->GetDatabaseIdentificationVersion());

    NN_RESULT_SUCCESS;
}

Result GetStepCount(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->GetStepCount());

    NN_RESULT_SUCCESS;
}

Result ResetStepCount(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->ResetStepCount());

    NN_RESULT_SUCCESS;
}

Result EnableStepCounter(bool isEnabled, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->EnableStepCounter(isEnabled));

    NN_RESULT_SUCCESS;
}

Result ReadApplicationSection(int32_t address, size_t readSize, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->ReadApplicationSection(address, readSize));

    NN_RESULT_SUCCESS;
}

Result WriteApplicationSection(int32_t address, const void* pBuffer, size_t size, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->WriteApplicationSection(address, reinterpret_cast<const uint8_t*>(pBuffer), size));

    NN_RESULT_SUCCESS;
}

Result ReadContentUniqueCode(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->ReadContentUniqueCode());

    NN_RESULT_SUCCESS;
}

Result SetContentUniqueCodeInvalid(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->SetContentUniqueCodeInvalid());

    NN_RESULT_SUCCESS;
}

Result SuspendFeature(uint32_t featureFlagSet, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->SuspendFeature(featureFlagSet));

    NN_RESULT_SUCCESS;
}

nn::hid::detail::RxPacketHistory GetRxPacketHistory(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;
    if (g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle).IsFailure())
    {
        nn::hid::detail::RxPacketHistory returnValue;
        returnValue.samplingCount = 0;
        return returnValue;
    }

    return pHandler->GetRxPacketHistory();
}

void DisableBle() NN_NOEXCEPT
{
    g_DeviceListManager.DisableBle();
}

Result ReadDevicePlayLog(uint16_t index, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->ReadPlayLog(index));

    NN_RESULT_SUCCESS;
}

Result ResetDevicePlayLog(uint16_t index, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->ResetPlayLog(index));

    NN_RESULT_SUCCESS;
}

void SetIsPalmaAllConnectable(bool connectable) NN_NOEXCEPT
{
    g_DeviceListManager.SetIsPalmaAllConnectable(connectable);
}

void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
{
    g_DeviceListManager.SetIsPalmaPairedConnectable(connectable);
}

Result PairPalma(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    BleDeviceInfo info;
    pHandler->GetDeviceInfo(&info);
    NN_RESULT_DO(g_DeviceListManager.PairPalma(info.address));
    NN_RESULT_SUCCESS;
}

void CancelWriteDatabaseEntry(const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    auto result = g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle);
    if (result.IsFailure())
    {
        return;
    }
    auto command = pHandler->GetBleCommand();
    command->CancelWriteDatabaseEntry();
}

Result PlayDeviceSystemActivity(uint16_t number, const BleConnectionHandle& handle) NN_NOEXCEPT
{
    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    auto command = pHandler->GetBleCommand();
    NN_RESULT_DO(command->PlayActivitySystem(number));

    NN_RESULT_SUCCESS;
}

Result GetAnalogStickValidRange(AnalogStickValidRange* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetAnalogStickValidRange(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetAnalogStickDeviceParameter(AnalogStickDeviceParameter* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetAnalogStickDeviceParameter();

    NN_RESULT_SUCCESS;
}

Result GetSensorCalibrationValue(SensorCalibrationValue* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    pHandler->GetSensorCalibrationValue(pOutValue);

    NN_RESULT_SUCCESS;
}

Result GetSensorHorizontalOffset(SensorState* pOutValue, BleConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    BleDeviceHandler* pHandler;

    NN_RESULT_DO(g_DeviceListManager.GetBleDeviceHandler(&pHandler, handle));
    *pOutValue = pHandler->GetSensorHorizontalOffset();

    NN_RESULT_SUCCESS;
}

bool IsFiftyConnected() NN_NOEXCEPT
{
    return g_DeviceListManager.IsFiftyConnected();
}

PadState GetFiftyPadState() NN_NOEXCEPT
{
    return g_DeviceListManager.GetFiftyPadState();
}

Result ReadSerialFlash(
    const uint32_t address,
    uint8_t* pOutBuffer,
    int size,
    nn::os::SystemEventType* pEvent,
    const DeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetSerialFLashAccessor()->Read(address, pOutBuffer, size, pEvent));
    NN_RESULT_SUCCESS;
}

Result WriteSerialFlash(
    const uint32_t address,
    const uint8_t* pBuffer,
    int size,
    nn::os::SystemEventType* pEvent,
    DeviceHandle& handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetSerialFLashAccessor()->Write(address, pBuffer, size, pEvent));
    NN_RESULT_SUCCESS;
}

Result GetSerialFlashResult(const DeviceHandle& handle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(g_DeviceListManager.GetDeviceHandler(&pHandler, handle));
    NN_RESULT_DO(pHandler->GetSerialFLashAccessor()->GetResult());
    NN_RESULT_SUCCESS;
}

}} // namespace nn::xcd
