﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/util/util_BitPack.h>
#include <nn/xcd/xcd_NfcTypes.h>
#include "xcd_TeraCommon.h"

namespace nn { namespace xcd { namespace detail {

/**
 * @brief   UID の最大長です。
 */
const size_t UidLengthMax = 10;

/**
 * @brief   検出可能なタグの最大数です。
 */
const size_t DetectableTagCount = 4;

/**
 * @brief   NFC パケットの最大ペイロード長です。
 *
 * ※ 仕様上は 2047 が上限だが、通信フォーマットで規定している最大値は NFC_WRITE_START の 990
 */
const size_t PayloadLengthMax = 990;

/**
 * @brief   タグ読み込み時の最大データサイズです。
 */
const size_t TagReadDataBytesMax = 969;

/**
 * @brief   認証情報のサイズです。
 */
const size_t SignatureBytes = 32;

/**
 * @brief   MIFARE アクセスに使用する NUID (4 バイト UID) の長さです。
 */
const size_t MifareNuidLength = 4;

/**
 * @brief   NFC ステート用の NFC_MANAGEMENT コマンドです。
 */
enum NfcCommand : uint8_t
{
    NfcCommand_StartDiscovery       =  1,   //!< タグの検出を開始
    NfcCommand_StopDiscovery        =  2,   //!< タグの検出を停止
    NfcCommand_Nop                  =  4,   //!< 何もしない (状態のポーリング用)
    NfcCommand_StateInf             =  5,   //!< NFC 処理の応答
    NfcCommand_ReadStart            =  6,   //!< タグ読み取り開始
    NfcCommand_ReadDataResponse     =  7,   //!< 読み取ったタグデータの応答
    NfcCommand_WriteStart           =  8,   //!< タグ書き込み開始
    NfcCommand_PassThru             =  9,   //!< パススルーデータ送信
    NfcCommand_PassThruResponse     = 10,   //!< パススルー応答
    NfcCommand_ModuleCheck          = 11,   //!< NFC のモジュール検査
    NfcCommand_ModuleCheckResponse  = 12,   //!< NFC のモジュール検査応答
    NfcCommand_ModuleSpi            = 13,   //!< NFC への SPI コマンド送信
    NfcCommand_ModuleSpiResponse    = 14,   //!< NFC からの SPI コマンド応答
    NfcCommand_MifareRequest        = 15,   //!< MIFARE データ送信
    NfcCommand_MifareResponse       = 16,   //!< MIFARE 応答
    NfcCommand_MifareKeyStartWrite  = 17    //!< MIFARE 鍵書き込み開始
};

/**
 * @brief   NFC 内部ステートです。
 */
enum NfcState : uint8_t
{
    NfcState_CommandWaiting         =  0,   //!< コマンド待ち受け状態
    NfcState_Polling                =  1,   //!< タグポーリング中
    NfcState_Reading                =  2,   //!< タグからの読み込み処理中
    NfcState_Writing                =  3,   //!< タグへの書き込み処理中
    NfcState_ReadFinish             =  4,   //!< 読み込み終了状態
    NfcState_WriteFinish            =  5,   //!< 書き込み終了状態
    NfcState_PassThruSending        =  6,   //!< パススルーコマンド送信中
    NfcState_Error                  =  7,   //!< エラー発生時
    NfcState_Deactivated            =  8,   //!< タグを喪失した状態
    NfcState_Detected               =  9,   //!< タグを検出した状態
    NfcState_FactoryMode            = 10,   //!< NFC の検査モード
    NfcState_Initializing           = 11,   //!< NFC の初期化を行っている状態
    NfcState_PassThruFinish         = 12,   //!< パススルーコマンド結果受信
    NfcState_ResetRequired          = 13,   //!< MCU のリセットが必要なエラーが発生
    NfcState_HwFatal                = 14,   //!< HW に致命的なエラーが発生
    NfcState_MifareSending          = 15,   //!< MIFARE コマンド送信中
    NfcState_MifareFinish           = 16,   //!< MIFARE コマンド結果受信
    NfcState_MifareKeyWriting       = 17,   //!< MIFARE 鍵書き込み中
    NfcState_MifareKeyWriteFinish   = 18    //!< MIFARE 鍵書き込み終了
};

/**
 * @brief   NFC タグの Discovery Type を表す ID です。
 */
enum DiscoveryTypeId : uint8_t
{
    DiscoveryTypeId_Type1        = 0x01,    //!< Type 1
    DiscoveryTypeId_Type2        = 0x02,    //!< Type 2
    DiscoveryTypeId_Type3        = 0x03,    //!< Type 3
    DiscoveryTypeId_Type4        = 0x04,    //!< Type 4 (ISO-DEP)
    DiscoveryTypeId_TypeIso15693 = 0x83,    //!< ISO 15693
    DiscoveryTypeId_TypeMifare   = 0x90     //!< MIFARE
};

/**
 * @brief   Tera の NFC に関するイベントの種類
 */
enum class InternalNfcEventType
{
    None,               //!< なし
    Error,              //!< エラー発生
    Information,        //!< NFC_STATE_INF 受信
    TagDetect,          //!< タグ検出
    TagLost,            //!< タグ喪失
    DataReceived,       //!< データ読み込み完了
    DataWritten,        //!< データ書き込み完了
    PassThru,           //!< パススルーコマンドの応答を受信
    ModuleCheck,        //!< NFC_MODULE_CHECK_RSP 受信
    ModuleSpi,          //!< NFC_MODULE_SPI_RSP 受信
    MifareKeyWritten,   //!< MIFARE 鍵書き込み完了
    Mifare              //!< NFC_MIFARE_RSP 受信
};

/**
 * @brief   ペイロードの属性です。
 */
struct NfcPacketPayloadAttributePack
{
    typedef ::nn::util::BitPack8::Field<0, 3, uint8_t> LengthHigh;      //!< ペイロード長 (上位 3 ビット)
    typedef ::nn::util::BitPack8::Field<3, 1, uint8_t> EndOfCommand;    //!< 最終パケット、または非分割パケットの場合は 1
    typedef ::nn::util::BitPack8::Field<4, 4, uint8_t> Reserved;        //   予約
};

/**
 * @brief   NFC 通信に使用する生パケットのヘッダです。
 */
struct NN_ALIGNAS(1) NfcPackedPacketHeader
{
    union
    {
        ::nn::util::BitPack8 mode;  //!< モード (MOSI 用)
        uint8_t resultCode;         //!< リザルトコード (MISO 用)
    };

    NfcCommand           command;           //!< 実行する NFC_MANAGEMENT コマンド
    uint8_t              sequenceNumber;    //!< 再送制御に使用するコマンドのシーケンス番号
    uint8_t              ackNumber;         //!< 最後に正常に受信した要再送パケットの sequence number
    ::nn::util::BitPack8 payloadAttribute;  //!< ペイロードの情報
    uint8_t              payloadLengthLow;  //!< ペイロード長 (下位 8 ビット). エンディアンの問題を楽に回避するために分けておく

    /**
     * @brief   Bluetooth モードを設定します。
     */
    void SetBluetoothMode(uint8_t bluetoothMode) NN_NOEXCEPT
    {
        mode.Set<PacketModePackForSend::BluetoothMode>(bluetoothMode);
    }

    /**
     * @brief   データプロトコルを設定します。
     */
    void SetProtocol(uint8_t protocol) NN_NOEXCEPT
    {
        mode.Set<PacketModePackForSend::Protocol>(protocol);
    }

    /**
     * @brief   コマンドの終端パケットかどうか判定します。
     */
    bool IsEndOfCommand() const NN_NOEXCEPT
    {
        return payloadAttribute.Get<NfcPacketPayloadAttributePack::EndOfCommand>() == 1;
    }

    /**
     * @brief   コマンドの終端パケットフラグを設定します。
     */
    void SetEndOfCommand(bool isEnd) NN_NOEXCEPT
    {
        payloadAttribute.Set<NfcPacketPayloadAttributePack::EndOfCommand>(isEnd ? 1 : 0);
    }

    /**
     * @brief   ペイロード長を取得します。
     */
    size_t GetPayloadLength() const NN_NOEXCEPT
    {
        return (payloadAttribute.Get<NfcPacketPayloadAttributePack::LengthHigh>() << 8) | payloadLengthLow;
    }

    /**
     * @brief   ペイロード長を設定します。
     */
    void SetPayloadLength(size_t length) NN_NOEXCEPT
    {
        payloadLengthLow = static_cast<uint8_t>(length & 0xFF);
        payloadAttribute.Set<NfcPacketPayloadAttributePack::LengthHigh>(
            static_cast<uint8_t>(length >> 8));
    }
};

/**
 * @brief   NFC Common コマンドの応答パケットの共通ヘッダです。
 */
struct NN_ALIGNAS(1) NfcPackedCommonResponseHeader
{
    uint8_t state;              //!< NFC 内部ステート
    uint8_t _reserved;          //   予約 (CC1.2 以前は NFC コマンドのリザルトコード)
    uint8_t shimApi;            //!< NFC shim layer API ID
    uint8_t shimError;          //!< NFC shim layer error ID
};

/**
 * @brief   検出した NFC タグの基本情報です。
 */
struct NN_ALIGNAS(1) NfcPackedTagInfo
{
    uint8_t rfDiscoveryId;      //!< RF Discovery ID
    uint8_t protocolType;       //!< タグのプロトコルタイプ (Type1, Type2, etc.)
    uint8_t nfcDiscoveryType;   //!< NFC Discovery Type (NFC-A, NFC-B, etc.)
    uint8_t uidLength;          //!< UID 長
    uint8_t uid[UidLengthMax];  //!< UID
};

/**
 * @breif   NFC_STATE_INF コマンドのペイロードです。
 */
struct NN_ALIGNAS(1) NfcPackedStateInfPayload
{
    NfcPackedCommonResponseHeader common;       //!< 共通ヘッダ
    uint8_t detectedTagNumber;                  //!< 検出したタグの数
    NfcPackedTagInfo tags[DetectableTagCount];  //!< 検出したタグの一覧

    /**
     * @brief   メンバを初期化します。
     */
    void Initialize() NN_NOEXCEPT
    {
        common.state      = NfcState_Initializing;
        common._reserved  = 0;
        common.shimApi    = 0;
        common.shimError  = 0;
        detectedTagNumber = 0;
        std::memset(tags, 0, sizeof(tags));
    }
};

/**
 * @breif   NFC_DATA_RSP コマンドのヘッダです。
 */
struct NN_ALIGNAS(1) NfcDataResponseHeader
{
    NfcPackedCommonResponseHeader common;       //!< 共通ヘッダ
    NfcPackedTagInfo tagInfo;                   //!< タグの情報
    uint8_t t2tVersion;                         //!< Type2 タグバージョン
    char signature[SignatureBytes];             //!< 認証情報
};

/**
 * @breif   NFC_PASS_THRU_RSP コマンドのヘッダです。
 */
struct NN_ALIGNAS(1) NfcPassThruResponseHeader
{
    NfcPackedCommonResponseHeader common;       //!< 共通ヘッダ
    NfcPackedTagInfo tagInfo;                   //!< タグの情報
};

/**
 * @breif   NFC_MIFARE_RSP コマンドのヘッダです。
 */
struct NN_ALIGNAS(1) NfcMifareResponseHeader
{
    NfcPackedCommonResponseHeader common;       //!< 共通ヘッダ
};

/**
 * @brief   NFC 通信に使用するパケットのヘッダです。
 */
struct NfcPacketHeaderType
{
    nn::Bit8   bluetoothMode;   //!< モード
    nn::Bit8   protocol;        //!< プロトコル
    NfcCommand command;         //!< 実行する NFC_MANAGEMENT コマンド
    uint8_t    sequenceNumber;  //!< 再送制御に使用するコマンドのシーケンス番号
    uint8_t    ackNumber;       //!< 最後に正常に受信した要再送パケットの sequence number
};

/**
 * @brief   NFC_START_DISCOVERY コマンドのパラメータを格納する構造体です。
 */
struct NfcStartDiscoveryCommandType
{
    NfcPollingMask pollingMask;     //!< ポーリング対象のタグを表すマスク
    uint16_t activationTimeoutMs;   //!< タグ検出ポーリングのタイムアウト [msec]
    uint16_t discoveryPeriodMs;     //!< ポーリング間隔 [msec]
};

/**
 * @brief   NFC コマンド全体のヘッダやペイロードを格納する構造体です。
 */
struct NfcCommandType
{
    NfcPacketHeaderType header;         //!< コマンドのヘッダ
    char payload[PayloadLengthMax];     //!< ペイロード
    size_t payloadLength;               //!< ペイロード長

    /**
     * @brief   再送が必要なコマンドか判定します。
     */
    bool IsResendRequired() const NN_NOEXCEPT
    {
        switch (header.command)
        {
        case detail::NfcCommand_WriteStart:
        case detail::NfcCommand_PassThru:
        case detail::NfcCommand_MifareRequest:
        case detail::NfcCommand_MifareKeyStartWrite:
            return true;
        default:
            return false;
        }
    }
};

/**
 * @brief   Type2 タグのデータを格納する構造体です。
 */
struct NN_ALIGNAS(1) NfcType2TagData
{
    NfcDataResponseHeader header;       //!< NFC_DATA_RSP のヘッダ
    char payload[TagReadDataBytesMax];  //!< ペイロード
};

/**
 * @brief   パススルーコマンドの応答を格納する構造体です。
 */
struct NN_ALIGNAS(2) NfcPassThruResponseData
{
    NfcPassThruResponseHeader header;       //!< NFC_PASS_THRU_RSP のヘッダ
    uint16_t dataSize;                      //!< ペイロードのサイズ
    char     data[NfcPassThruDataSizeMax];  //!< ペイロード
};

/**
 * @brief   NFC_MIFARE_KEY_WRITE_START コマンドの鍵パラメータを格納する構造体です。
 */
struct NN_ALIGNAS(1) NfcMifareKeyListForWrite
{
    uint8_t  keyCount;                  //!< 鍵の数 (1 - 12)
    uint8_t  keys[MifareKeyCountMax][MifareEncryptedKeyLength];  //!< 暗号化された MIFARE 鍵の値
};

/**
 * @brief   MIFARE コマンドのデータブロックを表す構造体です。
 */
struct NN_ALIGNAS(1) NfcMifareBlock
{
    uint8_t address;                    //!< ブロックアドレス
    uint8_t data[MifareBlockBytes];     //!< データ本体
};

/**
 * @brief   MIFARE コマンドの応答を格納する構造体です。
 */
struct NN_ALIGNAS(1) NfcMifareResponseData
{
    NfcMifareResponseHeader header;                     //!< NFC_MIFARE_RSP のヘッダ
    uint8_t        blockCount;                          //!< データブロック数
    NfcMifareBlock blocks[MifareReadBlockCountMax];     //!< データブロック
};

/**
 * @brief   MIFARE コマンドの属性です。
 */
struct NfcMifareCommandAttributePack
{
    typedef ::nn::util::BitPack8::Field<0, 1, uint8_t> WriteMode;       //!< 書き込みモードフラグ
    typedef ::nn::util::BitPack8::Field<1, 1, uint8_t> UseRawKey;       //!< 生鍵指定フラグ
    typedef ::nn::util::BitPack8::Field<2, 5, uint8_t> CommandCount;    //!< コマンド送信数
    typedef ::nn::util::BitPack8::Field<7, 1, uint8_t> Reserved;        //   予約
};

/**
 * @brief   MIFARE コマンドの動作を設定する構造体です。
 */
struct NfcMifareCommandConfig
{
    ::nn::util::BitPack8 _config;   // 設定値を格納する領域

    /**
     * @brief   設定を初期化します。
     */
    void Clear() NN_NOEXCEPT
    {
        _config.Clear();
    }

    /**
     * @brief   書き込みを行うかどうかを指定します。
     */
    void SetWriteMode(bool isWriteMode) NN_NOEXCEPT
    {
        _config.Set<NfcMifareCommandAttributePack::WriteMode>(isWriteMode ? 1 : 0);
    }

    /**
     * @brief   生鍵を使用するかどうかを指定します。
     */
    void SetUseRawKey(bool isRawKey) NN_NOEXCEPT
    {
        _config.Set<NfcMifareCommandAttributePack::UseRawKey>(isRawKey ? 1 : 0);
    }

    /**
     * @brief   送信する読み書きコマンドの数を指定します。
     */
    void SetCommandCount(uint8_t commandCount) NN_NOEXCEPT
    {
        _config.Set<NfcMifareCommandAttributePack::CommandCount>(commandCount);
    }

    /**
     * @brief   ペイロードに格納するための値を取得します。
     */
    char GetValue() NN_NOEXCEPT
    {
        return static_cast<char>(_config.GetMaskedBits(0xFF));
    }
};

/**
 * @brief   MCU から受信したタグに関する情報を格納する構造体です。
 */
struct ResponseTagData
{
    union
    {
        char rawData[sizeof(NfcType2TagData)];  //!< 生データ (共用体内で最大のサイズのメンバが入るサイズ)
        NfcType2TagData         type2Read;      //!< Type2 タグのデータ
        NfcPassThruResponseData passThru;       //!< パススルーの応答
        NfcMifareResponseData   mifare;         //!< MIFARE の応答
    };
    size_t size;    //!< 有効なデータサイズ

    /**
     * @brief   構造体のメンバを初期化します。
     */
    void Clear() NN_NOEXCEPT
    {
        size = 0;
        std::memset(rawData, 0, sizeof(rawData));
    }

    /**
     * @brief   格納したデータのサイズが上限に達しているか判定します。
     */
    bool IsFull() NN_NOEXCEPT
    {
        return size >= sizeof(rawData);
    }
};

/**
 * @brief   GetNfcInfo で取得するための情報を格納する構造体です。
 */
struct StatusForGetInfo
{
    NfcResultCode               resultCode;     //!< NFC リザルトコード
    NfcPackedStateInfPayload    stateInfo;      //!< NFC の状態
};

/**
 * @brief   指定した値を Little Endian でバイト列に格納
 */
template <typename OutType, typename InType>
void SetLittleEndianValue(OutType *pOutData, size_t outSize, InType value)
{
    NN_STATIC_ASSERT(sizeof(OutType) == 1);  // 1 バイトの型しか受け付けない

    NN_SDK_REQUIRES_NOT_NULL(pOutData);

    for (size_t i = 0; i < std::min(sizeof(InType), outSize); i++)
    {
        pOutData[i] = static_cast<OutType>((value >> (8 * i)) & 0xFF);
    }
}

/**
 * @brief   NFC のブロックを表示
 */
void DumpNfcBlocks(const char* pData, int startBlock, size_t blocks) NN_NOEXCEPT;

/**
 * @brief   NFC ステートの文字列を取得
 */
const char* GetNfcStateString(NfcState state) NN_NOEXCEPT;

/**
 * @brief   NFC リザルトコードの文字列を取得
 */
const char* GetNfcResultString(NfcResultCode result) NN_NOEXCEPT;

/**
 * @brief   内部タグタイプから公開タグタイプへ変換
 */
nn::xcd::NfcTagType ConvertTagType(uint8_t discoveryType, uint8_t internalTagType) NN_NOEXCEPT;

/**
 * @brief   コマンドコードを指定して NFC コマンドを初期化
 */
void ConfigureNfcCommand(NfcCommandType* pCommand, NfcCommand commandCode) NN_NOEXCEPT;

/**
 * @brief   NFC_NOP を指定してコマンドを初期化
 */
void ConfigureNfcNopCommand(NfcCommandType* pCommand) NN_NOEXCEPT;

/**
 * @brief   特定のコマンド送信が完了した際に遷移するステートであるかどうか判定
 *
 * @param[in]   command     送信中のコマンド
 * @param[in]   state       現在のステート
 */
bool IsExpectedNfcStateForSendFinish(NfcCommand command, NfcState state) NN_NOEXCEPT;

/**
 * @brief   MIFARE コマンドの共通ブロックパラメータを格納
 */
int SetMifareCommonBlockParameter(
    char* pOutParameter,
    bool isRawKey,
    const MifareKey& key,
    uint8_t blockAddress) NN_NOEXCEPT;

}}}
