﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/xcd/xcd_Device.h>
#include "../xcd_DeviceListManager.h"
#include "xcd_SystemServerImpl.h"
#include "xcd_TeraCommon.h"

namespace nn { namespace xcd {

DeviceListManager* GetDeviceListManager() NN_NOEXCEPT;

namespace detail {

nn::Result SystemServerImpl::GetDataFormat(
    nn::sf::Out<std::uint8_t> pOutDataFormat,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    *pOutDataFormat = pHandler->GetDataFormat();

    NN_RESULT_SUCCESS;
}

nn::Result SystemServerImpl::SetDataFormat(
    std::uint8_t dataFormat,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;

    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    pHandler->SetDataFormat(static_cast<nn::xcd::PeriodicDataFormat>(dataFormat));

    NN_RESULT_SUCCESS;
}

nn::Result SystemServerImpl::GetMcuState(
    nn::sf::Out<std::uint8_t> pOutMcuState,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    *pOutMcuState = pHandler->GetTeraBase()->GetMcuState();

    NN_RESULT_SUCCESS;
}

nn::Result SystemServerImpl::SetMcuState(
    std::uint8_t mcuState,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->SetMcuState(
        static_cast<nn::xcd::McuState>(mcuState));
}

nn::Result SystemServerImpl::SetMcuStateImmediate(
    std::uint8_t mcuState,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->SetMcuStateImmediate(
        static_cast<nn::xcd::McuState>(mcuState));
}

nn::Result SystemServerImpl::GetMcuVersionForNfc(
    nn::sf::Out<nn::xcd::McuVersionDataForNfc> pOutMcuVersionData,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    auto* pInternalPointer = pOutMcuVersionData.GetPointer();
    return pHandler->GetTeraBase()->GetMcuFirmwareVersionForNfc(pInternalPointer);
}

nn::Result SystemServerImpl::CheckNfcDevicePower(
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    return detail::CheckNfcDevicePowerImpl(pHandler);
}

nn::Result SystemServerImpl::SetNfcEvent(
    nn::sf::Out<nn::sf::NativeHandle> nfcEventHandle,
    nn::sf::Out<nn::sf::NativeHandle> nfcDetectEventHandle,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    ::nn::os::NativeHandle commonHandle;
    ::nn::os::NativeHandle detectHandle;

    pHandler->GetTeraBase()->SetNfcEvent(&commonHandle, &detectHandle);

    *nfcEventHandle       = nn::sf::NativeHandle(commonHandle, false);
    *nfcDetectEventHandle = nn::sf::NativeHandle(detectHandle, false);

    NN_RESULT_SUCCESS;
}

nn::Result SystemServerImpl::GetNfcInfo(
    nn::sf::Out<nn::xcd::NfcInfo> pOutValue,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));

    auto* pInternalPointer = pOutValue.GetPointer();

    // コマンド発行関数に、コマンド発行処理をラムダ式で渡す
    // (ラムダ式の引数には NfcProcessor のポインタが入るため、これに対してコマンド発行を指示する)
    return pHandler->GetTeraBase()->ProcessNfcCommand([&pInternalPointer](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->GetNfcInfo(pInternalPointer);
    });
}

nn::Result SystemServerImpl::StartNfcDiscovery(
    nn::xcd::NfcDiscoveryParameter discoveryParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&discoveryParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartDiscovery(discoveryParameter);
    });
}

nn::Result SystemServerImpl::StopNfcDiscovery(DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StopDiscovery();
    });
}

nn::Result SystemServerImpl::StartNtagRead(
    const nn::xcd::NtagReadParameter& readParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&readParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartNtagRead(readParameter);
    });
}

nn::Result SystemServerImpl::StartNtagWrite(
    const nn::xcd::NtagWriteParameter& writeParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&writeParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartNtagWrite(writeParameter);
    });
}

nn::Result SystemServerImpl::SendNfcRawData(
    const nn::xcd::NfcPassThruParameter& passThruParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&passThruParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->SendRawData(passThruParameter);
    });
}

nn::Result SystemServerImpl::RegisterMifareKey(
    const nn::xcd::MifareKeyWriteParameter& keyWriteParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&keyWriteParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->RegisterMifareKey(keyWriteParameter);
    });
}

nn::Result SystemServerImpl::ClearMifareKey(
    const nn::xcd::MifareKeyClearParameter& keyClearParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&keyClearParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->ClearMifareKey(keyClearParameter);
    });
}

nn::Result SystemServerImpl::StartMifareRead(
    const nn::xcd::MifareReadParameter& readParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&readParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartMifareRead(readParameter);
    });
}

nn::Result SystemServerImpl::StartMifareWrite(
    const nn::xcd::MifareWriteParameter& writeParameter,
    DeviceHandle deviceHandle) NN_NOEXCEPT
{
    DeviceHandler* pHandler;
    NN_RESULT_DO(GetDeviceListManager()->GetDeviceHandler(&pHandler, deviceHandle));
    return pHandler->GetTeraBase()->ProcessNfcCommand([&writeParameter](NfcProcessor* pProcessor)
    {
        NN_SDK_ASSERT_NOT_NULL(pProcessor);
        return pProcessor->StartMifareWrite(writeParameter);
    });
}

nn::Result SystemServerImpl::GetAwakeTriggerReasonForLeftRail(
    ::nn::sf::Out<uint64_t> pOutValue) NN_NOEXCEPT
{
    auto awakeReason = GetDeviceListManager()->GetAwakeTriggerReasonForLeftRail();
    pOutValue.Set(static_cast<uint64_t>(awakeReason));

    NN_RESULT_SUCCESS;
}

nn::Result SystemServerImpl::GetAwakeTriggerReasonForRightRail(
    ::nn::sf::Out<uint64_t> pOutValue) NN_NOEXCEPT
{
    auto awakeReason = GetDeviceListManager()->GetAwakeTriggerReasonForRightRail();
    pOutValue.Set(static_cast<uint64_t>(awakeReason));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::xcd::detail
