﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_LightEventTypes.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_BluetoothSettings.h>
#include <nn/xcd/xcd_Rail.h>
#include <nn/xcd/xcd_Sleep.h>
#include <nn/xcd/xcd_Usb.h>
#include "xcd_AardvarkAccessor-os.win32.h"
#include "xcd_Win32HidAccessor-os.win32.h"
#include "xcd_ILinkMonitorImpl.h"
#include "xcd_LinkMonitorTypes.h"
#include "../xcd_TaskManager.h"

namespace nn { namespace xcd { namespace detail{

const size_t LinkThreadStackSize = 1024 * 8;
const int HidDeviceCountMax = 16;

//!< デバイスの接続状態の監視を行うためのプラットフォームごとの実装クラス
class LinkMonitorImpl : public ILinkMonitorImpl
{
    NN_DISALLOW_MOVE(LinkMonitorImpl);
    NN_DISALLOW_COPY(LinkMonitorImpl);

private:
    //!< デバイスの接続状態を監視するスレッド
    nn::os::ThreadType m_Thread;
    //!< スレッドのスタック
    NN_ALIGNAS(4096) char m_ThreadStack[LinkThreadStackSize];

    //!< 多重待ち
    nn::os::MultiWaitType m_MultiWait;

    //!< 終了を通知するイベント
    nn::os::EventType m_TerminateEvent;
    nn::os::MultiWaitHolderType m_TerminateHolder;

    //!< 定期監視用
    nn::os::TimerEventType m_TimerEvent;
    nn::os::MultiWaitHolderType m_TimerHolder;

    //!< デバイス接続監視の終了を通知するイベント
    nn::os::LightEventType* m_pUpdatedEvent;

    //!< Win向け実装
    GUID m_HidGuid;

    //!< HIDデバイスのリスト
    static const size_t PathLengthMax = 1024;
    struct HidDeviceDescryptor
    {
        Win32HidAccessor win32Accessor;
        AardvarkAccessor aardvarkAccessor;
        TCHAR devicePath[PathLengthMax];
        HidDeviceInfo deviceInfo;

        Win32HidAccessor& GetHidAccessor()
        {
            return (aardvarkAccessor.IsActivated()) ? aardvarkAccessor : win32Accessor;
        }
    };

    HidDeviceDescryptor m_Devices[HidDeviceCountMax];

    bool m_Activated;
    bool m_IsAardvarkDetected;  //!< Aardvark の検出を 1 台に制限するためのフラグ

private:
    static void ThreadFunction(void* arg) NN_NOEXCEPT;
    void ThreadFunctionImpl() NN_NOEXCEPT;

    //!< デバイスをリストアップする。非スレッドセーフ
    void ScanDevices() NN_NOEXCEPT;
    bool ScanAardvark() NN_NOEXCEPT;
    bool RegisterDevice(LPCTSTR devicePath) NN_NOEXCEPT;
    void RemoveDevice(HidDeviceDescryptor& descryptor) NN_NOEXCEPT;

public:
    LinkMonitorImpl() NN_NOEXCEPT;
    ~LinkMonitorImpl() NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を開始する
    void StartMonitoring(nn::os::LightEventType* updateEvent) NN_NOEXCEPT;

    //!< デバイスの接続状態の監視を停止する
    void StopMonitoring() NN_NOEXCEPT;

    //!< Suspend します。
    void Suspend() NN_NOEXCEPT;

    //!< Resume します。
    void Resume() NN_NOEXCEPT;

    //!< 接続済みデバイスのハンドルを取得する
    size_t GetDevices(HidDeviceInfo *pOutValue, size_t deviceCount) NN_NOEXCEPT;

    //!< デバイスハンドルからHIDAccessorを取得する
    HidAccessor* GetHidAccessor(DeviceHandle deviceHandle) NN_NOEXCEPT;

    //!< Bluetooth の接続可能最大数を取得します
    virtual int GetMaxBluetoothLinks() NN_NOEXCEPT NN_OVERRIDE;
};

}}} // namespace nn::xcd::detail
